
#    This file is part of InEXEQ.
#
#    InEXEQ is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <https://www.gnu.org/licenses/>.
# 
# Copyright (C) Maciej Bartkowiak, 2014-2018, 2020

__doc__ = """
One of the key component of the EXED (E,q)-range calculators,
this module contains the instrument definition.
It reads in a .PAR file which defines all the detector pixels,
and then calculates the accessible range based on the other settings,
such as the chopper speed and wavelength band.
Can be made more useful by mixing with the sample definition.
Maciej Bartkowiak, 20-Jan-2014.

NEW: now it reads the XML file, and creates a list of detector banks,
instead of a single bank.
"""

import numpy as np
import scipy.interpolate as scint
from file_io import *
import scipy.spatial as scisp
from units import *
from geometry import *
import sys
import os

def resource_path(relative_path):
    """ Get absolute path to resource, works for dev and for PyInstaller """
    try:
        # PyInstaller creates a temp folder and stores path in _MEIPASS
        base_path = sys._MEIPASS
    except Exception:
        base_path = os.path.abspath(".")

    return os.path.join(base_path, relative_path)

edges = {'alpha': [[1,2], [5,6], [3,0], [7,4]], 
         'beta' : [[0,1], [4,5], [2,3], [6,7]],
         'ki'   : [[0,4], [1,5], [2,6], [3,7]]}

class Exed:
    def __init__(self):
        self.spectrum_longguide = self.spectrum_read("2slit_lambda.dat")
        self.minangle = -15
        self.segments = 10
        self.chopper = {"distance":2.4, "frequency":7200.0/60.0, "suppression":2.0}
        self.current_spectrum = []
        self.current_panels = []
        self.angle_limit = (-12.0, 2.5)
    def current_panels_at(self, inputangle):
        output = []
        angle = int(round(inputangle))
        for i in self.current_panels:
            if not i == []:
                output.append(i[angle])
            else:
                output.append([])
        return output
    def spectrum_read(self, fname = "2slit_lambda.dat"):
        """
        Reads in a wavelength spectrum of the instrument for future reference.
        At the moment we use a Vitess simulation result.
        """
        data = read_numbers(resource_path(fname))
        data[:,1] /= 1.3**2
        spec = scint.interp1d(data[:,0], data[:,1])
        return spec
    def limits_forward(self, ori):
        det_dist = 4.5
        h_angles = np.radians(-np.array([
                       -13.476973875513735, -7.3883587493712479,
                       -6.9532348362990346, 1.2580380710293861,
                       1.6491960956904845, 9.8619904593800936,
                       10.277161241040176, 16.360461469241837
                       ])[::-1] - ori)#  + np.pi/2.0
        v_angles = np.radians(np.array([
                       12.53, 12.53,
                       14.93, 14.93,
                       14.93, 14.93,
                       12.53, 12.53,
                       ]))
        cone_angle = np.radians(14.3)
        beamstop = [np.arctan2(x, 4.4) for x in [0.164, -0.27, -0.182, 0.2025]] # top, bottom, left, right
        return det_dist, h_angles, v_angles, cone_angle, beamstop
    def limits_backward(self):
        det_dist = 3.4
        h_angles = np.radians(np.array([
                       162.67693687457276, 169.8, # was 173.72152666715047 but we have the chopper housing there
                       -169.8, -161.34646779906987
                       ]))#  + np.pi/2.0
        v_angles = np.radians(np.array([
                       7.9, 7.7, 
                       7.7, 7.9
                       ]))
        cone_angle = np.radians(180.0-13.923)
        beamstop = []
        return det_dist, h_angles, v_angles, cone_angle, beamstop

        
