
#    This file is part of InEXEQ.
#
#    InEXEQ is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <https://www.gnu.org/licenses/>.
# 
# Copyright (C) Maciej Bartkowiak, 2014-2018, 2020

__doc__ = """
An attempt to separate the input validation from the interface.
Also, low-level caching of the results should happen here.
Maciej Bartkowiak, 20 Jul 2014
"""

from geometry import *
from units import *
from sample import Sample
from instrument import Exed
from pyexe import parse_input
from file_io import parse_input_2018
import plot
# import plot_agg as plot
import copy
import yaml

params = {}
params["Choppers"] = {}
params["Sample"] = {}
params["Field"] = {}
params["Lambda"] = {}
params["hkl"] = {}

limits = {}
limits["Choppers"] = {}
limits["Sample"] = {}
limits["Field"] = {}
limits["Lambda"] = {}
limits["hkl"] = {}

instrument = {}
instrument['Detectors'] = {}

params["Field"]["Direction"] = 0.0
params["Choppers"]["Frequency"] = 120
params["Choppers"]["Suppression"] = 1
params["Lambda"]["Res"] = 0.01
params["Lambda"]["Ei"] = 20
params["Lambda"]["Flux"] = 20
params["Lambda"]["FWHM"] = 0.7
params["Lambda"]["MaxLoss"] = 7.9
params["Sample"]["Constants"] = np.array([5.0, 5.0, 5.0])
params["Sample"]["Angles"] = np.array(3*[90.0])
# params["Sample"]["u"] = np.array([-0.136,0.046,0.989])
# params["Sample"]["v"] = np.array([-0.929,0.367,-0.033])
params["Sample"]["u"] = np.array([0.0, 0.0, 1.0])
params["Sample"]["v"] = np.array([1.0, 0.0, 0.0])
params["Sample"]["phi"] = 0
params["Sample"]["chi"] = 0
params["Sample"]["omega"] = 0
params["hkl"]["h"] = 'h'
params["hkl"]["k"] = '0'
params["hkl"]["l"] = 'l'
params["hkl"]["dE"] = '0:max'

limits["Field"]["Direction"] = (-15.0, 15.0)
limits["Choppers"]["Frequency"] = (5.0, 240.0)
limits["Choppers"]["Suppression"] = (1, 8)
limits["Lambda"]["Res"] = (5e-1, 1e3)
limits["Lambda"]["Ei"] = (1e-6, 120.0)
limits["Lambda"]["FWHM"] = (0.0, 60.0)
limits["Lambda"]["MaxLoss"] = (0.0, 120.0)
limits["Sample"]["Constants"] = (np.array([0.01, 0.01, 0.01]), np.array([40000.4, 40000.4, 40000.4]))
limits["Sample"]["Angles"] = (np.array(3*[0.01]), np.array(3*[179.99]))
limits["Sample"]["u"] = (np.array([-100.0, -100.0, -100.0]), np.array([100.0, 100.0, 100.0]))
limits["Sample"]["v"] = (np.array([-100.0, -100.0, -100.0]), np.array([100.0, 100.0, 100.0]))
limits["Sample"]["phi"] = (-360,360)
limits["Sample"]["chi"] = (-360,360)
limits["Sample"]["omega"] = (-360,360)
limits["hkl"]["h"] = (-10000, 10000)
limits["hkl"]["k"] = (-10000, 10000)
limits["hkl"]["l"] = (-10000, 10000)
limits["hkl"]["dE"] = (-10000, 10000)

instrument['Detectors']['Active'] = [1,1,1,1,1,1]
instrument['Detectors']['Forward'] = 'Fixed' # options: 'Fixed', 'SANS'
instrument['Detectors']['Backward'] = 'Symmetrical' # options: 'Symmetrical', 'Left side'
instrument['Guide'] = 'Long' # options: 'Long', 'Short', 'SANS'
instrument['Cryostat'] = 'Back' # options: 'Front', 'Back', 'None'

normals = [["Sample", "u"], ["Sample", "v"]]

fields = [
["Field","Direction"],
#["Choppers","Frequency"],
#["Choppers","Suppression"],
["Lambda","Ei"],
["Lambda","Res"],
["Lambda","FWHM"],
["Lambda","MaxLoss"],
["Sample","Constants"],
["Sample","Angles"],
["Sample","u"],
["Sample","v"],
["Sample","phi"],
["Sample","chi"],
["Sample","omega"],
["hkl","h"],
["hkl","k"],
["hkl","l"],
["hkl","dE"],
]

testfields = [
["Field","Direction"],
["Choppers","Frequency"],
["Choppers","Suppression"],
["Lambda","Ei"],
["Sample","Constants"],
["Sample","Angles"],
["Sample","u"],
["Sample","v"],
["Sample","phi"],
["Sample","chi"],
["Sample","omega"],
["hkl","h"],
["hkl","k"],
["hkl","l"],
["hkl","dE"],
]

# fangle = 2*np.degrees(np.arctan((0.06/100.0) / 0.01))

# fermi_at_600Hz = 15 * 1e-6 # pulse width of Fermi chopper at max speed
choppers = {}
choppers["Fermi"] = (20.359, 6.86)
choppers["1A"] = (20.51, 55.5)
choppers["1B"] = (20.56, 55.5)
choppers["2"] = (21.032, 40.4)
choppers["3"] = (22.034, 46.8)
choppers["4"] = (28.507, 69.3)
choppers["5"] = (48.134, 180)
choppers["6A"] = (71.1051, 2.6)
choppers["6B"] = (71.1351, 2.6)
detector_pos = 71.1351 + 2.42 + 3.0
    
class Kernel:
    """
    This entity takes in the input parameters, validates them,
    and then produces plots on demand.
    It also keeps track of parameter changes, and only recalculates
    the picture if the parameters have changed since the last call.
    """
    def __init__(self, figure = None, precision = 1.0):
        self.orientations = np.arange(0, 16, 1).astype(np.int)
        self.results = []
        for i in self.orientations:
            self.results.append({'forward':None, 'backward':None,
                                 'fax':None, 'bax':None,
                                 'symbol':None, 'description':None,
                                 'settings':None})
        self.param = params
        self.instrument = instrument
        # self.exed = Exed(fname = "exed.par", spectrum = "2slit_lambdann.dat", cryo = False)
        # self.exed_hc = Exed(fname = "exed_hc.par", spectrum = "2slit_lambda.dat", cryo = True)
        self.exed = Exed()
        # self.exed.configure(self.param, self.instrument)
        self.figure = figure
        self.markers = []
        self.multiplier = precision
        self.cryo = False
        
    def writeout_settings(self, tname):
        param = copy.deepcopy(self.param)
        param['Sample']['u'] = ','.join([str(x) for x in self.param['Sample']['u']])
        param['Sample']['v'] = ','.join([str(x) for x in self.param['Sample']['v']])
        param['Sample']['Angles'] = ','.join([str(x) for x in self.param['Sample']['Angles']])
        param['Sample']['Constants'] = ','.join([str(x) for x in self.param['Sample']['Constants']])
        if type(param['hkl']['h']) == type([]):
            param['hkl']['h'] = ':'.join([str(x) for x in self.param['hkl']['h']])
        if type(param['hkl']['k']) == type([]):
            param['hkl']['k'] = ':'.join([str(x) for x in self.param['hkl']['k']])
        if type(param['hkl']['l']) == type([]):
            param['hkl']['l'] = ':'.join([str(x) for x in self.param['hkl']['l']])
        if type(param['hkl']['dE']) == type([]):
            param['hkl']['dE'] = ':'.join([str(x) for x in self.param['hkl']['dE']])
        marks = []
        for i in self.markers:
            marks.append(','.join([str(x) for x in i]))
        if len(tname.split(".")) == 1:
            tname = tname + '.exq'
        if not tname.split(".")[-1] == 'exq':
            tname = ".".join(tname.split(".")[:-1]+['exq'])
        target = open(tname, 'w')
        yaml.dump([param, self.instrument, marks, self.multiplier], stream = target)
        target.close()
        
    def load_settings(self, tname):
        target = open(tname, 'r')
        firstline = target.readline()
        target.close()
        if 'exeq' in firstline:
            selcom, params, markers, prec, instrument = parse_input_2018(tname)
            self.take_parameters(params)
            self.take_instrument(instrument)
            self.marker_points(markers)
            self.set_precision(prec)
            return None
        else:
            target = open(tname, 'r')
            xxxx = yaml.load(target)
            target.close()
            try:
                xxxx.split()
            except AttributeError:
                self.take_parameters(xxxx[0])
                self.take_instrument(xxxx[1])
                self.marker_points(xxxx[2])
                self.set_precision(xxxx[3])
            else:
                selcom, params, markers, prec, totalview, instrument = parse_input(tname)
                self.take_parameters(params)
                self.take_instrument(instrument)
                self.marker_points(markers)
                self.set_precision(prec)
            
    def take_instrument(self, inst):
        """
        instrument['Detectors']['Active'] = [1,2,3,4]
        instrument['Detectors']['Forward'] = 'Fixed' # options: 'Fixed', 'SANS'
        instrument['Detectors']['Backward'] = 'Symmetrical' # options: 'Symmetrical', 'Left side'
        instrument['Guide'] = 'Short' # options: 'Long', 'Short'
        instrument['Cryostat'] = 'Front' # options: 'Front', 'Back'
        """
        # param = list_to_par(params)
        temp = inst
        try:
            len(temp)
        except TypeError:
            inst = copy.deepcopy(instrument)
        else:
            actlist = []
            try:
                a = inst['Detectors']['Active']
            except KeyError:
                for x in range(6):
                    actlist.append(True)
            else:
                for i in inst['Detectors']['Active']:
                    if i:
                        actlist.append(True)
                    else:
                        actlist.append(False)
            if 'ixed' in inst['Detectors']['Forward']:
                forward = 'Fixed'
            else:
                forward = 'SANS'
            if 'metrical' in inst['Detectors']['Backward']:
                backward = 'Symmetrical'
            else:
                backward = 'Left side'
            if 'ong' in inst['Guide']:
                guide = 'Long'
            elif 'hort' in inst['Guide']:
                guide = 'Short'
            else:
                guide = 'SANS'
            if 'ront' in inst['Cryostat']:
                cryo = 'Front'
            elif 'ack' in inst['Cryostat']:
                cryo = 'Back'
            else:
                cryo = 'None'
            self.instrument['Detectors']['Active'] = actlist
            self.instrument['Detectors']['Forward'] = forward
            self.instrument['Detectors']['Backward'] = backward
            self.instrument['Guide'] = guide
            self.instrument['Cryostat'] = cryo
    #         self.exed.configure(self.param, self.instrument)

    def take_parameters(self, params):
        # param = list_to_par(params)
        param = params
        # step by step, we will check all the values
        # lim = limits["Field"]["Direction"]
        # anglim = self.exed.angle_limit
        # lim = (-anglim, anglim)
        lim = self.exed.angle_limit
        # angs = param["Field"]["Direction"].split(":")
        # if len(angs) > 2:
        #     self.param["Field"]["Direction"] = ":".join([str(d) for d in angs[0:3]])
        #else:
            #try:
                #val = float(param["Field"]["Direction"])
            #except ValueError:
                #val = 0
            #if val >= lim[0] and val <= lim[1]:
                #self.param["Field"]["Direction"] = val
            #else:
                #self.param["Field"]["Direction"] = 0
        try:
            val = float(param["Field"]["Direction"])
        except ValueError:
            val = 0
        if val >= lim[0] and val <= lim[1]:
            self.param["Field"]["Direction"] = val
        else:
            self.param["Field"]["Direction"] = 0
        # choppers
        lim = limits["Lambda"]["Res"]
        try:
            val = float(param["Lambda"]["Res"])
        except ValueError:
            self.param["Lambda"]["Res"] = ""
            self.param["Choppers"]["Frequency"] = 120.0
            self.param["Choppers"]["Suppression"] = 1
        except KeyError:
            self.param["Lambda"]["Res"] = 20.0
            self.param["Choppers"]["Frequency"] = 120.0
            self.param["Choppers"]["Suppression"] = 1
        else:
            if val >= lim[0] and val <= lim[1]:
                self.param["Lambda"]["Res"] = val
                self.param["Choppers"]["Frequency"] = 120.0
                self.param["Choppers"]["Suppression"] = 1
        # wavelength
        lim = limits["Lambda"]["Ei"]
        try:
            val = float(param["Lambda"]["Ei"])
        except ValueError:
            val = 0.0
        except KeyError:
            val = 20.0
        if val >= lim[0] and val <= lim[1]:
            self.param["Lambda"]["Ei"] = val
        else:
            self.param["Lambda"]["Ei"] = 20.0
        #lim = limits["Lambda"]["Min"]
        #try:
            #val = float(param["Lambda"]["Min"])
        #except ValueError:
            #val = 0.7
        #if val >= lim[0] and val <= lim[1]:
            #self.param["Lambda"]["Min"] = val
        #else:
            #self.param["Lambda"]["Min"] = 0.7
        #lim = limits["Lambda"]["Max"]
        #try:
            #val = float(param["Lambda"]["Max"])
        #except ValueError:
            #val = 7.9
        #if val >= lim[0] and val <= lim[1]:
            #self.param["Lambda"]["Max"] = val
        #else:
            #self.param["Lambda"]["Max"] = 7.9
        #if self.param["Lambda"]["Max"] < self.param["Lambda"]["Min"]:
            #self.param["Lambda"]["Max"], self.param["Lambda"]["Min"] = self.param["Lambda"]["Min"], self.param["Lambda"]["Max"]
        # sample
        lim = limits["Sample"]["Constants"]
        val = param["Sample"]["Constants"]
        if ',' in val:
            val = val.split(',')
        else:
            val = val.split()
        for i in range(len(val)):
            try:
                a = float(val[i])
            except ValueError:
                val[i] = 4
        val = np.array(val).astype(np.float)
        if (val >= lim[0]).all() and (val <= lim[1]).all() and len(val) == 3:
            self.param["Sample"]["Constants"] = val
        else:
            self.param["Sample"]["Constants"] = np.array([4.0, 4.0, 4.0])
        lim = limits["Sample"]["Angles"]
        val = param["Sample"]["Angles"]
        if ',' in val:
            val = val.split(',')
        else:
            val = val.split()
        for i in range(len(val)):
            try:
                a = float(val[i])
            except ValueError:
                val[i] = 90
        val = np.array(val).astype(np.float)
        if (val >= lim[0]).all() and (val <= lim[1]).all() and len(val) == 3:
            self.param["Sample"]["Angles"] = val
        else:
            self.param["Sample"]["Angles"] = np.array([90.0, 90.0, 90.0])
        # sample orientation
        lim = limits["Sample"]["u"]
        val = param["Sample"]["u"]
        if ',' in val:
            val = val.split(',')
        else:
            val = val.split()
        for i in range(len(val)):
            try:
                a = float(val[i])
            except ValueError:
                val[i] = 0
        # val = normalise(np.array(val).astype(np.float))
        val = np.array(val).astype(np.float)
        if (val >= lim[0]).all() and (val <= lim[1]).all() and len(val) == 3 and not (val==0.0).all():
            self.param["Sample"]["u"] = val
        else:
            self.param["Sample"]["u"] = np.array([1.0, 0.0, 0.0])
        lim = limits["Sample"]["v"]
        val = param["Sample"]["v"]
        if ',' in val:
            val = val.split(',')
        else:
            val = val.split()
        for i in range(len(val)):
            try:
                a = float(val[i])
            except ValueError:
                val[i] = 0
        # val = normalise(np.array(val).astype(np.float))
        val = np.array(val).astype(np.float)
        if (val >= lim[0]).all() and (val <= lim[1]).all() and len(val) == 3 and not (val==0.0).all():
            self.param["Sample"]["v"] = val
        else:
            self.param["Sample"]["v"] = np.array([0.0, 0.0, 1.0])
        # goniometer angles
        h, k, l = param["Sample"]["phi"], param["Sample"]["chi"], param["Sample"]["omega"]
        hl, kl, ll = limits["Sample"]["phi"], limits["Sample"]["chi"], limits["Sample"]["omega"]
        count = 0
        try:
            val = float(h)
        except ValueError:
            count +=1
            self.param["Sample"]["phi"] = 0
        else:
            if val >= hl[0] and val <= hl[1]:
                self.param["Sample"]["phi"] = val
            else:
                self.param["Sample"]["phi"] = 0.0
        try:
            val = float(k)
        except ValueError:
            count +=1
            self.param["Sample"]["chi"] = 0
        else:
            if val >= kl[0] and val <= kl[1]:
                self.param["Sample"]["chi"] = val
            else:
                self.param["Sample"]["chi"] = 0.0
        try:
            val = float(l)
        except ValueError:
            count +=1
            self.param["Sample"]["omega"] = 0
        else:
            if val >= ll[0] and val <= ll[1]:
                self.param["Sample"]["omega"] = val
            else:
                self.param["Sample"]["omega"] = 0.0
        self.count = count
        # hkl
        try:
            h, k, l, dE = str(param["hkl"]["h"]), str(param["hkl"]["k"]), str(param["hkl"]["l"]), str(param["hkl"]["dE"])
        except KeyError:
            h, k, l, dE = str(param["hkl"]["h"]), str(param["hkl"]["k"]), str(param["hkl"]["l"]), '-1:1'
        hl, kl, ll, dEl = limits["hkl"]["h"], limits["hkl"]["k"], limits["hkl"]["l"], limits["hkl"]["dE"]
        count = 0
        hs = h.split(':')
        if len(hs) > 1:
            temp = []
            for p in hs:
                try:
                    val = float(p)
                except ValueError:
                    val = 0.0
                temp.append(val)
            self.param["hkl"]["h"] = ":".join([str(d) for d in temp[0:3]])
            self.hrange = temp
            count += 1
        else:
            try:
                val = float(h)
            except ValueError:
                count +=1
                self.param["hkl"]["h"] = 'h'
                self.hrange = []
            else:
                if val >= hl[0] and val <= hl[1]:
                    self.param["hkl"]["h"] = val
                    self.hrange = []
                else:
                    self.param["hkl"]["h"] = 0.0
                    self.hrange = []
        ks = k.split(':')
        if len(ks) > 1:
            temp = []
            for p in ks:
                try:
                    val = float(p)
                except ValueError:
                    val = 0.0
                temp.append(val)
            self.param["hkl"]["k"] = ":".join([str(d) for d in temp[0:3]])
            self.krange = temp
            count += 1
        else:
            try:
                val = float(k)
            except ValueError:
                count +=1
                self.param["hkl"]["k"] = 'k'
                self.krange = []
            else:
                if val >= kl[0] and val <= kl[1]:
                    self.param["hkl"]["k"] = val
                    self.krange = []
                else:
                    self.param["hkl"]["k"] = 0.0
                    self.krange = []
        ls = l.split(':')
        if len(ls) > 1:
            temp = []
            for p in ls:
                try:
                    val = float(p)
                except ValueError:
                    val = 0.0
                temp.append(val)
            self.param["hkl"]["l"] = ":".join([str(d) for d in temp[0:3]])
            self.lrange = temp
            count += 1
        else:
            try:
                val = float(l)
            except ValueError:
                count +=1
                self.param["hkl"]["l"] = 'l'
                self.lrange = []
            else:
                if val >= ll[0] and val <= ll[1]:
                    self.param["hkl"]["l"] = val
                    self.lrange = []
                else:
                    self.param["hkl"]["l"] = 0.0
                    self.lrange = []
        dEs = dE.split(':')
        if len(dEs) > 1:
            temp = []
            for p in dEs:
                try:
                    val = float(p)
                except ValueError:
                    if p == 'max':
                        val = self.param["Lambda"]["Ei"]
                    else:
                        val = 0.0
                temp.append(val)
            self.param["hkl"]["dE"] = ":".join([str(d) for d in temp[0:3]])
            self.dErange = temp
            count += 1
        else:
            try:
                val = float(dE)
            except ValueError:
                count +=1
                self.param["hkl"]["dE"] = '0'
                self.dErange = []
            else:
                if val >= ll[0] and val <= ll[1]:
                    self.param["hkl"]["dE"] = val
                    self.dErange = []
                else:
                    self.param["hkl"]["dE"] = 0.0
                    self.dErange = []
        self.count = count
        
    def marker_points(self, mark_points):
        new = []
        self.markers = []
        for val in mark_points:
            if ',' in val:
                nval = val.split(',')
            else:
                nval = val.split()
            if len(nval) == 4:
                try:
                    a = float(nval[0])
                    b = float(nval[1])
                    c = float(nval[2])
                    d = float(nval[3])
                except KeyError:
                    a = float(nval[0])
                    b = float(nval[1])
                    c = float(nval[2])
                    d = 0.0
                except ValueError:
                    new.append("")
                else:
                    nval = np.array(nval).astype(np.float)
                    new.append(" ".join([str(x)[:7] for x in [a,b,c,d]]))
                    self.markers.append(nval)
            else:
                new.append("")
        return new
        
    def set_precision(self, value):
        val = str(value)
        if ',' in val:
            nval = val.split(',')
        else:
            nval = val.split()
        if len(nval) == 1:
            try:
                a = float(nval[0])
            except ValueError:
                return str(self.multiplier)[:5]
            else:
                nval = float(nval[0])
        if (nval >= 0.5) and (nval <= 3.0):
            self.multiplier = nval
        self.exed.segments = 2*int(round(3 + 2.0*self.multiplier))
        # self.exed_hc.segments = 2*int(round(3 + 2.0*self.multiplier))
        return str(self.multiplier)[:5]
    
    def current_parameters(self):
        return self.param
    
    def current_parameters_str(self, filename = ""):
        param = copy.deepcopy(self.param)
        markers = []
        prec = -1
        for i in self.param.items():
            name1 = i[0]
            for j in i[1].items():
                name2 = j[0]
                try:
                    l = len(j[1])
                except TypeError:
                    l = 0
                if l:
                    if name1 == "hkl":
                        param[name1][name2] = []
                        val = j[1]
                        try:
                            val.split()
                        except AttributeError:
                            for k in range(len(val)):
                                if abs(val[k] - round(val[k])) < 1e-8:
                                    param[name1][name2].append(str(int(val[k])))
                                else:
                                    param[name1][name2].append(str(val[k]))
                            param[name1][name2] = ":".join(param[name1][name2])
                        else:
                            try:
                                round(val)
                            except TypeError:
                                param[name1][name2] = val
                            else:
                                if abs(val - round(val)) < 1e-8:
                                    param[name1][name2] = str(int(val))
                                else:
                                    param[name1][name2] = str(val)
                    else:
                        param[name1][name2] = []
                    # print( name1, name2
                        for k in range(l):
                            param[name1][name2].append("")
                            val = j[1][k]
                        # print( name1, name2, val
                            try:
                                round(val)
                            except TypeError:
                                param[name1][name2] = val
                            else:
                                if abs(val - round(val)) < 1e-8:
                                    param[name1][name2][k] = str(int(val))
                                else:
                                    param[name1][name2][k] = str(val)
                        param[name1][name2] = " ".join(param[name1][name2])
                    # print( param[name1][name2]
                else:
                    val = j[1]
                    # print( name1, name2, val
                    try:
                        round(val)
                    except TypeError:
                        param[name1][name2] = val
                    else:
                        if abs(val - round(val)) < 1e-8:
                            param[name1][name2] = str(int(val))
                        else:
                            param[name1][name2] = str(val)
        for i in range(len(self.markers)):
            try:
                l = len(self.markers[i])
            except TypeError:
                l = 0
            if l:
                markers.append([])
                for k in range(l):
                    val = self.markers[i][k]
                    if abs(val - round(val)) < 1e-8:
                        markers[-1].append(str(int(val)))
                    else:
                        markers[-1].append(str(val))
                markers[-1] = " ".join(markers[-1])
            else:
                markers.append([])
                val = j[1]
                if abs(val - round(val)) < 1e-8:
                    markers[-1].append(str(int(val)))
                else:
                    markers[-1].append(str(val))
        val = self.multiplier
        if abs(val - round(val)) < 1e-8:
            prec = (str(int(val)))
        else:
            prec = (str(val))
        if filename:
            dump = open(filename, 'w')
            for i in param.items():
                for j in i[1].items():
                    dump.write(i[0]+"_"+j[0]+"="+j[1]+'\n')
            #for i, j in enumerate(markers):
            #    dump.write("Point"+str(i)+"="+j+'\n')
            for i in range(5):
                try:
                    v = markers[i]
                except IndexError:
                    v = ""
                    markers.append("")
                dump.write("Point"+str(i+1)+"="+v+'\n')
            dump.write("Precision="+prec+'\n')
            dump.close()
        return param, markers, prec
    
    def request_parameters(self):
        fields = [
        ["Field","Direction"],
        # ["Choppers","Frequency"],
        # ["Choppers","Suppression"],
        ["Lambda","Ei"],
        ["Lambda","Res"],
        # ["Lambda","Min"],
        # ["Lambda","Max"],
        ["Sample","Constants"],
        ["Sample","Angles"],
        ["Sample","u"],
        ["Sample","v"],
        ["Sample","phi"],
        ["Sample","chi"],
        ["Sample","omega"],
        ["hkl","h"],
        ["hkl","k"],
        ["hkl","l"],
        ["hkl","dE"]]
        return fields

    def chopper_freqs(self):
        Ei, dEi = self.param["Lambda"]["Ei"], self.param["Lambda"]["Res"]
        freq = INS_chopper_speed(Ei, dEi/100.0 * Ei, dist = 4.5, Ld = 0.01, CH1_window = 18.5, CH6_window = 2.6)
        altfreq = INS_chopper_speed(Ei, dEi/100.0 * Ei, dist = 4.5, Ld = 0.01, CH1_window = 55.5, CH6_window = 9.23)
        if altfreq < 7200:
            which = 'Large windows, high flux'
            freq = altfreq
        elif freq > 7200:
            which = '1 pulse in 2'
        else:
            which = ''
        self.fpulse = freq
        self.fband = freq
        self.choppertype = which
        self.redfactor = intensity_drop_INS(freq, freq, which)
        return which, freq, freq

    def ins_resolution(self):
        Ei = self.param["Lambda"]["Ei"]
        if self.choppertype == 'Large windows, high flux':
            ch1window, ch6window = 55.5, 9.23
        else:
            ch1window, ch6window = 18.5, 2.6
        dE, Emax = INS_resolution(self.fband, Ei, CH1_window = ch1window, CH6_window = ch6window)
        exed = self.exed
        lam_i = n_lambda(Ei)
        flux = '{0:5.1e}'.format(self.redfactor * exed.spectrum_longguide(lam_i))
        if self.fband > 12000:
            flux = '0'
        return flux, dE, Emax
        
    def uv_range(self):
        parameters = self.param
        field = -parameters['Field']['Direction']
        bu = parameters["Sample"]["u"]
        bv = parameters["Sample"]["v"]
        gonio = [parameters["Sample"]["phi"], parameters["Sample"]["chi"], parameters["Sample"]["omega"]]
        sss = Sample(parameters["Sample"]["Constants"],parameters["Sample"]["Angles"])
        sss.orient(bu, bv, field, gonio)
        u, v = sss.u, sss.v
        # axis = np.cross(v, u)
        ## axis = np.array([0,0,1.0])
        # rmat = np.dot(np.dot(sss.Binv, arb_rotation(axis, rot)), sss.B)
        ## rotmin = np.dot(np.dot(sss.Binv, arb_rotation(axis, -15.0 - field)), sss.B)
        ## rotmax = np.dot(np.dot(sss.Binv, arb_rotation(axis, 15.0 - field)), sss.B)
        sss.orient(bu, bv, -11.85, gonio)
        newumin, newvmin = sss.u, sss.v
        sss.orient(bu, bv, 2.4, gonio)
        newumax, newvmax = sss.u, sss.v
        #newumax = np.dot(rotmax, u)
        #newumin = np.dot(rotmin, u)
        #newvmax = np.dot(rotmax, v)
        #newvmin = np.dot(rotmin, v)
        vecs = [newumax, newumin, newvmax, newvmin]
        newvecs = []
        for i in range(len(vecs)):
            try:
                l = len(vecs[i])
            except TypeError:
                l = 0
            if l:
                newvecs.append([])
                for k in range(l):
                    val = vecs[i][k]
                    if val > 0:
                        lim = 5
                    else:
                        lim = 6
                    if abs(val - round(val)) < 1e-8:
                        newvecs[-1].append(str(int(val)))
                    else:
                        newvecs[-1].append(str(val)[:lim])
                newvecs[-1] = " ".join(newvecs[-1])
            else:
                newvecs.append([])
                val = vecs[1]
                if val > 0:
                    lim = 5
                else:
                    lim = 6
                if abs(val - round(val)) < 1e-8:
                    newvecs[-1].append(str(int(val)))
                else:
                    newvecs[-1].append(str(val)[:5])
        return newvecs
        
    def show_slider_plot(self, outFile = ""):
        param = self.param
        exed = self.exed
        plot.inverse_plot(param, outFile)
        return True
        
    def show_total_eq_range(self, outFile = ""):
        self.update_intensity()
        coverage_fw = self.coverage_INS(forward = True, powder = True)
        coverage_bw = self.coverage_INS(forward = False, powder = True)
        plot.plot_eq_map(coverage_fw, coverage_bw, outFile, self.figure)
        return True
    
    def show_help(self, outFile = ""):
        plot.plot_help_message(outFile, self.figure)
        return True
    
    def show_welcome_message(self, outFile = ""):
        plot.plot_welcome_message(outFile, self.figure)
        return True
    
    def show_uv_range(self, outFile = ""):
        a = self.uv_range()
        plot.plot_uv_range(a, outFile, self.figure)
        return True
    
    def show_instrument(self, outFile = ""):
        param = self.param
        exed = self.exed
        ori = param["Field"]["Direction"]
        samp = Sample(param["Sample"]["Constants"], param["Sample"]["Angles"])
        gonio = [param["Sample"]["phi"], param["Sample"]["chi"], param["Sample"]["omega"]]
        if not samp.orient(param["Sample"]["u"], param["Sample"]["v"], -param["Field"]["Direction"], gonio):
            plot.error(outFile, self.figure)
            return False
        u, v = samp.u, samp.v
        vecs = [u, v]
        newvecs = []
        for i in range(len(vecs)):
            try:
                l = len(vecs[i])
            except TypeError:
                l = 0
            if l:
                newvecs.append([])
                for k in range(l):
                    val = vecs[i][k]
                    if val > 0:
                        lim = 5
                    else:
                        lim = 6
                    if abs(val - round(val)) < 1e-8:
                        newvecs[-1].append(str(int(val)))
                    else:
                        newvecs[-1].append(str(val)[:lim])
                newvecs[-1] = " ".join(newvecs[-1])
            else:
                newvecs.append([])
                val = vecs[1]
                if val > 0:
                    lim = 5
                else:
                    lim = 6
                if abs(val - round(val)) < 1e-8:
                    newvecs[-1].append(str(int(val)))
                else:
                    newvecs[-1].append(str(val)[:5])
        u, v = newvecs[0], newvecs[1]
        alltubes, goodtubes, shadowtubes = [], [], []
        for i in exed.current_panels_at(int(round(ori))):
            if i == []:
                continue
            for k in i.tubes:
                alltubes.append(k.frame)
                goodtubes += k.goodparts
                shadowtubes += k.badparts
        tubes = (alltubes, goodtubes, shadowtubes)
        # tubes = exed.tubes[exed.minangle - int(round(ori))]
        a = self.uv_range()
        a.append(u)
        a.append(v)
        plot.plot_instrument(ori, tubes, a, outFile = outFile, fig = self.figure, goniometer = gonio)
        return True
    
    def show_detector_panels(self, forward = True, outFile = ""):
        param = self.param
        exed = self.exed
        samp = Sample(param["Sample"]["Constants"], param["Sample"]["Angles"])
        ori = param["Field"]["Direction"]
        gonio = [param["Sample"]["phi"], param["Sample"]["chi"], param["Sample"]["omega"]]
        if not samp.orient(param["Sample"]["u"], param["Sample"]["v"], -param["Field"]["Direction"], gonio):
            plot.error(outFile, self.figure)
            return False
        u, v = samp.u, samp.v
        vecs = [u, v]
        rotmat = samp.themat
        if forward:
            det_dist, h_angles, v_angles, cone_angle, beamstop = exed.limits_forward(ori)
        else:
            det_dist, h_angles, v_angles, cone_angle, beamstop = exed.limits_backward()
        if forward:
            x_axis = np.linspace(ori - 16.0, ori + 16.0, 200)
            y_axis = np.linspace(-1.4, 1.4, 200)
        else:
            # x_axis = np.concatenate([np.linspace(-165.0, -180.0, 100), np.linspace(0.0, 15.0, 100)])
            x_axis = np.linspace(155.0, 205.0, 200)
            y_axis = np.linspace(-0.55, 0.55, 200)

        vectors = np.zeros((len(x_axis),len(y_axis),2))
        vectors[:,:,0] += x_axis.reshape((len(x_axis),1))
        vectors[:,:,1] += y_axis.reshape((1, len(y_axis)))
        
        allpoints = vectors.reshape((len(x_axis)*len(y_axis),2))
        the_image = np.ones((len(x_axis)*len(y_axis)))
        # cone angle:
        newones = np.arctan2(np.sqrt(allpoints[:,1]**2 + (np.tan(np.radians(allpoints[:,0]-ori - 0.3))*det_dist)**2), det_dist)
        if forward:
            the_image[np.where(newones > cone_angle)] = 0
        else:
            the_image[np.where(newones > np.pi - cone_angle)] = 0
        print( "cone angles min, max, limit : ", newones.min(), newones.max(), cone_angle)
        # det. edges
        temp_result = np.radians(allpoints[:,0]) #  + np.pi/2.0
        if forward:
            the_image[np.where(temp_result < -h_angles[-1])] = 0.0
            the_image[np.where(temp_result > -h_angles[0])] = 0.0
            print( "horizontal angles min, max, limits : ", temp_result.min(), temp_result.max(), h_angles[-1], h_angles[0])
        else:
            extratemp = temp_result.copy()#  + np.pi/2.0
            extratemp[np.where(extratemp < 0)] += 2*np.pi
            ang1, ang2 = h_angles[0], h_angles[-1]
            if ang1 < 0:
                ang1 += 2*np.pi
            if ang2 < 0:
                ang2 += 2*np.pi
            the_image[np.where(temp_result < ang1)] = 0.0
            the_image[np.where(temp_result > ang2)] = 0.0
            print( "Choose between these angles: ", ang1, ang2)
            print( "horizontal angles min, max, limits : ", temp_result.min(), temp_result.max(), ang1, ang2)
        if forward:
            for ann in range(int(len(h_angles)/2) - 1):
                the_image[np.where((temp_result < -h_angles[2*ann+1])*(temp_result > -h_angles[2*ann+2]) )] = 0.0
                print( "Discard between these angles: ", h_angles[2*ann+1], h_angles[2*ann+2])
        else:
            for ann in range(int(len(h_angles)/2) - 1):
                extratemp = temp_result.copy()# + np.pi/2.0
                extratemp[np.where(extratemp < 0)] += 2*np.pi
                ang1, ang2 = h_angles[2*ann+1], h_angles[2*ann+2]
                if ang1 < 0:
                    ang1 += 2*np.pi
                if ang2 < 0:
                    ang2 += 2*np.pi
                the_image[np.where((extratemp > ang1)*(extratemp < ang2) )] = 0.0
                print( "Discard between these angles: ", ang1, ang2)
        vert_result = np.tan(allpoints[:,1]/det_dist)
        if forward:
            print( "################")
            print( "Applying beamstop")
            print( "Beamstop angles: ", beamstop)
            print( "Vertical angles: ,", vert_result.min(), vert_result.max())
            # the_image[np.where((temp_result > (beamstop[2] + np.pi/2.0)) * (temp_result < beamstop[3] + np.pi/2.0) * (vert_result < beamstop[0]) * (vert_result > beamstop[1]))] = 0.0
            the_image[np.where((temp_result > (beamstop[2])) * (temp_result < beamstop[3]) * (vert_result < beamstop[0]) * (vert_result > beamstop[1]))] = 0.0
        else:
            print( "################")
            print( "Applying vertical angles")
            the_image[np.where(vert_result < -v_angles.min())] = 0.0
            cryo = np.tan([[206.0/1846.0],[(76.0 - 20.0)/(1846.0 - 1264.0)]])
            the_image[np.where(vert_result > cryo.min())] = 0.0
        the_image = the_image.reshape((len(x_axis), len(y_axis)))
        Ei = param["Lambda"]["Ei"]
        lam_i = n_lambda(Ei)
        markers = self.markers
        endmarkers = []
        if len(markers) > 0:
            start = np.row_stack(markers)
            points = rotate(rotmat, start[:,:3])
            local_res = []
            k_f = points.copy()
            q_z = n_k(n_lambda(Ei))
            k_f[:,2] += q_z
            # kf has been found, now we need the cone angle
            newones = cart_to_sph(rotate(rotY(-ori+0.3), k_f))
            newones = np.arctan2(np.sqrt(newones[:,0]**2 + newones[:,1]**2), newones[:,2])
            # as easy as this; now we can clip the point using the magnet cone opening
            # for each point we need the energy resolution
            resolution = []
            flux = 0.0
            if self.choppertype == 'Large windows, high flux':
                ch1window, ch6window = 55.5, 9.23
            else:
                ch1window, ch6window = 18.5, 2.6
            if forward:
                detector_distance = 4.44
            else:
                detector_distance = 3.4
            for x in start[:,3]:
                res, dump = INS_resolution(self.fband, Ei, dist = detector_distance, Ld= 0.01, CH1_window = ch1window, CH6_window = ch6window, Ef =  Ei - x)
                resolution.append(res)
            # we have the resolution at each point
            flux = self.redfactor * exed.spectrum_longguide(lam_i)
            # and now we have the flux too
            temp_result, temp_angles = reverse_q_vector_INS(points, Ei)
            # temp_result is |kf|, alpha, beta, temp_angles is cone angle, not needed
            calc_dE = Ei - n_energy(n_lambda_fromk(temp_result[:,0]))
            temp_values = np.ones((len(temp_result))) # logical values showing which point is visible
            # here the trimming occurs, where the marker points can be removed for all the reasons
            if forward:
                temp_values[np.where(newones > cone_angle)] = 0
            else:
                temp_values[np.where(newones > np.pi - cone_angle)] = 0
            temp_values[np.where(temp_result[:,1] < h_angles[0])] = 0.0
            temp_values[np.where(temp_result[:,1] > h_angles[-1])] = 0.0
            for ann in range(int(len(h_angles)/2) - 1):
                temp_values[np.where((temp_result[:,1] > h_angles[2*ann+1])*(temp_result[:,1] < h_angles[2*ann+2]) )] = 0.0
            for num, val in enumerate(resolution):
                if not temp_values[num]:
                    continue
                if abs((start[num, 3]) - calc_dE[num]) > val:
                    continue
                else:
                    endmarkers.append([np.degrees(temp_result[num,1]), np.tan(temp_result[num,2])*det_dist, None,
                                       calc_dE[num], val, markers[num]])
                    print( "output", endmarkers[-1])
        # tubes = exed.tubes[int(round(ori))-exed.minangle]
        # tubes, goodtubes, shadowtubes = exed.widetubes[int(round(ori))-exed.minangle]
        plot.plot_detector_panels(ori, [the_image, x_axis, y_axis],
                                  outFile = outFile, fig = self.figure,
                                  markers = endmarkers)
        return True

    def update_intensity(self):
        ttt, tttt, ttttt = self.chopper_freqs()
        match = 0
        try:
            freqa = float(self.fpulse)
        except ValueError:
            freqa = 1.0
        try:
            freqb = float(self.fband)
        except ValueError:
            freqb = 1.0
        try:
            float(self.param["Lambda"]["Res"])
        except ValueError:
            redfactor = 1.0
        else:
            redfactor = intensity_drop_INS(freqa, freqb, self.choppertype)
        self.redfactor = redfactor
     
    def guess_hkl_range(self, samp, panels, vecs, waves):
        lmin, lmax = waves[0], waves[1]
        vec1, vec2, origin = vecs[0], vecs[1], vecs[2]
        points = []
        rotmat = np.linalg.inv(samp.themat)
        for i in panels:
            for j in i.tubes:
                for knt in j.goodparts:
                    # k = j.angular
                    if len(knt) > 0:
                        k = tubepoints(knt)
                        line_a, line_b = k.copy(), k.copy()
                        # print( line_a
                        line_a[:,0] = n_k(lmin)
                        line_b[:,0] = n_k(lmax)
                        line_a, line_b = q_vector(line_a), q_vector(line_b)
                        points.append(rotate(rotmat, line_a))
                        points.append(rotate(rotmat, line_b))
        points = np.row_stack(points)
        pmax, pmin = points.max(0), points.min(0)
        range1 = (pmin[np.where(vec1 > 0.0)], pmax[np.where(vec1 > 0.0)])
        range2 = (pmin[np.where(vec2 > 0.0)], pmax[np.where(vec2 > 0.0)])
        return range1, range2
    
    def guess_hkl_range_3D(self, samp, panels, vecs, waves):
        
        lmin, lmax = waves[0], waves[1]
        vec1, vec2, vec3 = vecs[0], vecs[1], vecs[2]
        points = []
        rotmat = np.linalg.inv(samp.themat)
        for i in panels:
            for j in i.tubes:
                line_a, line_b = j.qvectors(lmin), j.qvectors(lmax)
                points.append(rotate(rotmat, line_a))
                points.append(rotate(rotmat, line_b))
        total = 0
        for i in points:
            total += len(i)
        if total > 0:
            points = np.row_stack(points)
            pmax, pmin = points.max(0), points.min(0)
            range1 = [pmin[0], pmax[0]]
            range2 = [pmin[1], pmax[1]]
            range3 = [pmin[2], pmax[2]]
        else:
            range1 = [1.0, 2.0]
            range2 = [1.0, 2.0]
            range3 = [1.0, 2.0]
        return range1, range2, range3

    def guess_hkl_range_INS(self, forward = True):
        param = self.param
        exed = self.exed
        ori = param["Field"]["Direction"]
        samp = Sample(param["Sample"]["Constants"], param["Sample"]["Angles"])
        gonio = [param["Sample"]["phi"], param["Sample"]["chi"], param["Sample"]["omega"]]
        if not samp.orient(param["Sample"]["u"], param["Sample"]["v"], -param["Field"]["Direction"], gonio):
            plot.error(outFile, self.figure)
            return False
        u, v = samp.u, samp.v
        vecs = [u, v]
        if forward:
            det_dist, h_angles, v_angles, cone_angle, beamstop = exed.limits_forward(ori)
        else:
            det_dist, h_angles, v_angles, cone_angle, beamstop = exed.limits_backward()
        Ei = param["Lambda"]["Ei"]
        Ef_max = 3*Ei
        Ef_min = n_energy_from_TOF(60.0/self.fband, det_dist)
        points = []
        for n, a in enumerate(h_angles):
            points.append([a, v_angles[n]])
            points.append([a, -v_angles[n]])
        points = np.array(points)
        rotmat = np.linalg.inv(samp.themat)
        group_a, group_b = q_vector_INS(points, Ei, Ef_min), q_vector_INS(points, Ei, Ef_max)
        result = np.row_stack([rotate(rotmat, group_a), rotate(rotmat, group_b)])
        total = len(result)
        if total > 0:
            points = np.row_stack(result)
            pmax, pmin = points.max(0), points.min(0)
            range1 = [pmin[0], pmax[0]]
            range2 = [pmin[1], pmax[1]]
            range3 = [pmin[2], pmax[2]]
            range4 = np.array([Ef_min, Ef_max]) - params["Lambda"]["MaxLoss"]
        else:
            range1 = [1.0, 2.0]
            range2 = [1.0, 2.0]
            range3 = [1.0, 2.0]
            range4 = [1.0, 2.0]
        print( range1, range2, range3, range4)
        return range1, range2, range3, range4
    
    def coverage_INS(self, forward = True, powder = False):
        param = self.param
        exed = self.exed
        ori = param["Field"]["Direction"]
        samp = Sample(param["Sample"]["Constants"], param["Sample"]["Angles"])
        gonio = [param["Sample"]["phi"], param["Sample"]["chi"], param["Sample"]["omega"]]
        if not samp.orient(param["Sample"]["u"], param["Sample"]["v"], -param["Field"]["Direction"], gonio):
            plot.error(outFile, self.figure)
            return False
        u, v = samp.u, samp.v
        vecs = [u, v]
        rotmat = samp.themat
        print( "### The calculations start here ###")
        print( "Is the direction forward? ", forward)
        if powder:
            ori = -12.0
        if forward:
            det_dist, h_angles, v_angles, cone_angle, beamstop = exed.limits_forward(ori)
        else:
            det_dist, h_angles, v_angles, cone_angle, beamstop = exed.limits_backward()
        # lmin, lmax = (param["Lambda"]["Min"], param["Lambda"]["Max"])
        Ei = param["Lambda"]["Ei"]
        ranges = []
        range_h, range_k, range_l, range_dE = self.guess_hkl_range_INS(forward)
        print( "Guessed range of h, k, l, dE: ", range_h, range_k, range_l, range_dE)
        h, k, l, dE = str(param["hkl"]["h"]), str(param["hkl"]["k"]), str(param["hkl"]["l"]), str(param["hkl"]["dE"])
        if len(h.split(':')) > 2:
            nums = [float(x) for x in h.split(':')]
            if nums[1] == 0:
                nums[1] = (nums[2]-nums[0])/20.0
            steps_h = max(int(round((nums[-1] - nums[0]) / nums[1])),1)
            range_h = [nums[0], nums[0] + steps_h * nums[1]]
            ranges.append('Axis : h')
        elif len(h.split(':')) > 1:
            nums = [float(x) for x in h.split(':')]
            range_h = [nums[0], nums[-1]]
            steps_h = 1
            ranges.append('Integrated : h '+ str(round(nums[0],2)) + 'to' + str(round(nums[-1],2)))
        else:
            steps_h = 1
            ranges.append('Integrated : h '+ str(round(range_h[0],2)) + 'to' + str(round(range_h[-1],2)))
        if len(k.split(':')) > 2:
            nums = [float(x) for x in k.split(':')]
            if nums[1] == 0:
                nums[1] = (nums[2]-nums[0])/20.0
            steps_k = max(int(round((nums[-1] - nums[0]) / nums[1])),1)
            range_k = [nums[0], nums[0] + steps_k * nums[1]]
            ranges.append('Axis : k')
        elif len(k.split(':')) > 1:
            nums = [float(x) for x in k.split(':')]
            range_k = [nums[0], nums[-1]]
            steps_k = 1
            ranges.append('Integrated : k '+ str(round(nums[0],2)) + 'to' + str(round(nums[-1],2)))
        else:
            steps_k = 1
            ranges.append('Integrated : k '+ str(round(range_k[0],2)) + 'to' + str(round(range_k[-1],2)))
        if len(l.split(':')) > 2:
            nums = [float(x) for x in l.split(':')]
            if nums[1] == 0:
                nums[1] = (nums[2]-nums[0])/20.0
            steps_l = max(int(round((nums[-1] - nums[0]) / nums[1])),1)
            range_l = [nums[0], nums[0] + steps_l * nums[1]]
            ranges.append('Axis : l')
        elif len(l.split(':')) > 1:
            nums = [float(x) for x in l.split(':')]
            range_l = [nums[0], nums[-1]]
            steps_l = 1
            ranges.append('Integrated : l '+ str(round(nums[0],2)) + 'to' + str(round(nums[-1],2)))
        else:
            steps_l = 1
            ranges.append('Integrated : l '+ str(round(range_l[0],2)) + 'to' + str(round(range_l[-1],2)))
        if len(dE.split(':')) > 2:
            nums = [float(x) for x in dE.split(':')]
            if nums[1] == 0:
                nums[1] = (nums[2]-nums[0])/20.0
            steps_dE = max(int(round((nums[-1] - nums[0]) / nums[1])),1)
            range_dE = [nums[0], nums[0] + steps_dE * nums[1]]
            ranges.append('Axis : dE')
        elif len(dE.split(':')) > 1:
            nums = [float(x) for x in dE.split(':')]
            range_dE = [nums[0], nums[-1]]
            steps_dE = 1
            ranges.append('Integrated : dE '+ str(round(nums[0],2)) + 'to' + str(round(nums[-1],2)))
        else:
            steps_dE = 1
            ranges.append('Integrated : h '+ str(round(range_dE[0],2)) + 'to' + str(round(range_dE[-1],2)))
        axes = np.array([True, True, True, True])
        shape = []
        if steps_h > 1:
            shape.append(steps_h)
        else:
            axes[0] = False
        if steps_k > 1:
            shape.append(steps_k)
        else:
            axes[1] = False
        if steps_l > 1:
            shape.append(steps_l)
        else:
            axes[2] = False
        if steps_dE > 1:
            shape.append(steps_dE)
        else:
            axes[3] = False
        dimensionality = axes.sum()
        print( "The requested output is " + str(dimensionality) + "-dimensional")
        # we may want to precondition this, so that more precision can be gained in one direction an the expense of another
        gridsize = steps_h * steps_k * steps_l * steps_dE
        refsize = 80**4
        newstep = int(max(int(round((float(refsize) / float(gridsize))**(1.0/(4-dimensionality)) / 10.0)*10), 10)*self.multiplier)
        print( "###################")
        print( "gridsize was ", gridsize)
        print( "newstep is ", newstep)
        if axes[0]:
            part1 = np.linspace(range_h[0], range_h[1], steps_h + 1)
            part1 = 0.5*(part1[1:]+part1[:-1])
        else:
            part1 = np.linspace(range_h[0], range_h[1], newstep)
        if axes[1]:
            part2 = np.linspace(range_k[0], range_k[1], steps_k + 1)
            part2 = 0.5*(part2[1:]+part2[:-1])
        else:
            part2 = np.linspace(range_k[0], range_k[1], newstep)
        if axes[2]:
            part3 = np.linspace(range_l[0], range_l[1], steps_l + 1)
            part3 = 0.5*(part3[1:]+part3[:-1])
        else:
            part3 = np.linspace(range_l[0], range_l[1], newstep)
        if axes[3]:
            part4 = np.linspace(range_dE[0], range_dE[1], steps_dE + 1)
            part4 = 0.5*(part4[1:]+part4[:-1])
        else:
            part4 = np.linspace(range_dE[0], range_dE[1], newstep)
        en_step_half = (part4[1:] - part4[:-1]).mean()*0.5
        print( "new gridsize is ", len(part1) * len(part2) * len(part3) * len(part4))
        # difficulties start here
        if not powder:
            the_volume = np.zeros((len(part1), len(part2), len(part3), len(part4)), dtype = np.float32)
        # first a working solution, then we can optimise
        vectors = np.zeros((len(part1),len(part2),len(part3),3), dtype = np.float32)
        vectors[:,:,:,0] += part1.reshape((len(part1),1,1))
        vectors[:,:,:,1] += part2.reshape((1, len(part2),1))
        vectors[:,:,:,2] += part3.reshape((1, 1, len(part3)))
        linvectors = vectors.reshape((len(part1)*len(part2)*len(part3), 3))
        points = rotate(rotmat, linvectors)
        
        k_f = points.copy()
        q_z = n_k(n_lambda(Ei))
        k_f[:,2] += q_z
        # newones = cart_to_sph(rotate(rotY(-ori+0.3), k_f))
        newones = rotate(rotY(-ori+0.3), k_f)
        newones = np.arctan2(np.sqrt(newones[:,0]**2 + newones[:,1]**2), newones[:,2])
        
        temp_result, temp_angles = reverse_q_vector_INS(points, Ei)
        calc_dE = Ei - n_energy(n_lambda_fromk(temp_result[:,0]))
        temp_values = np.ones((len(temp_result)))
        dump1,  dump2,  Emax = self.ins_resolution()
        temp_values[np.where(calc_dE > Emax)] = 0.0
        #print( "#################"
        #print( "Highest polar angle: ", temp_angles.max()
        #print( "Lowest polar angle: ", temp_angles.min()
        #print( "Cone cutoff angle: ", cone_angle
        # temp_values[np.where(temp_angles > cone_angle)] = 0
        print( "#################")
        print( "Highest polar angle: ", newones.max())
        print( "Lowest polar angle: ", newones.min())
        print( "Cone cutoff angle: ", cone_angle)
        if forward:
            temp_values[np.where(newones > cone_angle)] = 0
        else:
            temp_values[np.where(newones < cone_angle)] = 0
        if len(np.where(temp_values)[0]) > 0:
            print( "#################")
            print( "Highest in-plane angle: ", temp_result[:,1][np.where(temp_values)].max())
            print( "Lowest in-plane angle: ", temp_result[:,1][np.where(temp_values)].min())
            print( "Detector edge angles: ", h_angles[0], h_angles[-1])
        else:
            print( "#################")
            print( 'No points in range')
            print( "#################")
        if forward:
            temp_values[np.where(temp_result[:,1] < h_angles[0])] = 0.0
            temp_values[np.where(temp_result[:,1] > h_angles[-1])] = 0.0
        else:
            extratemp = temp_result[:,1].copy() # + np.pi/2.0
            extratemp[np.where(extratemp < 0)] += 2*np.pi
            ang1, ang2 = h_angles[0], h_angles[-1]
            if ang1 < 0:
                ang1 += 2*np.pi
            if ang2 < 0:
                ang2 += 2*np.pi
            temp_values[np.where(extratemp < ang1)] = 0.0
            temp_values[np.where(extratemp > ang2)] = 0.0
            print( "Choose between these angles: ", ang1, ang2)
            print( "horizontal angles min, max, limits : ", extratemp.min(), extratemp.max(), ang1, ang2)
        if forward:
            for ann in range(int(len(h_angles)/2) - 1):
                temp_values[np.where((temp_result[:,1] > h_angles[2*ann+1])*(temp_result[:,1] < h_angles[2*ann+2]) )] = 0.0
                print( "Discard between these angles: ", h_angles[2*ann+1], h_angles[2*ann+2])
        else:
            for ann in range(int(len(h_angles)/2) - 1):
                extratemp = temp_result[:,1].copy() # + np.pi/2.0
                extratemp[np.where(extratemp < 0)] += 2*np.pi
                ang1, ang2 = h_angles[2*ann+1], h_angles[2*ann+2]
                if ang1 < 0:
                    ang1 += 2*np.pi
                if ang2 < 0:
                    ang2 += 2*np.pi
                temp_values[np.where((extratemp > ang1)*(extratemp < ang2) )] = 0.0
                print( "Discard between these angles: ", ang1, ang2)
        if forward:
            print( "################")
            print( "Applying beamstop")
            temp_values[np.where((temp_result[:,1] > beamstop[2]) * (temp_result[:,1] < beamstop[3]) * (temp_result[:,2] < beamstop[0]) * (temp_result[:,2] > beamstop[1]))] = 0.0
            # temp_values[np.where((temp_result[:,1] > (beamstop[2] + np.pi/2.0)) * (temp_result[:,1] < beamstop[3] + np.pi/2.0) * (temp_result[:,2] < beamstop[0]) * (temp_result[:,2] > beamstop[1]))] = 0.0
        else:
            print( "################")
            print( "Applying vertical angles")
            temp_values[np.where(temp_result[:,2] < -v_angles.min())] = 0.0
            cryo = np.tan([[206.0/1846.0],[(76.0 - 20.0)/(1846.0 - 1264.0)]])
            temp_values[np.where(temp_result[:,2] > cryo.min())] = 0.0
        if powder:
            result = np.column_stack([np.sqrt((points**2).sum(1)), calc_dE])[np.where(temp_values)]
            return result
        for n, cE in enumerate(part4):
            en_values = temp_values.copy()
            en_values[np.where((calc_dE > cE + en_step_half)+(calc_dE < cE - en_step_half))] = 0.0
            the_volume[:,:,:,n] = en_values.reshape((len(part1),len(part2),len(part3)))
        #for n, cE in enumerate(part4):
            #temp_result, temp_angles = reverse_q_vector_INS(points, Ei)
            #temp_values = np.ones((len(temp_result)))
            #temp_values[np.where(temp_angles > cone_angle)] = 0
            #temp_values[np.where(temp_result[:,1] < h_angles[0])] = 0.0
            #temp_values[np.where(temp_result[:,1] > h_angles[-1])] = 0.0
            #for ann in range(len(h_angles)/2 - 1):
                #temp_values[np.where((temp_result[:,1] > h_angles[ann+1])*(temp_result[:,1] < h_angles[ann+2]) )] = 0.0
            #the_volume[:,:,:,n] = temp_values.reshape((len(part1),len(part2),len(part3)))
        # this concludes the calculation; now the slicing and binning can be done
        # these parts may have to be swapped
        if not axes[3]:
            the_volume = the_volume.sum(axis=3)
        if not axes[2]:
            the_volume = the_volume.sum(axis=2)
        if not axes[1]:
            the_volume = the_volume.sum(axis=1)
        if not axes[0]:
            the_volume = the_volume.sum(axis=0)
        # quick consistency check
        if not len(the_volume.shape) == dimensionality:
            return None
        # by now it should all be easy
        maximum = float(the_volume.max())
        the_volume = the_volume / maximum
        the_volume = 100.0 * the_volume
        
        parts = []
        if axes[0]:
            parts.append(part1)
        if axes[1]:
            parts.append(part2)
        if axes[2]:
            parts.append(part3)
        if axes[3]:
            parts.append(part4)
        
        tmarkers = self.markers
        Ei = param["Lambda"]["Ei"]
        lam_i = n_lambda(Ei)
        endmarkers,  markers = [],  []
        for m in tmarkers:
            if (np.abs(m[:3]) < 1e-14).all():
                continue
            else:
                markers.append(m)
        if len(markers) > 0:
            start = np.row_stack(markers)
            points = rotate(rotmat, start[:,:3])
            local_res = []
            k_f = points.copy()
            q_z = n_k(n_lambda(Ei))
            k_f[:,2] += q_z
            # kf has been found, now we need the cone angle
            newones = cart_to_sph(rotate(rotY(-ori+0.3), k_f))
            newones = np.arctan2(np.sqrt(newones[:,0]**2 + newones[:,1]**2), newones[:,2])
            # as easy as this; now we can clip the point using the magnet cone opening
            # for each point we need the energy resolution
            resolution = []
            flux = 0.0
            if self.choppertype == 'Large windows, high flux':
                ch1window, ch6window = 55.5, 9.23
            else:
                ch1window, ch6window = 18.5, 2.6
            if forward:
                detector_distance = 4.44
            else:
                detector_distance = 3.4
            for x in start[:,3]:
                res, dump = INS_resolution(self.fband, Ei, dist = detector_distance, Ld= 0.01, CH1_window = ch1window, CH6_window = ch6window, Ef =  Ei - x)
                resolution.append(res)
            # we have the resolution at each point
            flux = self.redfactor * exed.spectrum_longguide(lam_i)
            # and now we have the flux too
            temp_result, temp_angles = reverse_q_vector_INS(points, Ei)
            calc_dE = Ei - n_energy(n_lambda_fromk(temp_result[:,0]))
            temp_values = np.ones((len(temp_result)))
            temp_values[np.where(newones > cone_angle)] = 0
            temp_values[np.where(temp_result[:,1] < h_angles[0])] = 0.0
            temp_values[np.where(temp_result[:,1] > h_angles[-1])] = 0.0
            for ann in range(int(len(h_angles)/2) - 1):
                temp_values[np.where((temp_result[:,1] > h_angles[2*ann+1])*(temp_result[:,1] < h_angles[2*ann+2]) )] = 0.0
            for num, val in enumerate(resolution):
                if not temp_values[num]:
                    continue
                if abs((start[num, 3]) - calc_dE[num]) > val:
                    continue
                else:
                    endmarkers.append([start[num][np.where(axes)], None,
                                       calc_dE[num], val, markers[num]])
                    print( "output", endmarkers[-1])
        return the_volume, parts, endmarkers, samp.description(), ranges

    def show_INS_coverage(self, forward = True, outFile = ""):
        self.update_intensity()
        pic, ax, mark, desc, ranges = self.coverage_INS(forward, powder = False)
        dims = len(ax)
        if dims == 0:
            plot.apology(outFile = outFile, fig = self.figure)
            return None
        if dims == 4:
            plot.apology(outFile = outFile, fig = self.figure)
            return None
        #picf, ax1f, ax2f, markf, symf, desf, planf = self.new_total_plane(forward = True)
        #picb, ax1b, ax2b, markb, symb, desb, planb = self.new_total_plane(forward = False)
        #if self.choppertype == "Resolution too high":
            #desf = "INTENSITY INCORRECT\nPlease adjust the resolution settings"
        if dims == 1:
            plot.plot1D(pic, ax, mark, desc, ranges, outFile = outFile, fig = self.figure)
        if dims == 2:
            plot.plot2D(pic, ax, mark, desc, ranges, outFile = outFile, fig = self.figure)
        if dims == 3:
            plot.plot3D(pic, ax, mark, desc, ranges, outFile = outFile, fig = self.figure)
        #plot.plot_total_map((picf, picb), (ax1f, ax1b), (ax2f, ax2b), outFile = outFile, fig = self.figure,
                             #markers = (markf, markb), symbol = symf, description = desf, plane = planf)
                             


    def show_INS_total_coverage(self, forward = True, outFile = ""):
        self.update_intensity()
        last_ori = self.param["Field"]["Direction"]
        pics, axes, angles, marks, descs, m_ranges = [],[],[], [], [], []
        for dudley in np.arange(-12.0, 2.4, 2.0):
            self.param["Field"]["Direction"] = dudley
            pic, ax, mark, desc, ranges = self.coverage_INS(forward, powder = False)
            dims = len(ax)
            if dims == 0:
                plot.apology(outFile = outFile, fig = self.figure)
                return None
            if dims == 4:
                plot.apology(outFile = outFile, fig = self.figure)
                return None
            pics.append(pic)
            axes.append(ax)
            angles.append(dudley)
            marks.append(mark)
            descs.append(desc)
            m_ranges.append(ranges)
        #picf, ax1f, ax2f, markf, symf, desf, planf = self.new_total_plane(forward = True)
        #picb, ax1b, ax2b, markb, symb, desb, planb = self.new_total_plane(forward = False)
        #if self.choppertype == "Resolution too high":
            #desf = "INTENSITY INCORRECT\nPlease adjust the resolution settings"
        if dims == 1:
            plot.plot1Dtotal(pics, axes, marks, descs, m_ranges, outFile = outFile, fig = self.figure, angles = angles)
        if dims == 2:
            plot.plot2Dtotal(pics, axes, marks, descs, m_ranges, outFile = outFile, fig = self.figure, angles = angles)
        if dims == 3:
            plot.plot3Dtotal(pics, axes, marks, descs, m_ranges, outFile = outFile, fig = self.figure, angles = angles)
        #plot.plot_total_map((picf, picb), (ax1f, ax1b), (ax2f, ax2b), outFile = outFile, fig = self.figure,
                             #markers = (markf, markb), symbol = symf, description = desf, plane = planf)
        self.param["Field"]["Direction"] = last_ori
