
#    This file is part of InEXEQ.
#
#    InEXEQ is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <https://www.gnu.org/licenses/>.
# 
# Copyright (C) Maciej Bartkowiak, 2014-2018, 2020

__doc__ = """
A packet of utilities for re-calculating between different
properties of neutrons. This is a non-SI approach, as
the wavelengths are not expressed in metres, but in Angstroms.
Maciej Bartkowiak, 28 Mar 2014

A new feature is the calculation of the TOF instrument chopper
frequencies, wavelength band and wavelength resolution.
 Maciej Bartkowiak, 21 Aug 2014
"""

import numpy as np
import math

h = 6.626068 * 10**(-34) # it is Planck's constant given in J*s
meV = 1.602176487 * 10**(-22) # that is 1 meV expressed in Joule
mn = 1.67492729 * 10**(-27) # in kilograms
elc = 1.602176487 * 10**(-19) # e, the elementary charge in Coulombs
hbar = h / (2 * np.pi)

dL = 0.01 # the precision of our distance from chopper to the detector
          # set to 1 cm here, as we will have done calibration

def n_lambda(nenergy):
    """
    Converts energy in meV into wavelength in Angstrom.
    """
    wavel = h / np.sqrt(2*mn*nenergy*meV)
    return wavel * 10**10

def n_lambda_meters(nenergy):
    """
    Converts energy in meV into wavelength in Angstrom.
    """
    wavel = h / np.sqrt(2*mn*nenergy*meV)
    return wavel
    
def n_energy(nlambda):
    """
    Converts lambda in Angstrom into energy in meV.
    """
    e = h*h / (2*mn*nlambda*nlambda*10**(-20))
    return e / meV

def n_k(nlambda):
    """
    Gives the absolute value of quasimomentum that corresponds to the input wavelength.
    """
    # return 2*np.pi / nlambda
    return 1.0 / nlambda

def n_v(nlambda):
    """
    Returns the velocity [m/s] of a neutron with an input wavelength nlambda [A].
    """
    v = h / mn / (nlambda*10**-10)
    return v

def n_lambda_from_v(v):
    """
    Returns the wavelength nlambda [A] of a neutron with an input velocity [m/s].
    """
    nlambda = h / (v * mn) * 10**10
    return nlambda

def n_lambda_fromk(nk):
    """
    Gives the absolute value of quasimomentum that corresponds to the input wavelength.
    """
    # return 2*np.pi / nlambda
    return 1.0 / nk

def n_energy_from_TOF(TOF, L = 4.5):
    v = L/TOF
    return n_energy(n_lambda_from_v(v))

def bandwidth(f, L = 56.1961):
    return h/(mn*f*L) * 1e10
    
def lastchopper(band):
    L = 56.1961
    wave = band * 1e-10
    return h/(mn*wave*L)
    
def delta_lambda(f, window, wave, L = 56.1961):
    part1 = dL/L
    dt = (1.0/f) * (window/360.0)
    # print "dt, ", dt
    TOF = L / n_v(wave)
    # print "TOF, ", TOF
    part2 = dt/TOF
    result = np.sqrt(part1**2 + part2**2)
    return result

def firstchopper(res, wave):
    wind1 = 6.86
    wind2 = 18.5
    L1 = 56.1961
    L2 = 56.0451
    TOF1, TOF2 = L1 / n_v(wave), L2 / n_v(wave)
    # temp1, temp2 = res**2 - (dL/L1)**2, res**2 - (dL/L2)**2
    temp1, temp2 = res**2, res**2
    if temp1 > 0.0:
        # dtt1 = np.sqrt(res**2 - (dL/L1)**2)
        dtt1 = np.sqrt(res**2)
    else:
        dtt1 = 1e-10
    if temp2 > 0.0:
        # dtt2 = np.sqrt(res**2 - (dL/L2)**2)
        dtt2 = np.sqrt(res**2)
    else:
        dtt2 = 1e-10
    dt1, dt2 = dtt1 * TOF1, dtt2 * TOF2
    f1 = (1.0/dt1) * (wind1/(2*360.0)) # the factor of 2 is added to account for
    f2 = (1.0/dt2) * (wind2/(2*360.0)) # the FWHM of the peak, not the base width
    # print "Fermi at ", f1
    # print "CH1AB at ", f2
    if f2 <= 201.0:
        return "CH1", f2
    elif f1 <= 610.0:
        return "Fermi", f1
    else:
        return "Resolution too high", -1
    
def intensity_drop(fpulse, fband, choppertype = ""):
    if fpulse > 600:
        fpulse = 600
    if fpulse < 0:
        fpulse = 1200
    if choppertype == "Resolution too high":
        wind = 0.01
    elif choppertype == "CH1":
        wind = 18.5
    else:
        wind = 6.86
    int_per_sec = (min(fband,fpulse)/fpulse) * (wind/(2*360.0))
    # int_per_sec = (1.0/fpulse) * (wind/(2*360.0))
    return int_per_sec
    
def intensity_drop_INS(fpulse, fband, choppertype = "", CH1_window = 18.5, CH6_window = 2.6):
    if choppertype == 'Large windows, high flux':
        red1 = 55.5/720.0
        red2 = 9.23/720.0
    else:
        red1 = 18.5/720.0
        red2 = 2.6/720.0
    if max(fpulse, fband) > 7200.0:
        extrared = 0.5
    else:
        extrared = 1.0
    int_per_sec = red1*red2*extrared
    # int_per_sec = (1.0/fpulse) * (wind/(2*360.0))
    return int_per_sec

def INS_chopper_speed_old(Ei, dEi = 0.1, dist = 4.5, Ld = 0.01, CH1_window = 18.5, CH6_window = 2.6):
    """
    Calculates the chopper speed required to produce the requested resolution.
    This formula has to be checked; did not work in the beginning
    """
    Lsd = dist
    Lms = 2.44
    Lpm = -2.44 + 53.063
    
    Ef = Ei
    lam_i = n_lambda_meters(Ei)
    lam_f = n_lambda_meters(Ef)
    Ax = CH6_window/180.0 *(Lpm + Lms + Lsd*(lam_f/lam_i)**3)
    Bx = CH1_window/180.0 *(Lms + Lsd*(lam_f/lam_i)**3)
    C = mn/h*Lpm*Ld*lam_f
    prefactor = h**3 / mn**2 / lam_f**3 / Lsd / Lpm
    nu = math.sqrt((Ax**2 + Bx**2) / (dEi**2/(prefactor/meV)**2 - C**2))
    return nu * 60 # / mn

def INS_resolution_old(rpm, Ei, dist = 4.5, Ld = 0.01, CH1_window = 18.5, CH6_window = 2.6, Ef = None):
    """
    Calculates the chopper speed required to produce the requested resolution.
    This formula has to be checked; did not work in the beginning
    """
    if Ef == None:
        Ef = Ei
    Lsd = dist
    nu = rpm/60.0
    Lms = 2.44
    Lpm = -2.44 + 53.063
    tm = CH6_window/180.0 / nu
    tp = CH1_window/180.0 / nu
    lam_i = n_lambda_meters(Ei)
    lam_f = n_lambda_meters(Ef)
    A = tm*(Lpm + Lms + Lsd*(lam_f/lam_i)**3)
    B = tp*(Lms + Lsd*(lam_f/lam_i)**3)
    C = mn/h*Lpm*Ld*lam_f
    prefactor = h**3 / mn**2 / lam_f**3 / Lsd / Lpm
    dEi = prefactor*math.sqrt(A**2 + B**2 + C**2)/meV
    # return round(dEi,5), round(Ei - n_energy_from_TOF(60.0/rpm, dist)- dEi,5)
    return round(dEi,5), round(Ei - n_energy_from_TOF(60.0/rpm, dist),5)

def INS_resolution(rpm, Ei, dist = 4.5, Ld = 0.01, CH1_window = 18.5, CH6_window = 2.6, Ef = None):
    """
    Calculates the chopper speed required to produce the requested resolution.
    This formula has to be checked; did not work in the beginning
    """
    if Ef == None:
        Ef = Ei
    Lsd = dist
    nu = rpm/60.0
    Lms = 2.44
    Lpm = -2.44 + 53.063
    L1 = Lpm
    L2 = Lms
    L3 = Lsd
    lam_i = n_lambda(Ei)
    lam_f = n_lambda(Ef)
    v_i = n_v(lam_i)
    v_f = n_v(lam_f)
    tm = 1e6 * CH6_window/720.0 / nu
    tp = 1e6 * CH1_window/720.0 / nu
    tu = 1e6 * Ld / v_i
    A = L1*Ld*lam_f*252.77
    B = tp*(L2+L3*((lam_f/lam_i)**3))
    C = tm*(L1+L2+L3*((lam_f/lam_i)**3))
    dEi = 0.6472 * math.sqrt(A**2+B**2+C**2) / (lam_f**3 * L1 * L3)
    #print A
    #print B
    #print C
    #print lam_i
    #print lam_f
    #print mn
    #print tm
    #print tp
    #print dEi
    # return round(dEi,5), round(Ei - n_energy_from_TOF(60.0/rpm, dist)- dEi,5)
    return round(dEi,5), round(Ei - n_energy_from_TOF(60.0/rpm, dist),5)

def INS_chopper_speed(Ei, dEi = 0.1, dist = 4.5, Ld = 0.01, CH1_window = 18.5, CH6_window = 2.6):
    """
    Calculates the chopper speed required to produce the requested resolution.
    This formula has to be checked; did not work in the beginning
    """
    Lsd = dist
    Lms = 2.44
    Lpm = -2.44 + 53.063
    Ef = Ei
    L1 = Lpm
    L2 = Lms
    L3 = Lsd
    lam_i = n_lambda(Ei)
    lam_f = n_lambda(Ef)
    ABC = ((lam_f**3 * L1 * L3) * dEi / 0.6472)**2 # this is A^2 + B^2 + C^2 from INS_resolution
    BC = ABC - (L1*Ld*lam_f*252.77)**2
    nusq = 1.0/BC * ((CH1_window/720.0)**2 * (L2 + L3*(lam_f/lam_i)**3)**2 +
                     (CH6_window/720.0)**2 * (L1 + L2 + L3*(lam_f/lam_i)**3)**2)
    nu = math.sqrt(nusq) * 1e6
    #print "Chopper frequency = ", nu
    return nu * 60 # / mn

    
    

    
