#!/usr/bin/python

#    This file is part of EXEQ.
#
#    EXEQ is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <https://www.gnu.org/licenses/>.
# 
# Copyright (C) Maciej Bartkowiak, 2014-2018, 2020

__doc__ = """
A temporary interface to the EXEQ program.
This is the original, monolithic interface,
where one large window covers everything.
Maciej Bartkowiak, 28 Mar 2014
"""

mpl_scale = 1.0
import sys
import copy
major_version = '1'
minor_version = '0'
from geometry import *
from units import *
# import plot_mp as plot
from persistence import Kernel 
#
from PyQt5 import QtCore, QtWidgets
from PyQt5.QtCore import QObject, pyqtSlot
from VariablesGUI import VarBox,  RadioBox
#
import matplotlib
matplotlib.use("Qt5Agg")
matplotlib.rcParams['legend.fontsize'] = int(12*mpl_scale)
matplotlib.rcParams['legend.borderpad'] = 0.3*mpl_scale
matplotlib.rcParams['legend.labelspacing'] = 0.2*mpl_scale
matplotlib.rcParams['xtick.labelsize'] = int(14*mpl_scale)
matplotlib.rcParams['ytick.labelsize'] = int(14*mpl_scale)
matplotlib.rcParams['axes.titlesize'] = int(14*mpl_scale)
matplotlib.rcParams['axes.labelsize'] = int(14*mpl_scale)
matplotlib.rcParams['axes.grid'] = True
matplotlib.rcParams['font.size'] = int(12*mpl_scale)
import matplotlib.pyplot as mpl
from matplotlib.backends import qt_compat
from matplotlib.backends.backend_qt5agg import FigureCanvasQTAgg
from matplotlib.backends.backend_qt5agg import NavigationToolbar2QT as NavigationToolbar2QTAgg

###
### the parameter tree
###

params = {}
params["Choppers"] = {}
params["Sample"] = {}
params["Field"] = {}
params["Lambda"] = {}
params["hkl"] = {}

params["Field"]["Direction"] = 0.0
params["Choppers"]["Frequency"] = 120
params["Choppers"]["Suppression"] = 1
params["Lambda"]["Res"] = 0.01
params["Lambda"]["Min"] = 0.7
params["Lambda"]["Max"] = 7.9
params["Sample"]["Constants"] = np.array([5.0, 5.0, 5.0])
params["Sample"]["Angles"] = np.array(3*[90.0])
params["Sample"]["u"] = np.array([1.0, 0.0, 0.0])
params["Sample"]["v"] = np.array([0.0, 0.0, 1.0])
params["Sample"]["phi"] = 0
params["Sample"]["chi"] = 0
params["Sample"]["omega"] = 0
params["hkl"]["h"] = 'h'
params["hkl"]["k"] = '0'
params["hkl"]["l"] = 'l'

normals = [["Sample", "u"], ["Sample", "v"]]

sample_vars = [
{'Name': 'Lattice constants',  'Unit':'AA',  'Value':4.0*np.ones(3), 'Key' : 'abc', 
                                      'MinValue':0.01*np.ones(3),
                                      'MaxValue':1e5*np.ones(3),
                                      'Length': 3,  'Type':'float', 
                                      'Comment' : "Lattice parameters a, b, c, given in Angstroms."}, 
{'Name': 'Lattice angles',  'Unit':'deg.',  'Value':90*np.ones(3), 'Key' : 'angs', 
                                      'MinValue':1.0*np.ones(3),
                                      'MaxValue':180.0*np.ones(3),
                                      'Length': 3,  'Type':'float', 
                                      'Comment' : "Unit cell angles alpha, beta, gamma, given in degrees."}, 
{'Name': 'Orientation vector Bu',  'Unit':'r.l.u.',  'Value':np.array([0, 0, 1]), 'Key' : 'Bu', 
                                      'MinValue':-1e4*np.ones(3),
                                      'MaxValue':1e4*np.ones(3),
                                      'Length': 3,  'Type':'float', 
                                      'Comment' : "Bu is the sample vector along the magnetic field direction B. Together with Bv it defines the sample orientation in respect to the magnet axis."
                                      }, 
{'Name': 'Orientation vector Bv',  'Unit':'r.l.u.',  'Value':np.array([1, 0, 0]), 'Key' : 'Bv', 
                                      'MinValue':-1e4*np.ones(3),
                                      'MaxValue':1e4*np.ones(3),
                                      'Length': 3,  'Type':'float', 
                                      'Comment' :"Bv is the sample vector along the in-plane direction perpendicular to B. Together with Bu it defines the sample orientation in respect to the magnet axis."}, 
{'Name': 'Sample rotation around "Bu"',  'Unit':'deg.',  'Value':np.array([0.0]), 'Key' : 'phi', 
                                      'MinValue':-360*np.ones(1),
                                      'MaxValue':360*np.ones(1),
                                      'Length': 1,  'Type':'float', 
                                      'Comment' : 'Sample rotation around the Bu vector. Something one has to set by hand when gluing the sample to the mount.'}, 
{'Name': 'Sample rotation around "Bv"',  'Unit':'deg.',  'Value':np.array([0.0]), 'Key' : 'chi', 
                                      'MinValue':-360*np.ones(1),
                                      'MaxValue':360*np.ones(1),
                                      'Length': 1,  'Type':'float', 
                                      'Comment' : 'Sample rotation around the Bv vector. Something one has to set by hand when gluing the sample to the mount.'}, 
{'Name': 'Sample rotation around "Bu x Bv"',  'Unit':'deg.',  'Value':np.array([0.0]), 'Key' : 'omega', 
                                      'MinValue':-360*np.ones(1),
                                      'MaxValue':360*np.ones(1),
                                      'Length': 1,  'Type':'float', 
                                      'Comment' : 'Sample rotation around the vector perpendicular to Bu and Bv (typically the vertical axis). Something one has to set by hand when gluing the sample to the mount.'}, 
{'Name': 'Magnet rotation',  'Unit':'deg.',  'Value':np.array([0.0]), 'Key' : 'magrot', 
                                      'MinValue':-360*np.ones(1),
                                      'MaxValue':360*np.ones(1),
                                      'Length': 1,  'Type':'float', 
                                      'Comment' : 'Rotation of the magnet around the vertical axis. Moves the sample AND the forward detectors.'}, 
]

wave_vars = [
{'Name': 'Wavelength resolution',  'Unit':'N/A',  'Value':np.array([0.0076]), 'Key' : 'res', 
                                      'MinValue':1e-9*np.ones(1),
                                      'MaxValue':1*np.ones(1),
                                      'Length': 1,  'Type':'float', 
                                      'Comment' : 'OPTIONAL: The desired wavelength resolution (d_lambda/lambda). Determines the frequency of the first chopper.'}, 
{'Name': 'Wavelength range minimum',  'Unit':'AA',  'Value':np.array([0.7]), 'Key' : 'lammin', 
                                      'MinValue':0.7*np.ones(1),
                                      'MaxValue':15.0*np.ones(1),
                                      'Length': 1,  'Type':'float', 
                                      'Comment' : 'The beginning of the wavelength spectrum.'}, 
{'Name': 'Wavelength range maximum',  'Unit':'AA',  'Value':np.array([7.2]), 'Key' : 'lammax', 
                                      'MinValue':0.7*np.ones(1),
                                      'MaxValue':25.0*np.ones(1),
                                      'Length': 1,  'Type':'float', 
                                      'Comment' : 'The end of the wavelength spectrum.'}, 
]

chop_vars = [
{'Name': 'CH1 Frequency',  'Unit':'rpm',  'Value':np.array([3600]), 'Key' : 'ch1freq', 
                                      'MinValue':300*np.ones(1),
                                      'MaxValue':12000*np.ones(1),
                                      'Length': 1,  'Type':'float', 
                                      'Comment' : 'Freqeuncy of the first chopper. Determines the resolution.'}, 
{'Name': 'CH5 Frequency',  'Unit':'rpm',  'Value':np.array([600]), 'Key' : 'ch5freq', 
                                      'MinValue':300*np.ones(1),
                                      'MaxValue':7200*np.ones(1),
                                      'Length': 1,  'Type':'float', 
                                      'Comment' : 'Frequency of the last chopper. Determines the wavelength band.'},
]

aaa = "These fields define the indices of a reciprocal space plane used for creating the detector range maps. It is allowed to input a letter (h), a number (-0.2) or a range (-0.33:1.28). "
hkl_vars = [
{'Name': 'H',  'Unit':'r.l.u.',  'Value':['h'], 'Key' : 'rec_h', 
                                      'MinValue':'',
                                      'MaxValue':'',
                                      'Length': 1,  'Type':'str', 
                                      'Comment' : aaa}, 
{'Name': 'K',  'Unit':'r.l.u.',  'Value':['0'], 'Key' : 'rec_k', 
                                      'MinValue':'',
                                      'MaxValue':'',
                                      'Length': 1,  'Type':'str', 
                                      'Comment' : aaa}, 
{'Name': 'L',  'Unit':'r.l.u.',  'Value':['l'], 'Key' : 'rec_l', 
                                      'MinValue':'',
                                      'MaxValue':'',
                                      'Length': 1,  'Type':'str', 
                                      'Comment' : aaa}, 
{'Name': 'Grid Precision',  'Unit':'scaling',  'Value':1.0, 'Key' : 'prec', 
                                      'MinValue':0.25,
                                      'MaxValue':3.0,
                                      'Length': 1,  'Type':'float', 
                                      'Comment' : 'Scales the number of points on the grid, for smoother plots.'}, 
]
        
marker_vars = [
{'Name': 'Peak 1',  'Unit':'r.l.u.',  'Value':np.array([0, 0, 0]), 'Key' : 'poi1', 
                                      'MinValue':-1e4*np.ones(3),
                                      'MaxValue':1e4*np.ones(3),
                                      'Length': 3,  'Type':'float', 
                                      'Comment' :"A point in reciprocal space to be marked in the plots."}, 
{'Name': 'Peak 2',  'Unit':'r.l.u.',  'Value':np.array([0, 0, 0]), 'Key' : 'poi2', 
                                      'MinValue':-1e4*np.ones(3),
                                      'MaxValue':1e4*np.ones(3),
                                      'Length': 3,  'Type':'float', 
                                      'Comment' :"A point in reciprocal space to be marked in the plots."}, 
{'Name': 'Peak 3',  'Unit':'r.l.u.',  'Value':np.array([0, 0, 0]), 'Key' : 'poi3', 
                                      'MinValue':-1e4*np.ones(3),
                                      'MaxValue':1e4*np.ones(3),
                                      'Length': 3,  'Type':'float', 
                                      'Comment' :"A point in reciprocal space to be marked in the plots."}, 
{'Name': 'Peak 4',  'Unit':'r.l.u.',  'Value':np.array([0, 0, 0]), 'Key' : 'poi4', 
                                      'MinValue':-1e4*np.ones(3),
                                      'MaxValue':1e4*np.ones(3),
                                      'Length': 3,  'Type':'float', 
                                      'Comment' :"A point in reciprocal space to be marked in the plots."}, 
{'Name': 'Peak 5',  'Unit':'r.l.u.',  'Value':np.array([0, 0, 0]), 'Key' : 'poi5', 
                                      'MinValue':-1e4*np.ones(3),
                                      'MaxValue':1e4*np.ones(3),
                                      'Length': 3,  'Type':'float', 
                                      'Comment' :"A point in reciprocal space to be marked in the plots."}, 
]

def old_dict_keys(newkey):
    key1,  key2 = "",""
    if 'abc' in newkey:
        key1, key2 = "Sample",  "Constants"
    elif 'angs' in newkey:
        key1, key2 = "Sample",  "Angles"
    elif 'Bu' in newkey:
        key1, key2 = "Sample",  "u"
    elif 'Bv' in newkey:
        key1, key2 = "Sample",  "v"
    elif 'phi' in newkey:
        key1, key2 = "Sample",  "phi"
    elif 'chi' in newkey:
        key1, key2 = "Sample",  "chi"
    elif 'omega' in newkey:
        key1, key2 = "Sample",  "omega"
    elif 'magrot' in newkey:
        key1, key2 = "Field",  "Direction"
    elif 'res' in newkey:
        key1, key2 = "Lambda",  "Res"
    elif 'lammin' in newkey:
        key1, key2 = "Lambda",  "Min"
    elif 'lammax' in newkey:
        key1, key2 = "Lambda",  "Max"
    elif 'rec_h' in newkey:
        key1, key2 = "hkl",  "h"
    elif 'rec_k' in newkey:
        key1, key2 = "hkl",  "k"
    elif 'rec_l' in newkey:
        key1, key2 = "hkl",  "l"
    elif 'poi' in newkey:
        key1, key2 = "Marker",  "Any"
    elif 'prec' in newkey:
        key1, key2 = "Precision",  "Any"
    return key1,  key2

def new_dict_keys(key1,  key2):
    newkey = ""
    if  (key1, key2) == ("Sample",  "Constants"):
        newkey = 'abc'
    elif (key1, key2) == ("Sample",  "Angles"):
        newkey = 'angs'
    elif (key1,key2) == ( "Sample",  "u"):
        newkey = 'Bu'
    elif (key1,key2) == ( "Sample",  "v"):
        newkey = 'Bv'
    elif (key1,key2) == ( "Sample",  "phi"):
        newkey = 'phi'
    elif (key1,key2) == ( "Sample",  "chi"):
        newkey = 'chi'
    elif (key1,key2) == ( "Sample",  "omega"):
        newkey = 'omega'
    elif (key1,key2) == ( "Field",  "Direction"):
        newkey = 'magrot'
    elif (key1,key2) == ( "Lambda",  "Res"):
        newkey = 'res'
    elif (key1,key2) == ( "Lambda",  "Min"):
        newkey = 'lammin'
    elif (key1,key2) == ( "Lambda",  "Max"):
        newkey = 'lammax'
    elif (key1,key2) == ( "hkl",  "h"):
        newkey = 'rec_h'
    elif (key1,key2) == ( "hkl",  "k"):
        newkey = 'rec_k'
    elif (key1,key2) == ( "hkl",  "l"):
        newkey = 'rec_l'
    return newkey

###
### GUI-independent drawing functions
###

    
####
#### The GUI
####

# Font = ("Arial", 14)

class InterfaceEmbedded(QObject):
    def __init__(self, master, size, param):
        super().__init__(master)
        self.master = master
        # self.base = Notebook(master)
        self.base = QtWidgets.QWidget(master)
        self.base_layout = QtWidgets.QVBoxLayout(self.base)
        #
        self.guide = "Long"
        self.cryostat = "3He"
        #
        toppart = QtWidgets.QWidget(self.base)
        hbox = QtWidgets.QHBoxLayout(toppart)
        pane1 = QtWidgets.QWidget(toppart)
        vbox1 = QtWidgets.QVBoxLayout(pane1)
        pane2 = QtWidgets.QWidget(toppart)
        vbox2 = QtWidgets.QVBoxLayout(pane2)
        pane3 = QtWidgets.QWidget(toppart)
        vbox3 = QtWidgets.QVBoxLayout(pane3)
        for p in [pane1,  pane2, pane3]:
            hbox.addWidget(p)
        self.base_layout.addWidget(toppart)
        self.sample_vbox = VarBox(toppart, sample_vars, 'Sample definition')
        self.wave_vbox = VarBox(toppart, wave_vars, 'Wavelength spectrum')
        self.chop_vbox = VarBox(toppart, chop_vars, 'Chopper settings')
        self.hkl_vbox = VarBox(toppart, hkl_vars, 'Plotting range')
        self.marker_vbox = VarBox(toppart, marker_vars, 'Marked points')
        # buttons follow here        pow_button = self.MakeButton(base, 'Powder coverage', self.show_powder_inelastic, 'Shows the maximum E,q range of the instrument. Relevant to powder samples.')
        load_button = self.MakeButton(self.base, 'Load settings', self.load_profile,
            "Load the contents of all the input fields from a YAML text file. Intended mainly for users who do not enjoy typing in all the numbers every time they run EXEQ.")
        save_button = self.MakeButton(self.base, 'Save settings', self.save_profile, 
            "Write out all the input fields into a text file using YAML. Those parameters can be later read again from the same file.")
        orient_button = self.MakeButton(self.base, 'Show orientation', self.plot_orient_mag,
            "Updates the intstrument configuration to include the new sample orientation in respect to the magnetic field, and the new magnet orientation.")
        panel_button = self.MakeButton(self.base, 'Detector panels', self.plot_detector_panels,
            'Plots the layout of the detector tubes, and the position of the visible peaks on the detector.')
        plane2_button = self.MakeButton(self.base, 'Reciprocal space coverage', self.show_plane_bw,
            'The most important output option: map of the reciprocal space. Uses HKL limits and marker points.')
        tot_button = self.MakeButton(self.base, 'Total coverage', self.total_plane_bw,
            'Plots the reciprocal space coverage for the full range of the magnet rotation angles.')
        help_button = self.MakeButton(self.base, 'HELP!', self.show_help,
            'Prints some basic usage tips on the screen.')
        # need two radiogroups as well
        rb1 = RadioBox(self.base, 0, ["Neutron Guide", "SANS collimation"], 
                                title = "Optics")
#        rb2 = RadioBox(self.base, 0, ["3He/Dilution", "4He"], 
#                                title = "Cryostat")
        rb1.values_changed.connect(self.update_guide)
       # mdpi, winch, hinch = 75, 5.0, 6.0
        vbox1.addWidget(self.wave_vbox.base)
        vbox1.addWidget(self.chop_vbox.base)
        vbox1.addWidget(rb1.base)
        # vbox1.addWidget(rb2.base)
        vbox1.addWidget(load_button)
        vbox1.addWidget(save_button)
        vbox2.addWidget(self.sample_vbox.base)
        vbox2.addWidget(orient_button)
        vbox2.addWidget(panel_button)
        vbox3.addWidget(self.hkl_vbox.base)
        vbox3.addWidget(self.marker_vbox.base)
        vbox3.addWidget(plane2_button)
        vbox3.addWidget(tot_button)
        vbox3.addWidget(help_button)
        self.canvas,  self.figure,  self.flayout = self.MakeCanvas(self.base)
        self.base_layout.addWidget(self.canvas)
        self.kernel = Kernel(self.figure, precision = 1.3)
        self.oldstyle_params = copy.deepcopy(params)
        # self.switch_lambda()
        # self.orient_mag()
        # self.chopper_freq()
        for box in [self.chop_vbox,  self.sample_vbox,  self.hkl_vbox,  self.wave_vbox,  self.marker_vbox]:
            box.values_changed.connect(self.get_new_parameters)
        self.wave_vbox.values_changed.connect(self.chopper_freq)
        self.scar = QtWidgets.QScrollArea(master)
        self.scar.setWidget(self.base)
        self.master.setCentralWidget(self.scar)
        self.kernel.show_welcome_message()

    def MakeCanvas(self, parent):
        mdpi, winch, hinch = 75, 7.0, 6.0
        canvas = QtWidgets.QWidget(parent)
        layout = QtWidgets.QVBoxLayout(canvas)
        figure = mpl.figure(figsize = [2*winch, hinch], dpi=mdpi )#, frameon = False)
        figAgg = FigureCanvasQTAgg(figure)
        figAgg.setParent(canvas)
        figAgg.setSizePolicy(QtWidgets.QSizePolicy.Expanding,QtWidgets.QSizePolicy.Expanding)
        figAgg.updateGeometry()
        toolbar = NavigationToolbar2QTAgg(figAgg, canvas)
        # toolbar = NavigationToolbar2QTAgg(figAgg)
        toolbar.update()
        layout.addWidget(figAgg)
        layout.addWidget(toolbar)
        return canvas, figure, layout
        
    def MakeButton(self, parent, text, function, tooltip = ""):
        button = QtWidgets.QPushButton(text, parent)
        if tooltip:
            button.setToolTip(tooltip)
        button.clicked.connect(function)
        button.setSizePolicy(QtWidgets.QSizePolicy.Preferred,QtWidgets.QSizePolicy.Preferred)
        button.setMinimumSize(QtCore.QSize(80, 40))
        return button
        
    def trigger_get_new_parameters(self, event):
        self.get_new_parameters()
        self.orient_mag()
        self.chopper_freq()
    
    @pyqtSlot()
    def get_new_parameters(self):
        mark_points = []
        for box in [self.chop_vbox,  self.sample_vbox,  self.hkl_vbox,  self.wave_vbox,  self.marker_vbox]:
            varkeys,  vardict = box.returnValues()
            for kk in varkeys:
                val = str(vardict[kk]).strip("'{}[]()")
                oldkey1,  oldkey2 = old_dict_keys(kk)
                if len(oldkey1) < 1 or len(oldkey2) < 1:
                    continue
                if oldkey1 == 'Marker':
                    mark_points.append(val)
                elif oldkey1 == 'Precision':
                    self.kernel.set_precision(val)
                else:
                    self.oldstyle_params[oldkey1][oldkey2] = val
        self.kernel.take_parameters(self.oldstyle_params)
        new = self.kernel.marker_points(mark_points)
        # here the kernel can parse the variables
        newkeys,  newdict = [], {}
        param, mark_points, prec = self.kernel.current_parameters_str("temp.txt")
        for kk1 in param.keys():
            for kk2 in param[kk1].keys():
                nkey = new_dict_keys(kk1, kk2)
                if len(nkey) > 0:
                    newkeys.append(nkey)
                    if 'hkl' in kk1:
                        val = [param[kk1][kk2]]
                    else:
                        try:
                            val = np.array([float(x) for x in param[kk1][kk2].split()])
                        except:
                            val = param[kk1][kk2]
                    newdict[nkey] = val
        for mnum,  mark in enumerate(mark_points):
            nkey = 'poi' + str(mnum+1)
            newkeys.append(nkey)
            try:
                val = np.array([float(x) for x in mark.split()])
            except:
                val = mark
            newdict[nkey] = val
        for box in [self.chop_vbox,  self.sample_vbox,  self.hkl_vbox,  self.wave_vbox,  self.marker_vbox]:
            box.takeValues(newkeys,  newdict)
        return True

    def switch_lambda(self):
        self.lambda_minmax = not self.lambda_minmax
        param = self.kernel.current_parameters()
        if self.lambda_minmax:
            lmin, lmax = (param["Lambda"]["Min"],
                          param["Lambda"]["Max"])
            for i in self.lambda_band_fields:
                var = self.fields[i][0].get()
                self.fields[i][0].delete(0, len(var))
                self.fields[i][0].insert(0, "NOT USED")
                self.fields[i][0].configure(state = DISABLED)
            i = self.lambda_minmax_fields[0]
            self.fields[i][0].configure(state = NORMAL)
            var = self.fields[i][0].get()
            self.fields[i][0].delete(0, len(var))
            self.fields[i][0].insert(0, str(lmin))
            i = self.lambda_minmax_fields[1]
            self.fields[i][0].configure(state = NORMAL)
            var = self.fields[i][0].get()
            self.fields[i][0].delete(0, len(var))
            self.fields[i][0].insert(0, str(lmax))
        else:
            lmin, lmax = (param["Lambda"]["Min"],
                          param["Lambda"]["Max"])
            lmean = (lmin + lmax) / 2.0
            lwidth = lmax - lmin
            for i in self.lambda_minmax_fields:
                var = self.fields[i][0].get()
                self.fields[i][0].delete(0, len(var))
                self.fields[i][0].insert(0, "NOT USED")
                self.fields[i][0].configure(state = DISABLED)
            i = self.lambda_band_fields[0]
            self.fields[i][0].configure(state = NORMAL)
            var = self.fields[i][0].get()
            self.fields[i][0].delete(0, len(var))
            self.fields[i][0].insert(0, str(lmean))
            i = self.lambda_band_fields[1]  
            self.fields[i][0].configure(state = NORMAL) 
            var = self.fields[i][0].get()
            self.fields[i][0].delete(0, len(var))
            self.fields[i][0].insert(0, str(lwidth))     
        # return show_rspace_point(self.parameters, self.exed, "")
        self.get_new_parameters()
        self.orient_mag()
        self.chopper_freq()
        return None
    
    @pyqtSlot()
    def orient_mag(self):
        self.get_new_parameters()
        a = self.kernel.uv_range()
#        newumax, newumin, newvmax, newvmin = a[0], a[1], a[2], a[3]
#        for i in self.outputs:
#            temp = i[1]
#            name = i[0].cget('text')
#            temp.configure(state = NORMAL)
#            var = temp.get()
#            temp.delete(0, len(var))
#            if ("min" in name) and ('u' in name):
#                newvector = newumin
#            if ("max" in name) and ('u' in name):
#                newvector = newumax
#            if ("min" in name) and ('v' in name):
#                newvector = newvmin
#            if ("max" in name) and ('v' in name):
#                newvector = newvmax
#            temp.insert(0, newvector)
#            temp.configure(state = DISABLED)
        return None
        
    def plot_orient_mag(self):
        self.get_new_parameters()
        self.orient_mag()
        self.chopper_freq()
        parameters = self.kernel.current_parameters()
        self.kernel.show_instrument("")
        # self.kernel.show_uv_range()
        return None
        
    def plot_detector_panels(self):
        self.orient_mag()
        self.chopper_freq()
        return self.kernel.show_detector_panels("")
        
    def chopper_freq_precise(self):
        self.get_new_parameters()
        fr = self.kernel.chopper_freqs()
        for i, j in enumerate(self.choutputs):
            j[1].configure(state = NORMAL)
            var = j[1].get()
            j[1].delete(0, len(var))
            j[1].insert(0, fr[i])
            j[1].configure(state = DISABLED)
        param = self.kernel.current_parameters()
        lmin, lmax = (param["Lambda"]["Min"],
                      param["Lambda"]["Max"])
        lmean = (lmin + lmax)/2.0
        lband = bandwidth(float(fr[2]))
        lmin, lmax = lmean - lband/2.0, lmean + lband/2.0
        if self.lambda_minmax:
            for i in self.lambda_band_fields:
                var = self.fields[i][0].get()
                self.fields[i][0].delete(0, len(var))
                self.fields[i][0].insert(0, "NOT USED")
                self.fields[i][0].configure(state = DISABLED)
            i = self.lambda_minmax_fields[0]
            self.fields[i][0].configure(state = NORMAL)
            var = self.fields[i][0].get()
            self.fields[i][0].delete(0, len(var))
            self.fields[i][0].insert(0, str(lmin))
            i = self.lambda_minmax_fields[1]
            self.fields[i][0].configure(state = NORMAL)
            var = self.fields[i][0].get()
            self.fields[i][0].delete(0, len(var))
            self.fields[i][0].insert(0, str(lmax))
        else:
            for i in self.lambda_minmax_fields:
                var = self.fields[i][0].get()
                self.fields[i][0].delete(0, len(var))
                self.fields[i][0].insert(0, "NOT USED")
                self.fields[i][0].configure(state = DISABLED)
            i = self.lambda_band_fields[0]
            self.fields[i][0].configure(state = NORMAL)
            var = self.fields[i][0].get()
            self.fields[i][0].delete(0, len(var))
            self.fields[i][0].insert(0, str(lmean))
            i = self.lambda_band_fields[1]  
            self.fields[i][0].configure(state = NORMAL) 
            var = self.fields[i][0].get()
            self.fields[i][0].delete(0, len(var))
            self.fields[i][0].insert(0, str(lband)) 
        if fr[1]:
            if fr[0] == "Fermi":
                res = delta_lambda(float(fr[1]), window = 6.86, wave = lmean, L = 56.1961)
            else:
                res = delta_lambda(float(fr[1]), window = 18.5, wave = lmean, L = 56.0451)
            for i in self.fields:
                if (i[1] == "Lambda") and (i[2] == "Res"):
                    var = i[0].get()
                    i[0].delete(0, len(var))
                    i[0].insert(0, str(res))
        self.get_new_parameters()
        return None
    
    def EXED_configuration_dialog(self):
        popup = Toplevel(self.master)
        extraframe = Frame(popup)
        self.guidetype = StringVar()
        self.cryostat = StringVar()
        self.forward_mode = StringVar()
        self.backward_mode = StringVar()
        self.panel1 = IntVar()
        self.panel2 = IntVar()
        self.panel3 = IntVar()
        self.panel4 = IntVar()
        self.panel5 = IntVar()
        self.panel7 = IntVar()
        w = Label(extraframe, text="Guide:")
        temp1 = Radiobutton(extraframe, text="Long", variable=self.guidetype, value="Long",
                    command = self.update_configuration)
        temp1.grid(row=0, column =1)
        tip.ToolTip(temp1, "Focussing guide; increases the neutron flux and beam divergence, limits magnet rotation to +/- 5 degrees if the cryostat is on the side of the guide.")
        temp2 = Radiobutton(extraframe, text="Short", variable=self.guidetype, value="Short",
                    command = self.update_configuration)
        temp2.grid(row=0, column =2)
        tip.ToolTip(temp2, "Short guide; the standard option in EXEQ, limits magnet rotation to +/- 10 degrees if the cryostat is on the side of the guide.")
        temp3 = Radiobutton(extraframe, text="SANS collimation", variable=self.guidetype, value="SANS",
                    command = self.update_configuration)
        temp3.grid(row=0, column =3)
        tip.ToolTip(temp3, "SANS collimation: reduces the beam divergence and intensity, limits magnet rotation to +/- 5 degrees if the cryostat is on the side of the guide.")
        w.grid(row = 0, column =0)
        x = Label(extraframe, text="Cryostat:")
        temp4 = Radiobutton(extraframe, text="Upstream", variable=self.cryostat, value="Back",
                    command = self.update_configuration)
        temp4.grid(row=1, column =1)
        tip.ToolTip(temp4, "Cryostat between the guide and the sample position. Limits magnet rotation, produces no additional shadows in forward-scattering detectors.")
        temp5 = Radiobutton(extraframe, text="Downstream", variable=self.cryostat, value="Front",
                    command = self.update_configuration)
        temp5.grid(row=1, column =2)
        tip.ToolTip(temp5, "Cryostat between the sample and forward-scattering panels, allows to rotate the magnet up to +/-14 degrees.")
        temp6 = Radiobutton(extraframe, text="No cryostat", variable=self.cryostat, value="None",
                    command = self.update_configuration)
        temp6.grid(row=1, column =3)
        tip.ToolTip(temp6, "No cryostat, no angle limits or detector shadowing, room-temperature measurements only.")
        x.grid(row = 1, column =0)
        y = Label(extraframe, text="Forward detectors:")
        temp7 = Radiobutton(extraframe, text="Diffraction", variable=self.forward_mode, value="Fixed",
                    command = self.update_configuration)
        temp7.grid(row=2, column =1)
        tip.ToolTip(temp7, "Standard mode: the detector panels 1 and 3 follow the magnet rotation.")
        temp8 = Radiobutton(extraframe, text="SANS", variable=self.forward_mode, value="SANS",
                    command = self.update_configuration)
        temp8.grid(row=2, column =2)
        tip.ToolTip(temp8, "SANS mode: the detector panels 1 and 3 are approx. 5.7m away from the sample at a fixed position.")
        y.grid(row = 2, column =0)
        z = Label(extraframe, text="Backward detectors:")
        temp9 = Radiobutton(extraframe, text="Symmetrical", variable=self.backward_mode, value="Symmetrical",
                    command = self.update_configuration)
        temp9.grid(row=3, column =1)
        tip.ToolTip(temp9, "The detector panels 2 and 4 are positioned symmetrically on both sides of the neutron guide.")
        temp10 = Radiobutton(extraframe, text="Left side", variable=self.backward_mode, value="Left side",
                    command = self.update_configuration)
        temp10.grid(row=3, column =2)
        tip.ToolTip(temp10, "The detector panels 2 and 4 are both standing on the left side of the neutron guide.")
        z.grid(row = 3, column =0)
        # extraframe.grid(row = len(fields1)+3, columnspan = 2)
        extraframe.pack()
        Label(extraframe, text="Active detectors (Forward):").grid(row = 4, column = 0)
        Label(extraframe, text="Active detectors (Backward):").grid(row = 5, column = 0)
        temp11 = Checkbutton(extraframe, text = 'Panel 1', variable = self.panel1,
                    command = self.update_configuration)
        temp11.grid(row = 4, column = 1)
        temp12 = Checkbutton(extraframe, text = 'Panel 3', variable = self.panel3,
                    command = self.update_configuration)
        temp12.grid(row = 4, column = 2)
        temp15 = Checkbutton(extraframe, text = 'Panel 5', variable = self.panel5,
                    command = self.update_configuration)
        temp15.grid(row = 4, column = 3)
        temp16 = Checkbutton(extraframe, text = 'Panel 7', variable = self.panel7,
                    command = self.update_configuration)
        temp16.grid(row = 4, column = 4)
        temp13 = Checkbutton(extraframe, text = 'Panel 2', variable = self.panel2,
                    command = self.update_configuration)
        temp13.grid(row = 5, column = 1)
        temp14 = Checkbutton(extraframe, text = 'Panel 4', variable = self.panel4,
                    command = self.update_configuration)
        temp14.grid(row = 5, column = 2)
        tip.ToolTip(temp11, "Include panel 1, the front-right detector panel, in the coverage calculations.")
        tip.ToolTip(temp12, "Include panel 2, which may be placed on either side of the guide, in the coverage calculations.")
        tip.ToolTip(temp13, "Include panel 3, the front-left detector panel, in the coverage calculations.")
        tip.ToolTip(temp14, "Include panel 4, the back-left fixed detector panel, in the coverage calculations.")
        tip.ToolTip(temp15, "Include panel 5, the front far-right detector panel, in the coverage calculations.")
        tip.ToolTip(temp16, "Include panel 7, the front far-left detector panel, in the coverage calculations.")
        vals = self.kernel.instrument
        self.panel1.set(vals['Detectors']['Active'][0])
        self.panel2.set(vals['Detectors']['Active'][1])
        self.panel3.set(vals['Detectors']['Active'][2])
        self.panel4.set(vals['Detectors']['Active'][3])
        self.panel5.set(vals['Detectors']['Active'][4])
        self.panel7.set(vals['Detectors']['Active'][5])
        self.forward_mode.set(vals['Detectors']['Forward'])
        self.backward_mode.set(vals['Detectors']['Backward'])
        self.guidetype.set(vals['Guide'])
        self.cryostat.set(vals['Cryostat'])
    
    @pyqtSlot()
    def update_configuration(self):
        instrument = {}
        instrument['Detectors'] = {}
        instrument['Detectors']['Active'] = [True,  True,  True,  True]
        instrument['Detectors']['Forward'] = 'Fixed'
        instrument['Detectors']['Backward'] = 'Symmetrical'
        instrument['Guide'] = self.guide
        instrument['Cryostat'] = 'Back'
        # print instrument
        self.kernel.take_instrument(instrument)
        
    @pyqtSlot(int)
    def update_guide(self,  number):
        if number == 0:
            self.guide = "Long"
        else:
            self.guide = "SANS"    
        # print instrument
        self.update_configuration()        

    @pyqtSlot(int)
    def update_cryo(self,  number):
        if number == 0:
            self.cryostat = "Long"
        else:
            self.cryostat = "SANS"    
        # print instrument
        self.update_configuration()
    
    @pyqtSlot()
    def chopper_freq(self):
        self.get_new_parameters()
        fr = self.kernel.chopper_freqs()
        ckeys= ['ch1freq',  'ch5freq']
        cdict = {'ch1freq' : [int(fr[1])*60], 
                    'ch5freq' : [int(fr[2])*60]
         }
        self.chop_vbox.takeValues(ckeys,  cdict)
        return None
    
    def save_profile(self):
        self.get_new_parameters()        
        result, ftype = QtWidgets.QFileDialog.getSaveFileName(self.base, 'Save the current settings as:', '.',
           'EXEQ settings file (*.exq)')
#        result = tkFileDialog.asksaveasfilename(defaultextension = 'exq',
#                                              filetypes = [('EXEQ settings file', '*.exq')],
#                                              title = 'Save the current settings as:')
        # print "DEBUG: filename is ", result
        if result == "":
            return None
        else:
            self.kernel.writeout_settings(result)
    
    def load_profile(self):
        result, ftype = QtWidgets.QFileDialog.getOpenFileName(self.base, 'Load the current settings from:', '.',
           'EXEQ settings file (*.exq);;EXEQ web interface file (*.inp);;New EXEQ web interface file (*.exeq);;New InEXEQ web interface file (*.inexeq);;All files (*.*)')
        # print "DEBUG: filename is ", result
        if result == "":
            return None
        else:
            self.kernel.load_settings(result)
        newkeys,  newdict = [], {}
        param, mark_points, prec = self.kernel.current_parameters_str("temp.txt")
        for kk1 in param.keys():
            for kk2 in param[kk1].keys():
                nkey = new_dict_keys(kk1, kk2)
                if len(nkey) > 0:
                    newkeys.append(nkey)
                    if 'hkl' in kk1:
                        val = [param[kk1][kk2]]
                    else:
                        try:
                            val = np.array([float(x) for x in param[kk1][kk2].split()])
                        except:
                            val = param[kk1][kk2]
                    newdict[nkey] = val
        for mnum,  mark in enumerate(mark_points):
            nkey = 'poi' + str(mnum+1)
            newkeys.append(nkey)
            try:
                val = np.array([float(x) for x in mark.split()])
            except:
                val = mark
            newdict[nkey] = val
        for box in [self.chop_vbox,  self.sample_vbox,  self.hkl_vbox,  self.wave_vbox,  self.marker_vbox]:
            box.takeValues(newkeys,  newdict)
    
    def show_plane_fw(self):
        self.orient_mag()
        self.chopper_freq()
        return show_plane_alternative(self.parameters, True, "")
    
    def total_plane_bw(self):
        self.orient_mag()
        self.chopper_freq()
        return self.kernel.show_total_coverage("")
    
    def show_plane_bw(self):
        self.orient_mag()
        self.chopper_freq()
        return self.kernel.show_planes_both("")
        
    def show_eqrange(self):
        self.orient_mag()
        self.chopper_freq()
        return self.kernel.show_total_eq_range("")
        
    def slider_plot(self):
        self.get_new_parameters()
        self.kernel.show_slider_plot("")
        # return show_rspace_point(self.parameters, self.exed, "")
        return None

    def show_help(self):
        self.kernel.show_help()

    def end(self, event):
        # self.master.destroy()
        self.master.quit()
        
if __name__ == '__main__':
    app = QtWidgets.QApplication(sys.argv)
    root = QtWidgets.QMainWindow()
    root.setWindowTitle("EXEQ, the EXED (E,q)-range calculator, release " + '.'.join([major_version, minor_version]))
    # interface = InterfaceEmbedded(root, 800, params)    
    # interface = InterfaceTabbed(root, (1100, 900), params)    
    interface = InterfaceEmbedded(root, (900, 800), params)
    root.show()
    app.exec_()

