#    This file is part of EXEQ.
#
#    EXEQ is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <https://www.gnu.org/licenses/>.
# 
# Copyright (C) Maciej Bartkowiak, 2020

from PyQt5 import QtGui, QtCore, QtWidgets
from PyQt5.QtCore import QObject, pyqtSignal, pyqtSlot
import numpy as np
import copy

class MyDoubleValidator(QtGui.QDoubleValidator):
    def __init__(self, mins, maxes, flen, linedit):
        super().__init__( mins, maxes, flen, linedit)
    def fixup(self, s):
        try:
            val = float(s)
        except:
            return str((self.top() + self.bottom())/2.0)
        else:
            if val > self.top():
                return str(self.top())
            elif val < self.bottom():
                return str(self.bottom())
            else:
                return str((self.top() + self.bottom())/2.0)

class MyIntValidator(QtGui.QIntValidator):
    def __init__(self, mins, maxes, linedit):
        super().__init__( mins, maxes, linedit)
    def fixup(self, s):
        try:
            val = int(s)
        except:
            return str((self.top() + self.bottom())/2.0)
        else:
            if val > self.top():
                return str(self.top())
            elif val < self.bottom():
                return str(self.bottom())
            else:
                return str((self.top() + self.bottom())/2.0)

class InputVariable(QObject):
    values_changed = pyqtSignal()
    def __init__(self, parent, variable,  grid = None,  prec=4):
        super().__init__(parent)
        self.prec = prec
        self.v_name = variable['Name']
        self.v_unit = variable['Unit']
        self.v_len = variable['Length']
        self.v_value = variable['Value']
        self.v_minval = variable['MinValue']
        self.v_maxval = variable['MaxValue']
        self.v_type = variable['Type']
        self.tooltip=""
        for k in variable.keys():
            self.tooltip += str(k) + ": " + str(variable[k]) + '\n'
        self.maxismatched = False
        self.minismatched = False
        self.isinteger = False
        self.istext = False
        if 'int' in self.v_type:
            self.isinteger = True
        elif 'str' in self.v_type:
            self.istext = True
        self.veclen = -1
        self.minlen = -2
        self.maxlen = -3
        if self.istext:
            self.value = self.v_len*[""]
            self.minval = self.v_len*[""]
            self.maxval = self.v_len*["".join(30*['Z'])]
        else:
            self.value = np.zeros(self.v_len)
            self.minval = np.zeros(self.value.shape)
            self.maxval = np.zeros(self.value.shape)
        veclen = self.v_len
        try:
            maxlen = len(self.v_maxval)
        except:
            maxlen = 1
        try:
            minlen = len(self.v_minval)
        except:
            minlen = 1
        self.veclen = veclen
        self.maxlen = maxlen
        self.minlen = minlen
        if maxlen >= veclen:
            self.maxismatched = True
        if minlen >= veclen:
            self.minismatched = True
        if self.istext:
            self.value = self.v_value
        else:
            if self.v_len > 1:
                for n,  x in enumerate(self.v_value):
                    self.value[n] = x
                    try:
                        self.maxval[n] = np.array(self.v_maxval)[n]
                    except:
                        try:
                            self.maxval[n] = np.array(self.v_maxval)[0]
                        except:
                            self.maxval[n] = 1e12
                    try:
                        self.minval[n] = np.array(self.v_minval)[n]
                    except:
                        try:
                            self.minval[n] = np.array(self.v_minval)[0]
                        except:
                            self.minval[n] = 1e12
            else:
                self.value[0] = float(self.v_value)
                self.maxval[0] = float(self.v_maxval)
                self.minval[0] = float(self.v_minval)
            if self.isinteger:
                self.value = self.value.astype(np.int)
                self.maxval = self.maxval.astype(np.int)
                self.minval = self.minval.astype(np.int)
        base = QtWidgets.QWidget(parent)
        label = QtWidgets.QLabel(self.v_name, base)
        label2 = QtWidgets.QLabel(self.v_unit, base)
        label.setToolTip(self.tooltip)
        label2.setToolTip(self.tooltip)
        fields = []
        fbase = QtWidgets.QWidget(base)
        flay = QtWidgets.QHBoxLayout(fbase)
        for ln in range(self.veclen):
            inlin = QtWidgets.QLineEdit(str(self.value[ln]), base)
            inlin.setMinimumSize(QtCore.QSize(30, 10))
            inlin.setMaximumSize(QtCore.QSize(80, 25))
            if self.istext:
                # inlin.setMinimumHeight(20)
                inlin.setText(str(self.value[ln]))
                inlin.setToolTip(self.tooltip)
                inlin.returnPressed.connect(self.updateText)
                inlin.editingFinished.connect(self.updateText)
            else:
                if self.isinteger:
                    dval = MyIntValidator(self.minval[ln], self.maxval[ln], inlin)
                else:
                    dval = MyDoubleValidator(self.minval[ln], self.maxval[ln], self.prec, inlin)
                inlin.setValidator(dval)
                # inlin.setMinimumHeight(20)
                inlin.setText(str(self.value[ln]))
                inlin.setToolTip(self.tooltip)
                if self.isinteger:
                    inlin.returnPressed.connect(self.updateInt)
                    inlin.editingFinished.connect(self.updateInt)
                else:
                    inlin.returnPressed.connect(self.update)
                    inlin.editingFinished.connect(self.update)
            flay.addWidget(inlin)
            fields.append(inlin)
        self.fields = fields
        self.defvalue = copy.deepcopy(self.value)
        if grid is None:
            layout = QtWidgets.QHBoxLayout(base)
            # infield.setSizePolicy(QtWidgets.QSizePolicy.Preferred,QtWidgets.QSizePolicy.Preferred)
            layout.addWidget(label)
            layout.addWidget(fbase)
            layout.addWidget(label2)
        else:
            nGridRows = grid.rowCount()
            grid.addWidget(label,  nGridRows,  0)
            grid.addWidget(fbase,  nGridRows,  1)
            grid.addWidget(label2,  nGridRows,  2)
        if self.isinteger:
            self.updateInt()
        else:
            self.update()
    @pyqtSlot()
    def updateText(self):
        for n,  tfield in enumerate(self.fields):
            val = tfield.text()
            self.value[n] = val
        # print("update was called")
        self.values_changed.emit()
    @pyqtSlot()
    def update(self):
        for n,  tfield in enumerate(self.fields):
            val = tfield.text()
            try:
                val = float(val)
            except:
                val = self.defvalue[n]
                tfield.setText(str(val))
#            else:
#                if val > self.maxval[n]:
#                    val = self.maxval[n]
#                    tfield.setText(str(val))
#                elif val < self.minval[n]:
#                    val = self.minval[n]
#                    tfield.setText(str(val))
            self.value[n] = val
        # print("update was called")
        self.values_changed.emit()
    @pyqtSlot()
    def updateInt(self):
        for n,  tfield in enumerate(self.fields):
            val = tfield.text()
            try:
                val = int(float(val))
            except:
                val = self.defvalue[n]
                tfield.setText(str(val))
#            else:
#                if val > self.maxval[n]:
#                    val = self.maxval[n]
#                    tfield.setText(str(val))
#                elif val < self.minval[n]:
#                    val = self.minval[n]
#                    tfield.setText(str(val))
            self.value[n] = val
        # print("updateInt was called")
        self.values_changed.emit()
    def returnValue(self):
        return self.value
    def takeValue(self,  newvalue):
        for nn in range(len(newvalue)):
            if self.istext:
                tval = str(newvalue[nn])
            else:
                try:
                    tval = float(newvalue[nn])
                except:
                    continue
                if self.isinteger:
                    tval = int(tval)
                if tval > self.maxval[nn]:
                    tval = self.maxval[nn]
                if tval < self.minval[nn]:
                    tval = self.minval[nn]
            self.value[nn] = tval
            # print(self.v_name, "about to set the ",  nn,  "field to",  tval)
            if self.istext:
                self.fields[nn].setText(tval)
            else:
                self.fields[nn].setText(str(round(tval, self.prec)))

class VarBox(QObject):
    values_changed = pyqtSignal()
    def __init__(self, parent, setup_variables,  gname,  prec_override = 4):
        super().__init__(parent)
        self.base = QtWidgets.QGroupBox(gname,  parent)
        # self.base.setSizePolicy(QtWidgets.QSizePolicy.Maximum,QtWidgets.QSizePolicy.Maximum)
        self.glay = QtWidgets.QGridLayout(self.base)
        self.glay.setVerticalSpacing(1)
        self.sample_vars = {}
        self.var_names = []
        self.value_dict = {}
        for v in setup_variables:
            self.sample_vars[v['Key']] = InputVariable(self.base, v,  self.glay,  prec = prec_override)
            self.var_names.append(v['Key'])
            self.value_dict[v['Key']] = self.sample_vars[v['Key']].returnValue()
            self.sample_vars[v['Key']].values_changed.connect(self.updateValues)
    @pyqtSlot()
    def updateValues(self):
        for vn in self.var_names:
            self.value_dict[vn] = self.sample_vars[vn].returnValue()
        # print("SampleBox just did an updateValues")
        self.values_changed.emit()
    @pyqtSlot()
    def updateOutputs(self):
        for vn in self.var_names:
            self.sample_vars[vn].takeValue(self.value_dict[vn])
        # print("SampleBox just did an updateValues")
        self.values_changed.emit()
    def returnValues(self):
        return (self.var_names,  self.value_dict)
    def takeValues(self,  var_names, value_dict):
        for vn in var_names:
            if vn in self.var_names:
                self.sample_vars[vn].takeValue(value_dict[vn])
            
class RadioBox(QObject):
    values_changed = pyqtSignal(int)
    def __init__(self, parent, initval, names, title = "",  vertical = True):
        super().__init__(parent)
        button_stripe = QtWidgets.QWidget(parent)
        if vertical:
            button_layout = QtWidgets.QVBoxLayout(button_stripe)
        else:
            button_layout = QtWidgets.QHBoxLayout(button_stripe)
        if len(title) >0:
            titlab = QtWidgets.QLabel(title, button_stripe)
            button_layout.addWidget(titlab)
        self.radios = []
        self.names= names
        for n in names:
            rr = QtWidgets.QRadioButton(n, button_stripe)
            self.radios.append(rr)
            button_layout.addWidget(rr)
            rr.clicked.connect(self.on_change)
        self.radios[initval].setChecked(True)
        button_group = QtWidgets.QButtonGroup(button_stripe)
        for num, rbut in enumerate(self.radios):
            button_group.addButton(rbut, num)
        self.button_group = button_group
        self.base = button_stripe
    @pyqtSlot(bool)
    def on_change(self,  dummyval):
        self.values_changed.emit(self.button_group.checkedId())
