#    This file is part of EXEQ.
#
#    EXEQ is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <https://www.gnu.org/licenses/>.
# 
# Copyright (C) Maciej Bartkowiak, 2014-2018, 2020

__doc__ = """
In order to increase the flexibility of the code, I am
implementing a parser that will read the detector deifnition
from the XML file rather than from the PAR file.
This way we can split up the detectors into separate panels.
Maciej Bartkowiak, 07 May 2014
"""

import xml.etree.ElementTree as ET
from geometry import *
from units import *
import shadows

# beamstop_angle = np.degrees(np.arctan(8.0/150.0))
flimit = 14.3
blimit = np.degrees(np.arctan(31.92/128.76))

class NewPanels:
    def __init__(self, filename, direction, segments = 5, heavy_cryostat = False, input_data = [], whichpanel=1,
                 SANS = False):
        if input_data == []:
            self.direction = direction
            self.hc = heavy_cryostat
            self.SANS = SANS
            self.rotmat, self.rotback = rotY(-direction), rotY(direction)
            self.rawbanks = get_banks(filename)
            self.panels = []
            self.whichone = whichpanel
            self.sort_items()
        else:
            self.load_data(input_data)
    def dump_data(self):
        data = {}
        data['direction'] = self.direction
        data['hc']        = self.hc
        data['SANS']      = self.SANS
        data['rotmat']    = self.rotmat
        data['rotback']   = self.rotback
        data['rawbanks']  = self.rawbanks
        data['panels'] = []
        for i in self.panels:
            data['panels'].append(i.dump_data())

    def load_data(self, data):
        self.direction  = data['direction']
        self.hc         = data['hc']       
        self.SANS       = data['SANS']   
        self.rotmat     = data['rotmat']   
        self.rotback    = data['rotback']  
        self.rawbanks   = data['rawbanks'] 
        self.panels = []
        for i in data['panels']:
            self.panels.append(Panel(0,0,input_data = i))
        return data
    def sort_items(self):
        for i in self.rawbanks:
            panel = Panel(i, self.direction)
            self.panels.append(panel)
        for i in self.panels:
            # i.apply_topshadow()
            # i.apply_guidebox()
            if i.forward:
                i.apply_cone(limit = 14.3, cryo = self.hc)
            else:
                i.apply_cone(limit = 14.7, cryo = self.hc)
            i.apply_beamstop(self.SANS)
            if self.hc:
                i.apply_cryostat(self.direction)
            i.apply_guide()
            i.find_limits()
    def get_panel(self):
        for i in self.panels:
            if self.whichone == i.number:
                return i
    def make_shadow(self, element, panel):
        """
        Takes in a definition of a shadowing element,
        and of a detector panel in Cartesian coordinates.
        Returns a subset of pixels which create a shadow
        on the instrument coverage.
        """
        result, normresults = [], []
        a, b = element[0], element[1]
        if 't' in a:
            dark = []
            for k in panel:
                limit = np.tan(b) * np.abs(k[:,2])
                temp = k[np.where(k[:,1] >= limit)]
                if len(temp) > 1:
                    dark.append(rotate(self.rotmat, temp))
            if len(dark) > 2:
                result.append(dark)
                dark1 = []
                for i in dark:
                    temp1 = i.copy()
                    norm = length(temp1)
                    temp1[:,0] /= norm
                    temp1[:,1] /= norm
                    temp1[:,2] /= norm
                    dark1.append(temp1)
                normresults.append(dark1)
        if 'b' in a:
            dark = []
            for k in panel:
                limit = np.tan(b) * np.abs(k[:,2])
                temp = k[np.where(k[:,1] <= - limit)]
                if len(temp) > 1:
                    dark.append(rotate(self.rotmat, temp))
            if len(dark) > 2:
                result.append(dark)
                dark1 = []
                for i in dark:
                    temp1 = i.copy()
                    norm = length(temp1)
                    temp1[:,0] /= norm
                    temp1[:,1] /= norm
                    temp1[:,2] /= norm
                    dark1.append(temp1)
                normresults.append(dark1)
        if 'l' in a:
            dark = []
            for k in panel:
                limit = np.tan(b) * np.abs(k[:,2])
                temp = k[np.where(k[:,0] <= -limit)]
                if len(temp) > 1:
                    dark.append(rotate(self.rotmat, temp))
            if len(dark) > 2:
                result.append(dark)
                dark1 = []
                for i in dark:
                    temp1 = i.copy()
                    norm = length(temp1)
                    temp1[:,0] /= norm
                    temp1[:,1] /= norm
                    temp1[:,2] /= norm
                    dark1.append(temp1)
                normresults.append(dark1)
        if 'r' in a:
            dark = []
            for k in panel:
                limit = np.tan(b) * np.abs(k[:,2])
                temp = k[np.where(k[:,0] >= limit)]
                if len(temp) > 1:
                    dark.append(rotate(self.rotmat, temp))
            if len(dark) > 2:
                result.append(dark)
                dark1 = []
                for i in dark:
                    temp1 = i.copy()
                    norm = length(temp1)
                    temp1[:,0] /= norm
                    temp1[:,1] /= norm
                    temp1[:,2] /= norm
                    dark1.append(temp1)
                normresults.append(dark1)
        if 'c' in a:
            limit = np.degrees(b)
            dark = []
            for k in panel:
                temp = k[np.where(cart_to_sph(k)[:,1] <= limit)]
                if len(temp) > 1:
                    dark.append(rotate(self.rotmat, temp))
            if len(dark) > 2:
                result.append(dark)
                dark1 = []
                for i in dark:
                    temp1 = i.copy()
                    norm = length(temp1)
                    temp1[:,0] /= norm
                    temp1[:,1] /= norm
                    temp1[:,2] /= norm
                    dark1.append(temp1)
                normresults.append(dark1)
        if 's' in a:
            limit = np.degrees(b)
            dark = []
            for k in panel:
                temp = rotate(self.rotmat, k)
                temp = temp[np.where(cart_to_sph(temp)[:,1] <= limit)]
                if len(temp) > 1:
                    dark.append(temp)
            if len(dark) > 2:
                result.append(dark)
                dark1 = []
                for i in dark:
                    temp1 = i.copy()
                    norm = length(temp1)
                    temp1[:,0] /= norm
                    temp1[:,1] /= norm
                    temp1[:,2] /= norm
                    dark1.append(temp1)
                normresults.append(dark1)
        if 'X' in a:
            limits = b[0], b[1], b[2], b[3]
            dark = []
            for k in panel:
                temp = rotate(self.rotmat, k)
                temp = k[np.where(k[:,1] <= limits[0])]
                temp = k[np.where(k[:,1] >= limits[1])]
                temp = k[np.where(k[:,0] >= limits[2])]
                temp = k[np.where(k[:,0] <= limits[3])]
                if len(temp) > 1:
                    dark.append(temp)
            if len(dark) > 2:
                result.append(dark)
                dark1 = []
                for i in dark:
                    temp1 = i.copy()
                    norm = length(temp1)
                    temp1[:,0] /= norm
                    temp1[:,1] /= norm
                    temp1[:,2] /= norm
                    dark1.append(temp1)
                normresults.append(dark1)
        if 'g' in a:
            limit = np.degrees(b)
            dark = []
            for k in panel:
                temp = rotate(self.rotmat, k)
                temp = temp[np.where(cart_to_sph(temp)[:,1] >= (180.0 - limit))]
                if len(temp) > 1:
                    dark.append(temp)
            if len(dark) > 2:
                result.append(dark)
                dark1 = []
                for i in dark:
                    temp1 = i.copy()
                    norm = length(temp1)
                    temp1[:,0] /= norm
                    temp1[:,1] /= norm
                    temp1[:,2] /= norm
                    dark1.append(temp1)
                normresults.append(dark1)
        return result, normresults
    def add_shadows(self, hc = False):
        shadowlist = []
        normlist = []
        for i in range(len(self.rawbanks)):
            sets = self.relative_range[i]
            forward = self.rawbanks[i]['positions'][:,2].mean() > 0.0
            if forward:
                for k in shadows.forward:
                    a, b = self.make_shadow(k, sets)
                    shadowlist += a 
                    normlist += b
            else:
                for k in shadows.backward:
                    a, b = self.make_shadow(k, sets)
                    shadowlist += a 
                    normlist += b
                # for later
                if hc:
                    for k in shadows.extra:
                        a, b = self.make_shadow(k, sets)
                        shadowlist += a 
                        normlist += b
        return shadowlist, normlist
    def tubes_for_plotting(self):
        alltubes, goodtubes, shadowtubes = [], [], []
        for i in self.panels:
            for k in i.tubes:
                alltubes.append(k.frame)
                goodtubes += k.goodparts
                shadowtubes += k.badparts
        return alltubes, goodtubes, shadowtubes
    def tubes_for_exact(self):
        alltubes, goodtubes, shadowtubes = [], [], []
        for i in self.panels:
            for k in i.tubes:
                alltubes.append(k.frame)
                goodtubes += k.goodparts
                shadowtubes += k.badparts
        return alltubes, goodtubes, shadowtubes
 
def get_banks(filename = "normal_exed.xml"):
    
    tree = ET.parse(filename)
    root = tree.getroot()
    
    banknames = []  # this lists "bank1", "bank2", etc.
    banks = []      # this stores the coordinates of tubes in each bank
    tubetypes = []  # and this stores the names of tubes, i.e. 'tube_S', 'tube_S', 'tube'
    # In the end we can iterate over range(len(banknames))
    # and get name = banknames[i], positions = banks[i], tube_type = tubetypes[i]
    # for each bank.
    
    tubenames = []  # this is the list of names: 'tube_S', 'tube_M', 'tube_L', 'tube'.
    tubepixels = [] # cooridnates of pixels in the tube
    pixeltypes = [] # kind of pixel: 'pixel' or 'oldpixel'
    # Again we can iterate over those, and have matching records
    # so for each name of the tube we can get the corresponding coordinates.
    # The pixel type is not really useful here.
    
    for i in root.iter('type'):
        if 'bank' in i.attrib['name']:
            banks.append([])
            banknames.append(i.attrib['name'])
            for k in i.iter('component'):
                tubetypes.append(k.attrib['type'])
                for j in k.iter('location'):
                    banks[-1].append([float(x) for x in [j.attrib['x'], j.attrib['y'], j.attrib['z']]])
    
    for i in root.iter('type'):
        if 'tube' in i.attrib['name']:
            tubepixels.append([])
            tubenames.append(i.attrib['name'])
            for k in i.iter('component'):
                pixeltypes.append(k.attrib['type'])
                for j in k.iter('location'):
                    tubepixels[-1].append([float(x) for x in [j.attrib['x'], j.attrib['y'], j.attrib['z']]])
    
    result = []
    for i in range(len(banknames)):
        atube = np.array(banks[i])
        tangles = cart_to_sph(atube)
        btube = np.array(tubepixels[tubenames.index(tubetypes[i])])
        step_v = abs((atube[1:,1] - atube[:-1,1]).mean())
        step_h = abs((atube[1:,0] - atube[:-1,0]).mean())
        # print "atube shape ", atube.shape
        name1 = banknames[i]
        result.append({})
        result[-1]['name'] = name1
        result[-1]['positions'] = atube
        result[-1]['pixels'] = btube
        # new section
        #if (atube[:,0] >= 0.0).all() or (atube[:,0] <= 0.0).all():
            #result.append({})
            #result[-1]['name'] = name1
            #result[-1]['positions'] = atube
            #result[-1]['pixels'] = btube
        #elif (atube[:,2].mean() <= 0.0):
            #result.append({})
            #result[-1]['name'] = name1
            #result[-1]['positions'] = atube
            #result[-1]['pixels'] = btube
        #else:
            #result.append({})
            #result[-1]['name'] = name1 + '_left'
            #result[-1]['positions'] = atube[np.where(atube[:,0] >= 0.0)]
            #result[-1]['pixels'] = btube
            #result.append({})
            #result[-1]['name'] = name1 + '_right'
            #result[-1]['positions'] = atube[np.where(atube[:,0] <= 0.0)]
            #result[-1]['pixels'] = btube
        # end of new section
    # now we have a list, where every bank is a dictionary.
    # the entries of each bank are: 'name', 'positions' and 'pixels'.
    return result

def refine_banks_magnet_new(banks, direction = 0.0):
    forward = []
    backward = []
    fshadow = []
    bshadow = []
    fmax = 0.0
    bmax = 0.0
    for i in range(len(banks)):
        ii = banks[i]
        edge = []
        total = []
        pos = ii['positions']
        # pos1 = ii['positions'][np.unique(np.linspace(0, len(ii['positions'])-1, segmentation).astype(np.int))]
        for j in pos: #[selection.astype(np.int)]:
            points = j + ii['pixels'] # [selection.astype(np.int)]
            # points = points[np.where(cart_to_sph(points)[:,1] >= beamstop_angle)]
            absolute = cart_to_sph(rotate(rotY(direction), points))
            if absolute[:,1].mean() < 90.0:
                absolute_temp = cart_to_sph(rotate(rotY(direction+0.3), points))
                ind = [num for num, it in enumerate(absolute_temp[:,1]) if it < flimit]
            elif absolute[:,1].mean() > 90.0:
                ind = [num for num, it in enumerate(absolute[:,1]) if it > (180.0 - blimit)]
            if (len(ind) > 0) and (len(points) > 0):
                trimmed = points[ind]
            else:
                trimmed = []
            # print len(trimmed)
            if len(trimmed) > 0:
                edge.append(trimmed)
        if len(edge) > 0:
            ii['edge'] = edge
        else:
            ii['edge'] = np.array([False])
        # print ii['name'], ii['edge'].shape
        # print ii['edge']
        # now we need to pass the right coordinates to the rest of the code, as if nothing
        # has happened. This could be difficult, but what the heck, we are clever.
    for i in banks:
        for j in i['edge']: # and len(i['edge']) > 3:
            if j.any():
                n1 = length(j)
                corners = np.column_stack([j[:,0]/n1, j[:,1]/n1, j[:,2]/n1])
                if i['positions'][:,2].mean() >= 0.0:
                    forward.append(corners)
                    fmax = max(fmax, n1.max())
                    for sh in shadows.forward:
                        fshadow += make_shadow(sh, i, direction)
                    for sh in shadows.beamstop:
                        fshadow += make_shadow(sh, i, direction)
                else:
                    backward.append(corners)
                    bmax = max(bmax, n1.max())
                    for sh in shadows.backward:
                        bshadow += make_shadow(sh, i, direction)
                    for sh in shadows.chopper_housing:
                        bshadow += make_shadow(sh, i, direction)
    return forward, backward, fmax, bmax, fshadow, bshadow
    
def get_banks_old(filename = "normal_exed.xml"):
    
    tree = ET.parse(filename)
    root = tree.getroot()
    
    banknames = []  # this lists "bank1", "bank2", etc.
    banks = []      # this stores the coordinates of tubes in each bank
    tubetypes = []  # and this stores the names of tubes, i.e. 'tube_S', 'tube_S', 'tube'
    # In the end we can iterate over range(len(banknames))
    # and get name = banknames[i], positions = banks[i], tube_type = tubetypes[i]
    # for each bank.
    
    tubenames = []  # this is the list of names: 'tube_S', 'tube_M', 'tube_L', 'tube'.
    tubepixels = [] # cooridnates of pixels in the tube
    pixeltypes = [] # kind of pixel: 'pixel' or 'oldpixel'
    # Again we can iterate over those, and have matching records
    # so for each name of the tube we can get the corresponding coordinates.
    # The pixel type is not really useful here.
    
    for i in root.iter('type'):
        if 'bank' in i.attrib['name']:
            banks.append([])
            banknames.append(i.attrib['name'])
            for k in i.iter('component'):
                tubetypes.append(k.attrib['type'])
                for j in k.iter('location'):
                    banks[-1].append([float(x) for x in [j.attrib['x'], j.attrib['y'], j.attrib['z']]])
    
    for i in root.iter('type'):
        if 'tube' in i.attrib['name']:
            tubepixels.append([])
            tubenames.append(i.attrib['name'])
            for k in i.iter('component'):
                pixeltypes.append(k.attrib['type'])
                for j in k.iter('location'):
                    tubepixels[-1].append([float(x) for x in [j.attrib['x'], j.attrib['y'], j.attrib['z']]])
    
    result = []
    for i in range(len(banknames)):
        atube = np.array(banks[i])
        tangles = cart_to_sph(atube)
        btube = np.array(tubepixels[tubenames.index(tubetypes[i])])
        step_v = abs((atube[1:,1] - atube[:-1,1]).mean())
        step_h = abs((atube[1:,0] - atube[:-1,0]).mean())
        # print "atube shape ", atube.shape
        name1 = banknames[i]
        # step 1: away from the beamstop
        pos = []
        end = 0
        for it, obj in enumerate(tangles):
            if obj[1] >= beamstop_angle:
                pos.append(atube[it])
            else:
                end = it
                break
        if len(pos) > 0:
            result.append({})
            pos = np.row_stack(pos)
            result[-1]['name'] = name1 + "_left"
            result[-1]['positions'] = pos
            result[-1]['pixels'] = btube
        # step 2: at the beamstop
        pos = []
        for it in range(end, len(atube)):
            if (end > 0) and (end < (len(atube) - 1)):
                pos.append(atube[end-1])
            if tangles[it, 1] < beamstop_angle:
                pos.append(atube[it])
            else:
                if (it > end +2):
                    pos.append(atube[it])
                end = it
                break
        if len(pos) > 0:
            name2 = name1 + "_up"
            result.append({})
            pos = np.row_stack(pos)
            result[-1]['name'] = name2
            result[-1]['positions'] = pos
            result[-1]['pixels'] = btube[np.where(btube[:,1] > 0.0)]
            name2 = name1 + "_down"
            result.append({})
            result[-1]['name'] = name2
            result[-1]['positions'] = pos
            result[-1]['pixels'] = btube[np.where(btube[:,1] < 0.0)]
        # step 3: after the beamstop
        pos = []
        for it in range(end, len(atube)):
            if (end > 0) and (end < (len(atube) - 1)):
                pos.append(atube[end-1])
            if tangles[it, 1] >= beamstop_angle:
                pos.append(atube[it])
            else:
                end = it
                break
        if len(pos) > 0:
            result.append({})
            pos = np.row_stack(pos)
            result[-1]['name'] = name1 + "_right"
            result[-1]['positions'] = pos
            result[-1]['pixels'] = btube
    # now we have a list, where every bank is a dictionary.
    # the entries of each bank are: 'name', 'positions' and 'pixels'.
    return result
    
def refine_banks_s(banks, segmentation = 2):
    forward = []
    backward = []
    fmax = 0.0
    bmax = 0.0
    for i in banks:
        ind1 = [int(x) for x in np.linspace(0, len(i['positions']) - 1, segmentation)]
        ind2 = [int(x) for x in np.linspace(0, len(i['pixels']) - 1, 2)]
        a, b = i['positions'][ind1], i['pixels'][ind2]
        lll = []
        for x in b:
            lll.append(a+x)
        corners = np.row_stack(lll)
        n1 = length(corners)
        corners = np.column_stack([corners[:,0]/n1, corners[:,1]/n1, corners[:,2]/n1])
        forward.append(corners)
        fmax = max(fmax, n1.max())
        # if i['positions'][:,2].mean() >= 0.0:
        #     forward.append(corners)
        #     fmax = max(fmax, n1.max())
        # else:
        #     backward.append(corners)
        #    bmax = max(bmax, n1.max())
    return forward, forward, fmax, fmax
    
def refine_banks_magnet_exact(banks, segmentation = 4, direction = 0.0, limit = 15.0):
    forward = []
    backward = []
    fmax = 0.0
    bmax = 0.0
    for i in range(len(banks)):
        ii = banks[i]
        edge = []
        #
        if ("left" in ii['name']) or ("right" in ii['name']):
            pos = ii['positions'][np.unique(np.linspace(0, len(ii['positions'])-1, segmentation).astype(np.int))]
        else:
            pos = ii['positions']
        #
        for j in pos: #[selection.astype(np.int)]:
            points = j + ii['pixels'] # [selection.astype(np.int)]
            points = points[np.where(cart_to_sph(points)[:,1] >= beamstop_angle)]
            absolute = cart_to_sph(rotate(rotY(direction), points))
            if absolute[:,1].mean() < 90.0:
                ind = [num for num, it in enumerate(absolute[:,1]) if it < limit]
            elif absolute[:,1].mean() > 90.0:
                ind = [num for num, it in enumerate(absolute[:,1]) if it > (180.0 - limit)]
            if (len(ind) > 0) and (len(points) > 0):
                trimmed = points[ind]
            else:
                trimmed = []
            # print len(trimmed)
            if len(trimmed) > 1:
                sel2 = np.unique(np.linspace(0, len(trimmed)-1, segmentation).astype(np.int))
                edge.append(trimmed[sel2])
            elif len(trimmed) == 1:
                edge.append(trimmed[0])
        if len(edge) > 0:
            ii['edge'] = np.row_stack(edge)
        else:
            ii['edge'] = np.array([False])
        # print ii['name'], ii['edge'].shape
        # print ii['edge']
        # now we need to pass the right coordinates to the rest of the code, as if nothing
        # has happened. This could be difficult, but what the heck, we are clever.
    for i in banks:
        if i['edge'].any(): # and len(i['edge']) > 3:
            n1 = length(i['edge'])
            corners = np.column_stack([i['edge'][:,0]/n1, i['edge'][:,1]/n1, i['edge'][:,2]/n1])
            if i['positions'][:,2].mean() >= 0.0:
                forward.append(corners)
                fmax = max(fmax, n1.max())
            else:
                backward.append(corners)
                bmax = max(bmax, n1.max())
#    for i in forward:
#        print i.shape
#    for j in backward:
#        print j.shape
    return forward, backward, fmax, bmax
    
def tubes_for_plotting(banks, direction = 0.0, limit = 15.0):
    tubes = []
    for i in range(len(banks)):
        ii = banks[i]
        edge = []
        pos = ii['positions']
        for j in pos:
            points = j + ii['pixels'] # [selection.astype(np.int)]
            points = points[np.where(cart_to_sph(points)[:,1] >= beamstop_angle)]
            absolute = cart_to_sph(rotate(rotY(direction), points))
            if absolute[:,1].mean() < 90.0:
                ind = [num for num, it in enumerate(absolute[:,1]) if it < limit]
            elif absolute[:,1].mean() > 90.0:
                ind = [num for num, it in enumerate(absolute[:,1]) if it > (180.0 - limit)]
            if (len(ind) > 0) and (len(points) > 0):
                trimmed = points[ind]
            else:
                trimmed = []
            # print len(trimmed)
            if len(trimmed) > 1:
                tubes.append((trimmed[0], trimmed[-1]))
    return tubes

class Tube:
    def __init__(self, pos, pixels, width = 1.27, input_data = []):
        if input_data == []:
            direction = rotate(rotY(90.0),pos)
            edge = normalise(direction)*width/200.0
            self.frame = simple_sort_points(np.row_stack((pos+pixels[0]+edge,pos+pixels[-1]+edge,pos+pixels[-1]-edge,pos+pixels[0]-edge)))[0]
            self.plane = [self.frame[0], normalise(np.cross(self.frame[1]-self.frame[0], self.frame[2] - self.frame[0]))]
            self.direction = direction
            self.middle = pos + pixels.mean(0)
            points = pixels + pos
            ptlen = length(points)
            self.points = np.column_stack([points[:,0]/ptlen, points[:,1]/ptlen, points[:,2]/ptlen])
            self.goodparts = [self.frame]
            self.badparts = []
            self.angular = tubepoints(self.frame)
        else:
            self.load_data(input_data)
    def dump_data(self):
        data = {}
        data['frame'] = self.frame
        data['plane'] = self.plane
        data['direction'] = self.direction
        data['middle'] = self.middle
        data['points'] = self.points
        data['goodparts'] = self.goodparts
        data['badparts'] = self.badparts
        data['angular'] = self.angular
        return data
    def load_data(self, data):
        self.frame = data['frame']
        self.plane = data['plane']
        self.direction = data['direction']
        self.middle = data['middle']
        self.points = data['points']
        self.goodparts = data['goodparts']
        self.badparts = data['badparts']
        self.angular = data['angular']
    def clip_plane(self, plane = np.zeros((3,2)), middle = np.zeros(3), inside = True):
        good, bad = [], []
        normal1 = normalise(np.cross(plane[0], plane[1]))
        normal2 = self.plane[1]
        h1, h2 = np.dot(normal1, plane[0]), np.dot(normal2, self.plane[0])
        c1 = (h1 - h2 * np.dot(normal1, normal2)) / (1- np.dot(normal1, normal2)**2)
        c2 = (h2 - h1 * np.dot(normal1, normal2)) / (1- np.dot(normal1, normal2)**2)
        n3 = np.cross(normal1, normal2)
        p1 = c1*normal1 + c2*normal2 # + self.plane[0]
        center = p1 + np.dot(n3, middle-p1)*n3
        ref = middle - center
        # print "middle", middle
        # print "centre", center
        # print "reference", ref
        for i in self.goodparts:
            t = len(i)
            if t > 1:
                points_in = i[np.where(np.dot(i-center, ref) >= 0.0)]
                points_out = i[np.where(np.dot(i-center, ref) < 0.0)]
                cuts = []
                for k in range(t):
                    worked, cut = intersection(np.row_stack([i[k],i[(k+1)%t]]), plane)
                    if worked:
                        cuts.append(cut)
                if len(cuts) > 0:
                    cuts = np.row_stack(cuts)
                    if inside:
                        good.append(simple_sort_points(np.row_stack([points_in, cuts]))[0])
                        bad.append(simple_sort_points(np.row_stack([points_out, cuts]))[0])
                    else:
                        good.append(simple_sort_points(np.row_stack([points_out, cuts]))[0])
                        bad.append(simple_sort_points(np.row_stack([points_in, cuts]))[0])
                else:
                    if inside:
                        good.append(simple_sort_points(points_in)[0])
                        bad.append(simple_sort_points(points_out)[0])
                    else:
                        good.append(simple_sort_points(points_out)[0])
                        bad.append(simple_sort_points(points_in)[0])
                # print good
                # print bad
                # import sys
                # sys.exit(0)
        self.goodparts = good
        self.badparts += bad
    def clip_hplane(self, angle, inside = True):
        good, bad = [], []
        for i in self.goodparts:
            if len(i) > 2:
                newh = i[:,2]*np.tan(angle)
                mask = i[:,1] < newh
                cuts = i.copy()
                cuts[:,1] = newh
                if inside:
                    newframe = i[np.where(mask)]
                    badpart = i[np.where(np.logical_not(mask))]
                else:
                    newframe = i[np.where(np.logical_not(mask))]
                    badpart = i[np.where(mask)]
                good.append(simple_sort_points(np.row_stack([newframe, cuts]))[0])
                bad.append(simple_sort_points(np.row_stack([badpart, cuts]))[0])
                print(good)
                print(bad)
        self.goodparts = good
        self.badparts += bad
        import sys
        sys.exit(0)
    def clip_2hplanes(self, angle1, angle2, inside = True):
        good, bad = [], []
        for i in self.goodparts:
            if len(i) > 2:
                newh1 = i[:,2]*np.tan(angle1)
                mask1 = i[:,1] > newh1
                cuts1 = i.copy()
                cuts1[:,1] = newh1
                newh2 = i[:,2]*np.tan(angle2)
                mask2 = i[:,1] < newh2
                cuts2 = i.copy()
                cuts2[:,1] = newh2
                newframe1 = i[np.where(mask1)]
                newframe2 = i[np.where(mask2)]
                if (len(newframe1) > 0) and (len(newframe2) > 0):
                    if inside:
                        bad.append(simple_sort_points(np.row_stack([newframe1, cuts1]))[0])
                        bad.append(simple_sort_points(np.row_stack([newframe2, cuts2]))[0])
                        good.append(simple_sort_points(np.row_stack([cuts1, cuts2]))[0])
                    else:
                        good.append(simple_sort_points(np.row_stack([newframe1, cuts1]))[0])
                        good.append(simple_sort_points(np.row_stack([newframe2, cuts2]))[0])
                        bad.append(simple_sort_points(np.row_stack([cuts1, cuts2]))[0])
                # print good
                # print bad
                    self.goodparts = good
                    self.badparts += bad
        # import sys
        # sys.exit(0)
    def clip_2planes(self, plane1 = np.zeros((3,2)), plane2 = np.zeros((3,2)), middle = np.zeros(3), inside = True):
        good, bad = [], []
        normal1 = normalise(np.cross(plane[0], plane[1]))
        normal2 = self.plane[1]
        h1, h2 = np.dot(normal1, plane[0]), np.dot(normal2, self.plane[0])
        c1 = (h1 - h2 * np.dot(normal1, normal2)) / (1- np.dot(normal1, normal2)**2)
        c2 = (h2 - h1 * np.dot(normal1, normal2)) / (1- np.dot(normal1, normal2)**2)
        n3 = np.cross(normal1, normal2)
        p1 = c1*normal1 + c2*normal2 # + self.plane[0]
        center = p1 + np.dot(n3, middle-p1)*n3
        ref = middle - center
        # print "middle", middle
        # print "centre", center
        # print "reference", ref
        for i in self.goodparts:
            t = len(i)
            if t > 1:
                points_in = i[np.where(np.dot(i-center, ref) >= 0.0)]
                points_out = i[np.where(np.dot(i-center, ref) < 0.0)]
                cuts = []
                for k in range(t):
                    worked, cut = intersection(np.row_stack([i[k],i[(k+1)%t]]), plane)
                    if worked:
                        cuts.append(cut)
                if len(cuts) > 0:
                    cuts = np.row_stack(cuts)
                    if inside:
                        good.append(simple_sort_points(np.row_stack([points_in, cuts]))[0])
                        bad.append(simple_sort_points(np.row_stack([points_out, cuts]))[0])
                    else:
                        good.append(simple_sort_points(np.row_stack([points_out, cuts]))[0])
                        bad.append(simple_sort_points(np.row_stack([points_in, cuts]))[0])
                else:
                    if inside:
                        good.append(simple_sort_points(points_in)[0])
                        bad.append(simple_sort_points(points_out)[0])
                    else:
                        good.append(simple_sort_points(points_out)[0])
                        bad.append(simple_sort_points(points_in)[0])
                # print good
                # print bad
                # import sys
                # sys.exit(0)
        self.goodparts = good
        self.badparts += bad

    def clip_cryostat(self, angle, inside = True, forward = True):
        if forward:
            face1 = np.array([[-500.0,206.0,1846.0],[500.0,206.0,1846.0]])
            face2 = np.array([[-500.0,76.0 - 20.0,1846.0 - 1264.0],[500.0, 76.0-20.0, 1846.0 - 1264.0]])
            # cone1 = (np.array([0.0, 0.0, 1.0]), np.degrees(np.arctan(140.0/(1846-1264.0))))
            # cone2 = (np.array([0.0, 0.0, 1.0]), np.degrees(np.arctan(468.0/(1846.0))))
        else:
            face1 = np.array([[-500.0,206.0,-1846.0],[500.0,206.0,-1846.0]])
            face2 = np.array([[-500.0,76.0 - 20.0,-(1846.0 - 1264.0)],[500.0, 76.0-20.0, -(1846.0 - 1264.0)]])
            # cone1 = (np.array([0.0, 0.0, -1.0]), np.degrees(np.arctan(140.0/(1846-1264.0))))
            # cone2 = (np.array([0.0, 0.0, -1.0]), np.degrees(np.arctan(468.0/(1846.0))))
        # self.clip_cone(cone1[0], cone1[1])
        # self.clip_cone(cone2[0], cone2[1])
        face1 = rotate(rotY(angle), face1)
        face2 = rotate(rotY(angle), face2)
        mean1, mean2 = face1.mean(0), face2.mean(0)
        # print "means: ", mean1, mean2
        mean1[1] = 0.0
        mean2[1] = 0.0
        nor1, nor2 = normalise(mean1), normalise(mean2)
        p1, n1 = self.plane[0], self.plane[1]
        lam1 = (np.dot(p1, n1) - np.dot(mean1, n1))/np.dot(nor1,n1)
        lam2 = (np.dot(p1, n1) - np.dot(mean2, n1))/np.dot(nor2,n1)
        if np.dot(self.middle, mean1) > 0.0:
            self.clip_plane(plane = np.row_stack([face1[0], face1[1]]),
                            middle = mean1 + lam1*nor1, inside = inside)
        if np.dot(self.middle, mean2) > 0.0:
            self.clip_plane(plane = np.row_stack([face2[0], face2[1]]),
                            middle = mean2 + lam2*nor2, inside = inside)
    def clip_allplanes(self, hangle, vangle, inside = True):
        good, bad = [], []
        for i in self.goodparts:
            if len(i) > 2:
                if (np.fmod(np.abs(tubepoints(i)[:,1] - np.pi), 2*np.pi) < np.abs(vangle)).all():
                    half = i[np.where(i[:,1] > 0.0)]
                    if len(half) > 1:
                        newh = -half[:,2]*np.tan(hangle)
                        mask = half[:,1] > newh
                        cuts = half.copy()
                        cuts[:,1] = newh
                        newframe = half[np.where(np.logical_not(mask))]
                        badpart = half[np.where(mask)]
                        pum = simple_sort_points(np.row_stack([newframe, cuts]))[0]
                        if len(pum) > 2:
                            good.append(pum)
                        bad.append(simple_sort_points(np.row_stack([badpart, cuts]))[0])
                    half = i[np.where(i[:,1] < 0.0)]
                    if len(half) > 1:
                        newh = half[:,2]*np.tan(hangle)
                        mask = half[:,1] < newh
                        cuts = half.copy()
                        cuts[:,1] = newh
                        newframe = half[np.where(mask)]
                        badpart = half[np.where(np.logical_not(mask))]
                        pum = simple_sort_points(np.row_stack([newframe, cuts]))[0]
                        if len(pum) > 2:
                            good.append(pum)
                        bad.append(simple_sort_points(np.row_stack([badpart, cuts]))[0])
                else:
                    good = self.goodparts
        self.goodparts = good
        self.badparts += bad
    def clip_cone(self, axis, angle = 15.0, inside = True):
        good, bad = [], []
        for i in self.goodparts:
            if len(i) > 2:
                inparts, outparts = cone_intersection(i, axis, angle)
                good += inparts
                bad += outparts
        if inside:
            self.goodparts = good
            self.badparts += bad
        else:
            self.goodparts = bad
            self.badparts += good
    def qvectors(self, wave):
        k = 1.0/wave
        result = k*(self.points - np.array([0.0, 0.0, 1.0]))
        return result

class Panel:
    def __init__(self, rawdata, mag_angle, input_data = []):
        if input_data == []:
            self.tubecoords = rawdata['positions']
            self.pixels = rawdata['pixels']
            self.number = int(rawdata['name'].split('bank')[-1])
            self.tubes = []
            self.limits, self.tubelimits = [], []
            self.grid, self.stepsize, self.tubecount = [], 0.0, 0
            self.middle = self.tubecoords.mean(0) + self.pixels.mean(0)
            #self.corners = tubepoints(np.row_stack(
                            #[self.tubecoords.min(0) + self.pixels.min(0),
                            #self.tubecoords.min(0) + self.pixels.max(0),
                            #self.tubecoords.max(0) + self.pixels.min(0),
                            #self.tubecoords.max(0) + self.pixels.max(0)
                            #]))
            self.normal = normalise(self.middle)
            for i in self.tubecoords:
                self.tubes.append(Tube(i, self.pixels))
            if self.tubecoords[:,2].mean() > 0.0:
                self.forward = True
            else:
                self.forward = False
            self.magnet = mag_angle
            self.rotmat, self.rotback = rotY(-mag_angle), rotY(mag_angle)
        else:
            self.load_data(input_data)
    def dump_data(self):
        data = {}
        data['tubecoords'] = self.tubecoords
        data['pixels']     = self.pixels
        data['limits']     = self.limits
        data['tubelimits'] = self.tubelimits
        data['grid']       = self.grid
        data['stepsize']   = self.stepsize
        data['tubecount']  = self.tubecount
        data['middle']     = self.middle
        data['normal']     = self.normal
        data['forward']    = self.forward
        data['magnet']     = self.magnet
        data['rotmat']     = self.rotmat
        data['rotback']    = self.rotback
        data['tubes'] = []
        for i in self.tubes:
            data['tubes'].append(i.dump_data())
        return data
    def load_data(self, data):
        self.tubecoords  =    data['tubecoords']
        self.pixels  =    data['pixels']    
        self.limits  =    data['limits']    
        self.tubelimits  =    data['tubelimits']
        self.grid  =    data['grid']      
        self.stepsize  =    data['stepsize']  
        self.tubecount  =    data['tubecount'] 
        self.middle  =    data['middle']    
        self.normal  =    data['normal']    
        self.forward  =    data['forward']   
        self.magnet  =    data['magnet']    
        self.rotmat  =    data['rotmat']    
        self.rotback  =    data['rotback']   
        self.tubes = []
        for i in data['tubes']:
            self.tubes.append(Tube(0, 0, 0, input_data = i))
    def apply_cone(self, limit = 15.0, cryo = False):
        if self.forward:
            axis = rotate(rotY(self.magnet + 0.3), np.array([0.0, 0.0, 1.0]))
        else:
            axis = rotate(self.rotback, np.array([0.0, 0.0, 1.0]))
        if cryo:
            limlim = min(np.degrees(np.arctan(140.0/(1846-1264.0))), np.degrees(np.arctan(468.0/(1846.0))))
        else:
            limlim = limit
        for i in self.tubes:
            i.clip_cone(axis, angle = limlim, inside = True)
    def apply_beamstop(self, SANS):
        if self.forward:
            axis = np.array([0.0, 0.0, 1.0])
            if SANS:
                # limit = 0.25
                limit = 0.4
            else:
                limit = np.degrees(np.arctan(8.0/150.0))
            for i in self.tubes:
                if (i.middle[0] > -0.182 ) and (i.middle[0] < 0.2025):
                    i.clip_2hplanes(angle1 = np.arctan(0.164/4.495), angle2 = np.arctan(-0.27/4.495), inside = False)
                    # i.clip_cone(i.middle - np.array([0, (0.164-0.27)/2.0, 0.0 ]), angle = (np.arctan(-0.27/4.495)+ np.arctan(0.164/4.495))/2.0, inside = False)
    def apply_cryostat(self, direction):
        # limit = np.radians(direction)
        for i in self.tubes:
            i.clip_cryostat(direction, inside = True, forward = self.forward)
    def apply_topshadow(self):
        if not self.forward:
            axis = np.array([0.0, 0.0, 1.0])
            plane_point = np.array([0.0, 9.35, -93.15])
            normal = normalise(np.array([0.0, 93.15, 9.35]))
            for i in self.tubes:
                i.clip_hplane(angle = np.arctan(9.35/-93.15))
    def apply_guidebox(self):
        if not self.forward:
            axis = np.array([0.0, 0.0, 1.0])
            plane_point = np.array([0.0, 9.35, -93.15])
            normal = normalise(np.array([0.0, 93.15, 9.35]))
            for i in self.tubes:
                i.clip_allplanes(hangle = np.arctan(12.5/-100.0),vangle = np.arctan(7.5/-100.0))
    def apply_guide(self):
        axis = np.array([0.0, 0.0, 1.0])
        if not self.forward:
            limit = np.degrees(np.arctan(15.0/250.0))
            for i in self.tubes:
                if (i.middle[0] > -0.6 ) and (i.middle[0] < 0.6):
                    i.badparts += i.goodparts
                    i.goodparts = []
    def find_limits(self):
        temp = []
        for i in self.tubes:
            temp.append(i.angular)
        self.tubelimits = temp
        temp2 = np.row_stack(temp)
        # self.limits = [np.fmod(temp2.min(0)+4*np.pi, 2*np.pi), np.fmod(temp2.max(0) + 4*np.pi, 2*np.pi)]
        # self.limits[0] = self.limits[0] - 2*np.pi*(self.limits[0] > 3.0/2.0*np.pi)
        # self.limits[1] = self.limits[1] - 2*np.pi*(self.limits[1] > 3.0/2.0*np.pi)
        self.limits = (temp2.min(0), temp2.max(0))
        self.stepsize = (self.limits[1][1] - self.limits[0][1])/len(temp)
        self.grid = np.linspace(self.limits[0][1], self.limits[1][1], len(temp) + 1)
        self.tubecount = len(temp)
    def righttube(self, coords):
        if self.forward:
            alpha = coords[1]
            pos = (alpha < self.grid).sum()
            if (pos <= 0) or (pos > self.tubecount):
                return None
            else:
                return self.tubes[pos-1]
        else:
            alpha = coords[1]
            pos = (alpha < self.grid).sum()
            if (pos <= 0) or (pos > self.tubecount):
                return None
            else:
                return self.tubes[pos-1]
    def point_in_range(self, coords):
        tube = self.righttube(coords)
        if tube == None:
            return False
        if coords[2] > tube.angular[:,2].max() or coords[2] < tube.angular[:,2].min():
            return False
        else:
            for wrong in tube.goodparts:
                if len(wrong) > 0:
                    i = tubepoints(wrong)
                    top, bottom = i.max(0)[2], i.min(0)[2]
                    if (coords[2] <= top) and (coords[2] >= bottom):
                        return True
        return False
    def point_in_bigframe(self, coords):
        if coords[2] > self.limits[1][2] or coords[2] < self.limits[0][2]:
            return False
        else:
            if (coords[1] <= self.limits[1][1]) and (coords[1] >= self.limits[0][1]):
                return True
        return False
