#    This file is part of EXEQ.
#
#    EXEQ is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <https://www.gnu.org/licenses/>.
# 
# Copyright (C) Maciej Bartkowiak, 2014-2018, 2020

__doc__ = """
Simple calculations in 3D space,
working on single points or on whole arrays.
Based on numpy functions, should work at a sensible pace.
Maciej Bartkowiak, 28 Mar 2014
"""

import numpy as np
import scipy.optimize as scopt
from itertools import permutations

zdir = np.array([0.0, 0.0, 1.0])
ydir = np.array([0.0, 1.0, 0.0])

def distance(point1, point2):
    """
    Calculates Carthesian distance between two points
    expressed as same-sized numpy arrays.
    """
    return np.sqrt(((point1-point2)*(point1-point2)).sum(1))

def length(vector):
    """
    Calculates the length of a vector (or n vectors) expressed
    in Carthesian coordinates as a numpy array.
    Returns an array of shape (n,1)
    """
    if len(vector.shape) > 1:
        return np.sqrt((vector*vector).sum(1))
    else:
        return np.sqrt((vector*vector).sum())

def normalise(vector):
    """
    Divides a vector, expressed in Carthesian coordinates,
    by its length, returning an unary vector expressing
    a direction.
    """
    return vector / length(vector)

def dist_coords(indices, base, point):
    result = indices[0]*base[0] + indices[1]*base[1] + indices[2]*base[2]
    return np.array((point - result))[0]
    
def find_coords(indices, base, point):
    newind = scopt.leastsq(dist_coords, indices, args = (base, point))[0]
    return newind

def rotY(degrees):
    angle = np.radians(degrees)
    rotmat = np.array([[ np.cos(angle), 0.0 , -np.sin(angle)],
                       [ 0.0, 1.0 , 0.0],
                       [ np.sin(angle), 0.0, np.cos(angle)]])
    return rotmat

def rotX(degrees):
    angle = np.radians(degrees)
    rotmat = np.array([[ 1.0, 0.0, 0.0],
                       [ 0.0, np.cos(angle),  np.sin(angle) ],
                       [ 0.0, -np.sin(angle), np.cos(angle)]])
    return rotmat

def rotZ(degrees):
    angle = np.radians(degrees)
    rotmat = np.array([[ np.cos(angle),  np.sin(angle), 0.0],
                       [ -np.sin(angle), np.cos(angle), 0.0],
                       [ 0.0, 0.0, 1.0]])
    return rotmat
    
def rotate(rotmat, arr):
    """
    A rotation matrix 'rotmat' of the form of a 3x3 array
    is used to mupliply a (x, 3) array 'arr' of coordinates.
    """
    return rotmat.dot(arr.T).T

def sph_to_cart(coords):
    """
    Converts a [:,3] array of spherical coordinates
    into a same-sized array of Cartesian coordinates.
    """
    xyz = coords.copy()
    r, theta, phi = coords[:, 0], np.radians(coords[:, 1]), np.radians(coords[:, 2])
    xyz[:,0] = r * np.sin(theta) * np.cos(phi)
    xyz[:,1] = r * np.sin(theta) * np.sin(phi)
    xyz[:,2] = r * np.cos(theta)
    return xyz
    
def cart_to_sph(coords):
    """
    Converts a [:,3] array of Cartesian coordinates
    into a same-sized array of spherical coordinates.
    """
    x, y, z = coords[:, 0], coords[:, 1], coords[:, 2]
    rtp = coords.copy()
    #rtp[:,0] = np.sqrt(x*x + y*y + z*z)
    #rtp[:,1] = np.arccos(z / rtp[:,0])
    #rtp[:,2] = np.arctan(y / x)
    #rtp[:,1:3] = np.nan_to_num(rtp[:,1:3])
    xy = x**2 + y**2
    rtp[:,0] = np.sqrt(xy + z**2)
    rtp[:,1] = np.arctan2(np.sqrt(xy), z)
    rtp[:,2] = np.arctan2(y, x)
    rtp[:,1:] = np.degrees(rtp[:,1:])
    rtp[:,1:3] = np.nan_to_num(rtp[:,1:3])
    return rtp

def arb_rotation(axis, theta):
    """
    Return the rotation matrix associated with counterclockwise rotation about
    the given axis by theta degrees.
    """
    axis = np.asarray(axis)
    theta = np.asarray(np.radians(theta))
    axis = axis/np.sqrt(np.dot(axis, axis))
    a = np.cos(theta/2.0)
    b, c, d = -axis*np.sin(theta/2.0)
    aa, bb, cc, dd = a*a, b*b, c*c, d*d
    bc, ad, ac, ab, bd, cd = b*c, a*d, a*c, a*b, b*d, c*d
    return np.array([[aa+bb-cc-dd, 2*(bc+ad), 2*(bd-ac)],
                     [2*(bc-ad), aa+cc-bb-dd, 2*(cd+ab)],
                     [2*(bd+ac), 2*(cd-ab), aa+dd-bb-cc]])
                     
def arb_rotation_rad(axis, theta):
    """
    Return the rotation matrix associated with counterclockwise rotation about
    the given axis by theta degrees.
    """
    axis = np.asarray(axis)
    theta = np.asarray(theta)
    axis = axis/np.sqrt(np.dot(axis, axis))
    a = np.cos(theta/2.0)
    b, c, d = -axis*np.sin(theta/2.0)
    aa, bb, cc, dd = a*a, b*b, c*c, d*d
    bc, ad, ac, ab, bd, cd = b*c, a*d, a*c, a*b, b*d, c*d
    return np.array([[aa+bb-cc-dd, 2*(bc+ad), 2*(bd-ac)],
                     [2*(bc-ad), aa+cc-bb-dd, 2*(cd+ab)],
                     [2*(bd+ac), 2*(cd-ab), aa+dd-bb-cc]])
 
def tubepoints(tube):
    """
    Converts the tube coordinates from Cartesian to a
    simplified spherical system, leaving out the R.
    For detector tube coverage, R depends only on lambda.
    """
    abpoints = np.zeros((len(tube), 3))
    # abpoints[:,1] = - (np.arctan(tube[:,0]/tube[:,2]) - np.pi * (tube[:,2] < 0.0))# x/z
    # abpoints[:,2] = np.arctan(tube[:,1]/np.sqrt(tube[:,0]**2+tube[:,2]**2)) # y/r
    # abpoints[:,1] = np.arctan2(tube[:,2], tube[:,0])
    abpoints[:,1] = np.arctan2(tube[:,2], tube[:,0])
    abpoints[:,2] = np.arctan2(tube[:,1], np.sqrt(tube[:,0]**2+tube[:,2]**2))
    return abpoints

#def reverse_q_vector(xyz):
    #q_z = (xyz[:,0]**2 + xyz[:,1]**2 + xyz[:,2]**2)/(-2.0*xyz[:,2])
    #kf = xyz.copy()
    ## print "xyz, ", xyz
    #kf[:,2] += q_z
    ## print "k_f, ", kf
    #rtp = kf.copy()
    #x, y, z = kf[:,0], kf[:,1], kf[:,2]
    #xy = x**2 + y**2
    #rtp[:,0] = np.sqrt(xy + z**2)
    #rtp[:,1] = np.arctan2(np.sqrt(xy), z)
    ## print "sqrt(xy), z, ", np.sqrt(xy), z
    ## print "atan1, ", rtp[:,1]
    #rtp[:,2] = np.arctan2(y, x)
    ## print "y, x, ", y, x
    ## print "atan2, ", rtp[:,2]
    #rtp[:,1:3] = np.nan_to_num(rtp[:,1:3])
    #return rtp

#def q_vector(coords):
    ## ki, alpha, beta = coords[:, 0], np.radians(coords[:, 1]), np.radians(coords[:, 2])
    #ki, alpha, beta = coords[:, 0], coords[:, 1], coords[:, 2]
    #xyz = coords.copy()
    #xyz[:,0] = ki * np.sin(alpha) * np.cos(beta)
    #xyz[:,1] = ki * np.sin(alpha) * np.sin(beta)
    #xyz[:,2] = ki * (np.cos(alpha) - 1.0)
    #return xyz

def q_vector(coords):
    # ki, alpha, beta = coords[:, 0], np.radians(coords[:, 1]), np.radians(coords[:, 2])
    ki, alpha, beta = coords[:, 0], coords[:, 1], coords[:, 2]
    xyz = coords.copy()
    xyz[:,0] = ki * np.cos(alpha) * np.cos(beta)
    xyz[:,1] = ki * np.sin(beta)
    xyz[:,2] = ki * (np.sin(alpha) * np.cos(beta) - 1.0)
    return xyz

def reverse_q_vector(xyz):
    k_f = xyz.copy()
    q_z = (xyz[:,0]**2 + xyz[:,1]**2 + xyz[:,2]**2)/(-2.0*xyz[:,2])
    k_f[:,2] += q_z
    coords = np.zeros(xyz.shape)
    coords[:,0] = np.sqrt(k_f[:,0]**2 + k_f[:,1]**2 + k_f[:,2]**2)
    coords[:,1] = np.arctan2(k_f[:,2], k_f[:,0])
    coords[:,2] = np.arctan2(k_f[:,1], np.sqrt(k_f[:,0]**2+k_f[:,2]**2))
    return coords

def old_intercept(vol, hkl, ubinv, edges):
    points = rotate(ubinv, vol)
    base = np.eye(3)
    h, k, l = hkl[0], hkl[1], hkl[2]
    # symbol = " ".join([str(x) for x in symbol])
    if h == "h" and k == "k":
        vec1, vec2 = base[0], base[1]
        normal = np.cross(vec1, vec2)
        origin = float(l) * base[2]
        ind = (0, 1)
        perp = 2
        symbol = 'hk'
    if h == "h" and l == "l":
        vec1, vec2 = base[0], base[2]
        normal = np.cross(vec1, vec2)
        origin = float(k) * base[1]
        ind = (0, 2)
        perp = 1
        symbol = 'hl'
    if l == "l" and k == "k":
        vec1, vec2 = base[1], base[2]
        normal = np.cross(vec1, vec2)
        origin = float(h) * base[0]
        ind = (1, 2)
        perp = 0
        symbol = 'kl'
    result = []
    for i in edges['alpha']:
        a, b = points[i[0]], points[i[1]]
        n1 = normalise(np.cross(a,b)) # /abs(np.dot(a,b))
        h1, h2 = a, origin
        nn = np.dot(n1, normal)
        c1 = (h1 - h2*nn)/(1.0-nn**2)
        c2 = (h2 - h1*nn)/(1.0-nn**2)
        #if a[perp] > b[perp]:
            #a,b = b,a
        #if origin[perp] > a[perp] and origin[perp] < b[perp]:
            #ratio = (origin[perp] - a[perp])/(b[perp] - a[perp])
            #point = a + (b-a)*ratio 
            #result.append(point)
    for i in edges['beta']:
        a, b = points[i[0]], points[i[1]]
        if a[perp] > b[perp]:
            a,b = b,a
        if origin[perp] > a[perp] and origin[perp] < b[perp]:
            ratio = (origin[perp] - a[perp])/(b[perp] - a[perp])
            point = a + (b-a)*ratio 
            result.append(point)
    for i in edges['ki']:
        a, b = points[i[0]], points[i[1]]
        if a[perp] > b[perp]:
            a,b = b,a
        if origin[perp] > a[perp] and origin[perp] < b[perp]:
            ratio = (origin[perp] - a[perp])/(b[perp] - a[perp])
            point = a + (b-a)*ratio 
            result.append(point)
    return result, ind

def contour(inp, segment = 6):
    cont, description, centra, radii = inp[0], np.array(inp[1]), np.array(inp[2]), np.array(inp[3])
    points, sort1, sort2 = simple_sort_points(np.row_stack(cont))
    in1 = sort1[0].astype(np.int)
    in2 = sort2[0].astype(np.int)
    in3 = sort1[1].astype(np.int)
    in4 = sort2[1].astype(np.int)
    try:
        description = np.concatenate((description[in1][in2], description[in3][in4]))
    except TypeError:
        #print description
        #print description[in1]
        #print description[in3]
        return None
    centra = np.row_stack((centra[in1][in2], centra[in3][in4]))
    radii = np.concatenate((radii[in1][in2], radii[in3][in4]))
    points = np.row_stack((points, points[0]))
    description = np.concatenate((description, description[0:1]))
    centra = np.row_stack((centra, centra[0]))
    radii = np.concatenate((radii, radii[0:1]))
    # now everything is sorted
    edge = []
    x, y = len(points), len(points[0])
    path = np.zeros((segment ,y ))
    for t in range(y):
        path[:,t] = np.arange(0,segment)/float(segment)
    if x > 2:
        for i in range(x):
            #if ((description[i] == "edge") and (description[(i+1)%x] == "edge")) or (not (description[(i+1)%x] == description[i])):
                #edge.append(points[i] + (points[(i+1)%x]-points[i])*path)
            #elif (description[i][0] == "s") and (description[(i+1)%x] == description[i]):
                #edge.append(arc_edge([points[i], points[(i+1)%x]], centra[i], radii[i]))
            edge.append(points[i] + (points[(i+1)%x]-points[i])*path)
        edge.append(points[0])
        edge = np.row_stack(edge)
        return edge

def sort_points(pts):
    """
    Takes an array of coplanar points
    and arranges them in a clockwise order.
    """
    if len(pts) < 4:
        return pts
    else:
        middle = pts.mean(0)
        points = pts - middle
        cross = np.cross(ydir, points).sum(1)
        return pts[np.argsort(cross)]

def simplest_sort_points(pts):
    if len(pts) < 4:
        return pts
    else:
        middle = pts.mean(0)
        points = pts - middle
        angles = np.arctan(points[:,0]/points[:,1]) - np.pi * (points[:,1] < 0.0)
        angles[np.where(np.isinf(angles))] = 0.0
        angles[np.where(np.isneginf(angles))] = np.pi
        inds = np.argsort(angles)
        return pts[inds]

def simple_sort_points(pts):
    if len(pts) < 4:
        return pts, [np.arange(len(pts)), np.array([])], [np.arange(len(pts)), np.array([])]
    else:
        middle = pts.mean(0)
        points = pts - middle
        sorter = points.max(0) - points.min(0)
        if length(sorter) == 0.0:
            sorter = np.array([1.0, 1.0, 1.0])
        # #print "Sort"
        # #print "Dot", np.dot(sorter, points.T).T
        # #print "lengths", length(points)*length(sorter)
        # #print "cos theta", np.dot(sorter, points.T).T / (length(points)*length(sorter))
        costheta = np.dot(sorter, points.T).T / (length(points)*length(sorter))
        costheta = np.minimum(costheta, 1.0)
        costheta = np.maximum(costheta, -1.0)
        angles = np.arccos(costheta)
        # #print "angles", angles
        direc = np.cross(sorter, points)
        # largest = direc[np.where(length(direc) == length(direc).max())]
        # halves = np.dot(direc[0], direc.T).T
        halves = np.dot(direc[np.where(length(direc) == length(direc).max())][0], direc.T).T
        sort1a = np.where(halves >= 0.0)
        half1 = pts[sort1a]
        angl1 = angles[sort1a]
        sort1b = np.where(halves < 0.0)
        half2 = pts[sort1b]
        angl2 = angles[sort1b]
        sort2 = []
        if len(half1) > 0:
            part1 = np.argsort(angl1)
            half1 = half1[part1]
            sort2.append(part1)
        else:
            sort2.append(np.array([]))
        # #print len(half2), len(angl2)
        # #print np.argsort(angl2)[::-1]
        if len(half2) > 0:
            part2 = np.argsort(angl2)[::-1]
            half2 = half2[part2]
            sort2.append(part2)
        else:
            sort2.append(np.array([]))
        #print "Sort1a: ", sort1a[0]
        #print "Sort1b: ", sort1b[0]
        #print "Sort2a: ", sort2[0]
        #print "Sort2b: ", sort2[1]
        return np.row_stack([half1, half2]), [sort1a[0], sort1b[0]], sort2
        
def contour_length(pts):
    temp = np.row_stack((pts, pts[0]))
    cont = temp[1:] - temp[:-1]
    result = np.sqrt((cont**2).sum(1)).sum()
    return result
        
def sort_contour(pts):
    """
    Takes an array of coplanar points
    and arranges them in a clockwise order.
    """
    if len(pts) < 4:
        return pts
    else:
        result = contour_length(pts)
        sequence = range(len(pts))
        for i in permutations(range(len(pts))):
            if i[0] == 0:
                newres = contour_length(pts[list(i)])
                if newres < result:
                    result = newres
                    sequence = list(i)
        return pts[sequence]

def intersection(points = np.array([[0,0,1.0],[1.0,0,0]]), plane = [np.array([1.0, 1.0, 1.0]), np.array([0.2, 0.2, 0.2])]):
    """
    The line is defined as a pair of points.
    The plane is defined as a pair of points too.
    The third point defining the plane is the origin (0,0,0)
    The output is a point, if the line intersects the plane,
    or nothing, if there is no intersection.
    """
    normal = normalise(np.cross(plane[0], plane[1]))
    origin = np.zeros(3)
    line_a, line_b = points[0], points[1]
    num = np.dot((origin - line_a), normal)
    denom = np.dot(line_b - line_a, normal)
    if not denom == 0.0:
        lam = num/denom
        pt = line_a + (line_b-line_a)*lam
        # print lam
        if (lam >= 0.0) and (lam <= 1.0):
            condition = (np.dot(pt, plane[0]) > 0.0) and (np.dot(pt, plane[1]) > 0.0)
            if condition:
                return True, pt
    return False, None

def shadowplane(points = np.array([[0,0,1.0],[1.0,0,0]])):
    normal = np.cross(points[0],points[1])/np.sqrt(np.dot(points[0], points[1]))
    return [normal, points[0]]

def center_distance_squared(points, axis = np.zeros(3)):
    nn = points[1] - points[0]
    n = normalise(np.cross(nn[:3], axis))
    distance = abs(np.dot(n, -points[0]))
    point = points[0][:3] + nn[:3]*(distance-points[0][3])/nn[3]
    return distance, point

def line_cone_intersection(points, axis = np.zeros(3), angle = 15.0):
    nn = points[1] - points[0]
    # n = normalise(np.cross(nn[:3], axis))
    A = np.tan(np.radians(angle))**2
    a1, a2, a3 = points[0][0], points[0][1], points[0][2]
    b1, b2, b3 = nn[0], nn[1], nn[2]
    a = b3**2 - (b1**2 + b2**2)/A
    b = 2.0*(a3*b3 - (a1*b1 + a2*b2)/A)
    c = a3**2 - (a1**2 + a2**2)/A
    if (b**2 - 4*a*c) < 0.0:
        return "No", "intersection"
    part = np.sqrt(b**2 - 4*a*c)
    sol1, sol2 = (-b + part) / (2*a), (-b - part) / (2*a)
    sol1 = (points[0] + nn*sol1)[:3]
    sol2 = (points[0] + nn*sol2)[:3]
    if np.dot(sol1 - points[0][:3], sol1 - points[1][:3]) < 0.0:
        return sol1, sol2
    else:
        return sol2, sol1
    if (sol1 >= points.min(0)).all() and (sol1 <= points.max(0)).all():
        if (sol2 >= points.min(0)).all() and (sol2 <= points.max(0)).all():
            if np.dot(sol1 - points[0][:3], sol1 - points[1][:3]) < 0.0:
                return sol1, sol2
            else:
                return sol2, sol1
        else:
            return sol1, "No"
    else:
        if (sol2 >= points.min(0)).all() and (sol2 <= points.max(0)).all():
            if np.dot(sol2 - points[0][:3], sol2 - points[1][:3]) < 0.0:
                return sol2, "No"
        else:
            return "No", "intersection"
    
def cone_intersection(points, axis, angle = 15.0, up = np.array([0.0, 1.0, 0.0])):
    """
    Points are the frame of a tube.
    The angle is the cone opening angle,
    from the axis to the outside.
    """
    split = 0
    rotax = np.cross(axis, zdir)
    if length(rotax) > 0.0:
        rotang = np.arccos(np.dot(axis, zdir)/length(axis))
        mat, invmat = arb_rotation_rad(rotax, rotang), arb_rotation_rad(rotax, -rotang)
        pts = simple_sort_points(points)[0]
        pts = rotate(mat, pts)
    else:
        mat, invmat = np.eye(3), np.eye(3)
        pts = simple_sort_points(points)[0]
    radii = np.abs(pts[:,2]*np.tan(np.radians(angle)))
    distance = pts[:,2].copy()
    pts = np.column_stack((pts,radii))
    segment = []
    aa = len(pts)
    keep, new, reject = [], [], []
    crit = length(pts[:,:2]) < pts[:,3]
    keep = [pts[np.where(crit)][:,:3]]
    reject = [pts[np.where(np.logical_not(crit))][:,:3]]
    for i in range(aa):
        segment.append(np.row_stack((pts[i], pts[(i+1)%aa])))
    for j in range(len(segment)):
        crit = length(segment[j][:,:2]) < segment[j][:,3]
        if crit.sum() == 1:
            inpoint = segment[j][np.where(crit)]
            outpoint = segment[j][np.where(np.logical_not(crit))]
            newpoint, nopoint = line_cone_intersection(segment[j], zdir, angle)
            if (newpoint[:3] <= pts[:,:3].max(0)).all() and (newpoint[:3] >= pts[:,:3].min(0)).all():
                new.append(newpoint[:3])
        elif crit.sum() == 0:
            new1, new2 = line_cone_intersection(segment[j], zdir, angle)
            if not (new1 == "No"):
                if (new1[:3] <= pts[:,:3].max(0)).all() and (new1[:3] >= pts[:,:3].min(0)).all():
                    if (new2[:3] <= pts[:,:3].max(0)).all() and (new2[:3] >= pts[:,:3].min(0)).all():
                        split += 1
                        new.append(np.row_stack((new1, new2))[:,:3])
    if split < 1:
        a = np.row_stack(keep + new)
        if (len(a)) > 2:
            frame = simple_sort_points(a)[0]
            frame = rotate(invmat, frame)
        else:
            frame = np.array([])
        a = np.row_stack(reject + new)
        if (len(a)) > 2:
            bad = simple_sort_points(a)[0]
            bad = rotate(invmat, bad)
        else:
            bad = np.array([])
        return [frame], [bad]
    else:
        # #print "split"
        a = np.row_stack(keep + new)
        if (len(a)) > 2:
            frame = simple_sort_points(a)[0]
            frame = rotate(invmat, frame)
        else:
            frame = np.array([])
        bads = []
        # #print new
        a = np.row_stack(reject + new)
        crit = np.dot(a, up) >= 0.0
        parts = [a[np.where(crit)], a[np.where(np.logical_not(crit))]]
        for i in range(len(parts)):
            if (len(parts[i])) > 2:
                parts[i] = simple_sort_points(parts[i])[0]
                parts[i] = rotate(invmat, parts[i])
                bads.append(parts[i])
        return [frame], bads

def intercept(vol, hkl, ubinv, krange):
    # points = rotate(ubinv, vol)
    base = np.eye(3)
    h, k, l = hkl[0], hkl[1], hkl[2]
    # symbol = " ".join([str(x) for x in symbol])
    if h == "h" and k == "k":
        vec1, vec2 = base[0], base[1]
        normal = np.cross(vec1, vec2)
        origin = float(l) * base[2]
        ind = (0, 1)
        perp = 2
        symbol = 'hk'
    if h == "h" and l == "l":
        vec1, vec2 = base[0], base[2]
        normal = np.cross(vec1, vec2)
        origin = float(k) * base[1]
        ind = (0, 2)
        perp = 1
        symbol = 'hl'
    if l == "l" and k == "k":
        vec1, vec2 = base[1], base[2]
        normal = np.cross(vec1, vec2)
        origin = float(h) * base[0]
        ind = (1, 2)
        perp = 0
        symbol = 'kl'
    result, description, centra, radii = [], [], [], []
    # here we use the UB-1 matrix to get the definition of the hkl plane in q coordinates
    # d = -origin[perp] # this may be wrong
    d = origin[perp]
    a, b, c = ubinv[perp,0], ubinv[perp,1], ubinv[perp,2]
    denom = a**2 + b**2 + c**2
    x0, y0, z0 = a*d/denom, b*d/denom, c*d/denom
    planen, plane0 = np.array([a, b, c]), np.array([x0, y0, z0])
    #print "plane normal", planen
    #print "plane origin", plane0
    planen = normalise(planen)
    # we have the plane definition, now we need the lines
    line_a, line_b = vol.copy(), vol.copy()
    line_a[:,0], line_b[:,0] = krange[0], krange[-1]
    #print "vol, ", vol
    #print "line_a, ", line_a
    #print "line_b, ", line_b
    line_a, line_b = q_vector(line_a), q_vector(line_b)
    #print "q_vector line_a, ", line_a
    #print "q_vector line_b, ", line_b
    for i in range(len(vol)):
        num = np.dot((plane0 - line_a[i]),planen)
        denom = np.dot(line_b[i] - line_a[i], planen)
        if not denom == 0.0:
            lam = num/denom
            # pt = line_a[i] + (line_b[i] - line_a[i])*lam
            # if abs(np.dot(pt - plane0, planen)) < 1e-12:
            if lam >= 0.0 and lam <= 1.0:
                result.append(line_a[i] + (line_b[i] - line_a[i])*lam)
                #print "Edge, between: ", line_a[i], line_b[i], result[-1]
                description.append("edge")
                centra.append(np.array([0.0, 0.0, 0.0]))
                radii.append(0.0)
        elif num == 0.0:
            result.append(line_a[i])
            #print "Edge, in-plane: ", line_a[i], line_b[i], result[-1]
            description.append("edge")
            centra.append(np.array([0.0, 0.0, 0.0]))
            radii.append(0.0)
            result.append(line_b[i])
            description.append("edge")
            centra.append(np.array([0.0, 0.0, 0.0]))
            radii.append(0.0)
    # those were the straight lines, now the spheres
    center = np.array([0.0, 0.0, -krange[0]])
    cast_center = center + np.dot(planen, plane0 - center)*planen
    limit = len(vol)
    for i in range(limit):
        pt1, pt2 = line_a[i], line_a[(i+1)%limit]
        vec1, vec2 = pt1 - center, pt2 - center
        ax = normalise(np.cross(vec1, vec2))
        vecs = plane_circ_intersection_between([planen, plane0], [ax, center], [vec1, vec2])
        if not (vecs == []):
            cast_radius = np.sqrt(np.dot(center, center) - np.dot(cast_center-center, cast_center-center))
            for glam in vecs:
                result.append(glam)
                description.append("s1")
                centra.append(cast_center)
                radii.append(cast_radius)
    # one radius finished
    center = np.array([0.0, 0.0, -krange[-1]])
    cast_center = center + np.dot(planen, plane0 - center)*planen
    limit = len(vol)
    for i in range(limit):
        pt1, pt2 = line_b[i], line_b[(i+1)%limit]
        vec1, vec2 = pt1 - center, pt2 - center
        ax = normalise(np.cross(vec1, vec2))
        vecs = plane_circ_intersection_between([planen, plane0], [ax, center], [vec1, vec2])
        if not (vecs == []):
            cast_radius = np.sqrt(np.dot(center, center) - np.dot(cast_center-center, cast_center-center))
            for glam in vecs:
                result.append(glam)
                description.append("s2")
                centra.append(cast_center)
                radii.append(cast_radius)
    # that should cover the basic intersection points
    #if len(result) > 0:
        #result = rotate(ubinv,np.row_stack(result))
    return [result, description, centra, radii], ind

def plane_circ_intersection(plane = [np.array((0.0, 0.0, 1.0)), np.array((1.0, 1.0, 1.0))],
            circ = [np.array((1.0, 0.0, 0.0)), np.array((2.0, 0.0, 0.0))]):
    planen, plane0 = normalise(plane[0]), plane[1]
    ax, center = normalise(circ[0]), circ[1]
    # cast_center = center + np.dot(planen, plane0 - center)*planen
    # #print cast_center
    result = []
    h1, h2 = np.dot(planen, plane0), np.dot(ax, center)
    c1 = (h1 - h2 * np.dot(planen, ax)) / (1- np.dot(planen, ax)**2)
    c2 = (h2 - h1 * np.dot(planen, ax)) / (1- np.dot(planen, ax)**2)
    n3, r = np.cross(planen, ax), length(center)
    # #print n3
    # #print c1*planen + c2*ax
    p1 = c1*planen + c2*ax - center
    # #print p1
    u = np.dot(n3, p1)
    v = np.dot(p1, p1)
    delta = 4*(u**2 -v + r**2)
    if delta >= 0.0:
        l1, l2 = -2*u + np.sqrt(delta)/2.0, -2*u - np.sqrt(delta)/2.0
        vec3, vec4 = p1 + l1*n3, p1 + l2*n3
        return vec3, vec4
    else:
        return None, None
        
def arc_edge(points, center, radius, segments = 6):
    pt1, pt2 = points[0], points[-1]
    vec1, vec2 = pt1 - center, pt2 - center
    ax = normalise(np.cross(vec1, vec2))
    angle = np.dot(vec1, vec2)/(length(vec1) * length(vec2))
    rots = np.arange(1.0, segments) / segments * angle
    out = [vec1]
    for i in rots:
        out.append(rotate(arb_rotation_rad(ax, i), vec1))
    out.append(vec2)
    return np.row_stack(out) + center
    
def plane_line_intersection(plane = [np.array((0.0, 0.0, 1.0)), np.array((1.0, 1.0, 1.0))],
            line = [np.array((1.0, 0.0, 0.0)), np.array((2.0, 0.0, 0.1))]):
    planen, plane0 = plane[0], plane[1]
    line_a, line_b = line[0], line[1]
    num = np.dot((plane0 - line_a),planen)
    denom = np.dot(line_b - line_a, planen)
    if not denom == 0.0:
        lam = num/denom
        pt = line_a + (line_b-line_a)*lam
        # print lam
        if lam > 0.0:
            return pt

def plane_circ_intersection_between(plane = [np.array((0.0, 0.0, 1.0)), np.array((1.0, 1.0, 1.0))],
            circ = [np.array((1.0, 0.0, 0.0)), np.array((2.0, 0.0, 0.0))],
            limits = [np.array((0.0, 1.0, 10.0)), np.array((0.0, 10.0, -1.0))]):
    planen, plane0 = normalise(plane[0]), plane[1]
    ax, center = normalise(circ[0]), circ[1]
    vec1, vec2 = normalise(limits[0]), normalise(limits[1])
    refangle = np.arccos(np.dot(vec1,vec2))
    cast_center = center + np.dot(planen, plane0 - center)*planen
    # print "cast_center", cast_center
    # cast_center = center + np.dot(planen, plane0 - center)*planen
    # #print cast_center
    result = []
    h1, h2 = np.dot(planen, plane0), np.dot(ax, center)
    c1 = (h1 - h2 * np.dot(planen, ax)) / (1- np.dot(planen, ax)**2)
    c2 = (h2 - h1 * np.dot(planen, ax)) / (1- np.dot(planen, ax)**2)
    n3, r = np.cross(planen, ax), length(center)
    # print n3, r
    # #print c1*planen + c2*ax
    p1 = c1*planen + c2*ax # - center
    # print p1
    u = np.dot(n3, p1)
    v = np.dot(p1, p1)
    delta = 4*(u**2 -v + r**2)
    if delta >= 0.0:
        l1, l2 = -2*u + np.sqrt(delta)/2.0, -2*u - np.sqrt(delta)/2.0
        vec3, vec4 = p1 + l1*n3, p1 + l2*n3
        # print vec3, vec4
        ref1, ref2 = normalise(vec3), normalise(vec4)
        cast_radius = np.sqrt(np.dot(center, center) - np.dot(cast_center-center, cast_center-center))
        # print "vec3: ", vec3, "\n pt3: ", vec3+center, "\n cross1: ", np.cross(vec1, vec3), "\n cross2: ", np.cross(vec2, vec3), "\n dot: ", np.dot(np.cross(vec1, vec3), np.cross(vec2, vec3))
        # print "vec4: ", vec4, "\n pt3: ", vec4+center, "\n cross1: ", np.cross(vec1, vec4), "\n cross2: ", np.cross(vec2, vec4), "\n dot: ", np.dot(np.cross(vec1, vec4), np.cross(vec2, vec4))
        if np.dot(np.cross(vec1, vec3), ax) > 0.0 and np.dot(np.cross(vec2, vec3), ax) < 0.0:
        #sum1 = np.arccos(np.dot(vec1, ref1)) + np.arccos(np.dot(vec2, ref1))
        #sum2 = np.arccos(np.dot(vec1, ref2)) + np.arccos(np.dot(vec2, ref2))
        #if sum1 <= 1.0001*refangle:
            result.append([vec3+center])
        if np.dot(np.cross(vec1, vec4), ax) > 0.0 and np.dot(np.cross(vec2, vec4), ax) < 0.0:
        #if sum2 <= 1.0001*refangle:
            result.append([vec4+center])
        return result
    else:
        return []
