#!/usr/bin/python

#    This file is part of EXEQ.
#
#    EXEQ is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <https://www.gnu.org/licenses/>.
# 
# Copyright (C) Maciej Bartkowiak, 2014-2018, 2020

__doc__ = """
One of the key component of the EXED (E,q)-range calculators,
this module contains the instrument definition.
It reads in a .PAR file which defines all the detector pixels,
and then calculates the accessible range based on the other settings,
such as the chopper speed and wavelength band.
Can be made more useful by mixing with the sample definition.
Maciej Bartkowiak, 20-Jan-2014.

NEW: now it reads the XML file, and creates a list of detector banks,
instead of a single bank.
"""

import numpy as np
import scipy.interpolate as scint
from file_io import *
from units import *
from geometry import *
import exed_parser as parser
import pickle as cp
import gzip
import sys
import os

def resource_path(relative_path):
    """ Get absolute path to resource, works for dev and for PyInstaller """
    try:
        # PyInstaller creates a temp folder and stores path in _MEIPASS
        base_path = sys._MEIPASS
    except Exception:
        base_path = os.path.abspath(".")

    return os.path.join(base_path, relative_path)

edges = {'alpha': [[1,2], [5,6], [3,0], [7,4]], 
         'beta' : [[0,1], [4,5], [2,3], [6,7]],
         'ki'   : [[0,4], [1,5], [2,6], [3,7]]}

class Exed:
    def __init__(self, inst = []):
        parfiles_fw = [('forward1_nocryo.par', 'EXED_FWpanels_', 'degrees.xml', False, 1),
                        ('forward1_cryo.par', 'EXED_FWpanels_', 'degrees.xml', True, 1),
                        ('forward3_nocryo.par', 'EXED_FWpanels_', 'degrees.xml', False, 3),
                        ('forward3_cryo.par', 'EXED_FWpanels_', 'degrees.xml', True, 3),
                        ('forward5_nocryo.par', 'EXED_FWpanels_', 'degrees.xml', False, 5),
                        ('forward5_cryo.par', 'EXED_FWpanels_', 'degrees.xml', True, 5),
                        ('forward7_nocryo.par', 'EXED_FWpanels_', 'degrees.xml', False, 7),
                        ('forward7_cryo.par', 'EXED_FWpanels_', 'degrees.xml', True, 7)
                        ]
        parfiles_bw = [('backward2right_nocryo.par','EXED_BWpanels_middle.xml',False, 2),
                        ('backward2right_cryo.par','EXED_BWpanels_middle.xml',True, 2),
                        ('backward4_nocryo.par','EXED_BWpanels_middle.xml',False, 4),
                        ('backward4_cryo.par','EXED_BWpanels_middle.xml',True, 4),
                        ('backward2left_nocryo.par','EXED_BWpanels_right.xml',False, 2),
                        ('backward2left_cryo.par','EXED_BWpanels_right.xml',True, 2),
                        ]
        parfiles_SANS = [('SANS1_nocryo.par', 'EXED_SANSpanels.xml', False, 1),
                        ('SANS1_cryo.par', 'EXED_SANSpanels.xml', True, 1),
                        ('SANS3_nocryo.par', 'EXED_SANSpanels.xml', False, 3),
                        ('SANS3_cryo.par', 'EXED_SANSpanels.xml', True, 3)
                        ]
        if inst == []:
            self.spectrum_shortguide = self.spectrum_read("2slit_lambdann.dat")
            self.spectrum_longguide = self.spectrum_read("2slit_lambda.dat")
            self.spectrum_SANSguide = self.spectrum_read("sans_lambda.dat")
            self.det1_cryo = {}
            self.det1_nocryo = {}
            self.det2right_cryo = {}
            self.det2right_nocryo = {}
            self.det2left_cryo = {}
            self.det2left_nocryo = {}
            self.det3_cryo = {}
            self.det3_nocryo = {}
            self.det5_cryo = {}
            self.det5_nocryo = {}
            self.det7_cryo = {}
            self.det7_nocryo = {}
            self.det4_cryo = {}
            self.det4_nocryo = {}
            self.SANS1_cryo = {}
            self.SANS1_nocryo = {}
            self.SANS3_cryo = {}
            self.SANS3_nocryo = {}
            self.tubes = []
            self.widetubes = []
            self.minangle = -15
            self.segments = 10
            dics = [self.det1_nocryo , self.det1_cryo, self.det3_nocryo, self.det3_cryo, self.det5_nocryo , self.det5_cryo, self.det7_nocryo, self.det7_cryo,
                    self.det2right_nocryo, self.det2right_cryo, self.det4_nocryo, self.det4_cryo,
                    self.det2left_nocryo, self.det2left_cryo,
                    self.SANS1_cryo, self.SANS1_nocryo, self.SANS3_cryo, self.SANS3_nocryo]
            for nnn, zzz in enumerate(parfiles_fw):
                fname = zzz[0]
                dic = dics[nnn]
                try:
                    target = gzip.GzipFile(fname,'rb')
                    num = cp.load(target)
                    target.close()
                except IOError:
                    for i in range(self.minangle,16):
                        temp = parser.NewPanels(resource_path(zzz[1] + str(i) + zzz[2]), -i,heavy_cryostat = zzz[3], whichpanel=zzz[4])
                        dic[-i] = temp.get_panel()
                        # self.dets.append(parser.NewPanels("mock_exed.xml", i,heavy_cryostat = True))
                        #a, b, c = self.dets[i-self.minangle].tubes_for_plotting()
                        #self.tubes.append((a, b, c))
                        #aa, bb, cc = self.dets[i-self.minangle].tubes_for_exact()
                        #self.widetubes.append((aa, bb, cc))
                    self.dump_data(fname, dic)
                    dics[nnn] = dic
                    print( "Written a PAR file: ", fname)
                else:
                    dics[nnn] = self.load_data(num)
                    print( "Detector definition loaded from PAR file: ", fname)
                    print( "If you need to regenerate the detector definition, please delete this PAR file")
            for nnn, zzz in enumerate(parfiles_bw):
                fname = zzz[0]
                dic = dics[nnn+8]
                try:
                    target = gzip.GzipFile(fname,'rb')
                    num = cp.load(target)
                    target.close()
                except IOError:
                    for i in range(self.minangle,16):
                        temp = parser.NewPanels(resource_path(zzz[1]), -i,heavy_cryostat = zzz[2], whichpanel=zzz[3])
                        dic[-i] = temp.get_panel()
                        # self.dets.append(parser.NewPanels("mock_exed.xml", i,heavy_cryostat = True))
                        #a, b, c = self.dets[i-self.minangle].tubes_for_plotting()
                        #self.tubes.append((a, b, c))
                        #aa, bb, cc = self.dets[i-self.minangle].tubes_for_exact()
                        #self.widetubes.append((aa, bb, cc))
                    self.dump_data(fname, dic)
                    dics[nnn+8] = dic
                    print( "Written a PAR file: ", fname)
                else:
                    dics[nnn+8] = self.load_data(num)
                    print( "Detector definition loaded from PAR file: ", fname)
                    print( "If you need to regenerate the detector definition, please delete this PAR file")
            for nnn, zzz in enumerate(parfiles_SANS):
                fname = zzz[0]
                dic = dics[nnn+14]
                try:
                    target = gzip.GzipFile(fname,'rb')
                    num = cp.load(target)
                    target.close()
                except IOError:
                    for i in range(self.minangle,16):
                        temp = parser.NewPanels(resource_path(zzz[1]), -i,heavy_cryostat = zzz[2], whichpanel=zzz[3], SANS = True)
                        dic[-i] = temp.get_panel()
                        # self.dets.append(parser.NewPanels("mock_exed.xml", i,heavy_cryostat = True))
                        #a, b, c = self.dets[i-self.minangle].tubes_for_plotting()
                        #self.tubes.append((a, b, c))
                        #aa, bb, cc = self.dets[i-self.minangle].tubes_for_exact()
                        #self.widetubes.append((aa, bb, cc))
                    self.dump_data(fname, dic)
                    dics[nnn+14] = dic
                    print( "Written a PAR file: ", fname)
                else:
                    dics[nnn+14] = self.load_data(num)
                    print( "Detector definition loaded from PAR file: ", fname)
                    print( "If you need to regenerate the detector definition, please delete this PAR file")
            [self.det1_nocryo , self.det1_cryo, self.det3_nocryo, self.det3_cryo, self.det5_nocryo , self.det5_cryo, self.det7_nocryo, self.det7_cryo,
                    self.det2right_nocryo, self.det2right_cryo, self.det4_nocryo, self.det4_cryo,
                    self.det2left_nocryo, self.det2left_cryo,
                    self.SANS1_nocryo, self.SANS1_cryo, self.SANS3_nocryo, self.SANS3_cryo] = dics
            self.fmins, self.bmins, self.fmaxs, self.bmaxs = [], [], [], []
            self.chopper = {"distance":2.4, "frequency":7200.0/60.0, "suppression":2.0}
            self.current_spectrum = []
            self.current_panels = []
        else:
            if inst['Guide'] == 'Short':
                spec = self.spectrum_read("2slit_lambdann.dat")
                # if abs(angle) > 5:
                #     dets = [], [], [], []
            elif inst['Guide'] == 'SANS':
                spec = self.spectrum_read("sans_lambda.dat")
            else:
                spec = self.spectrum_read("2slit_lambda.dat")
                # if abs(angle) > 10:
                #     dets = [], [], [], []
            if inst['Cryostat'] == 'Front':
                self.angle_limit = (-14.0, 2.5)
            elif inst['Cryostat'] == 'None':
                self.angle_limit = (-14.0, 2.5)
            else:
                if inst['Guide'] == 'Short':
                    self.angle_limit = (-12.0, 2.5)
                elif inst['Guide'] == 'SANS':
                    self.angle_limit = (-7.0, 2.5)
                else:
                    self.angle_limit = (-12.0, 2.5)
            self.current_spectrum = spec
            while len(active_list) < 6:
                active_list.append(True)
            det1, det2, det3, det4, det5, det7 = [], [], [], [], [], []
            if inst['Detectors']['Active'][0]:
                if inst['Detectors']['Forward'] == 'Fixed':
                    if inst['Cryostat'] == 'Front':
                        fname = 'forward1_cryo.par'
                        target = gzip.GzipFile(fname,'rb')
                        num = cp.load(target)
                        target.close()
                        det1 = self.load_data(num)
                    else:
                        fname = 'forward1_nocryo.par'
                        target = gzip.GzipFile(fname,'rb')
                        num = cp.load(target)
                        target.close()
                        det1 = self.load_data(num)
                else:
                    if inst['Cryostat'] == 'Front':
                        fname = 'SANS1_cryo.par'
                        target = gzip.GzipFile(fname,'rb')
                        num = cp.load(target)
                        target.close()
                        det1 = self.load_data(num)
                    else:
                        fname = 'SANS1_nocryo.par'
                        target = gzip.GzipFile(fname,'rb')
                        num = cp.load(target)
                        target.close()
                        det1 = self.load_data(num)
            if inst['Detectors']['Active'][2]:
                if inst['Detectors']['Forward'] == 'Fixed':
                    if inst['Cryostat'] == 'Front':
                        fname = 'forward3_cryo.par'
                        target = gzip.GzipFile(fname,'rb')
                        num = cp.load(target)
                        target.close()
                        det3 = self.load_data(num)
                    else:
                        fname = 'forward3_nocryo.par'
                        target = gzip.GzipFile(fname,'rb')
                        num = cp.load(target)
                        target.close()
                        det3 = self.load_data(num)
                else:
                    if inst['Cryostat'] == 'Front':
                        fname = 'SANS3_cryo.par'
                        target = gzip.GzipFile(fname,'rb')
                        num = cp.load(target)
                        target.close()
                        det3 = self.load_data(num)
                    else:
                        fname = 'SANS3_nocryo.par'
                        target = gzip.GzipFile(fname,'rb')
                        num = cp.load(target)
                        target.close()
                        det3 = self.load_data(num)
            if inst['Detectors']['Active'][1]:
                if inst['Detectors']['Backward'] == 'Symmetrical':
                    if inst['Cryostat'] == 'Back':
                        fname = 'backward2right_cryo.par'
                        target = gzip.GzipFile(fname,'rb')
                        num = cp.load(target)
                        target.close()
                        det2 = self.load_data(num)
                    else:
                        fname = 'backward2right_nocryo.par'
                        target = gzip.GzipFile(fname,'rb')
                        num = cp.load(target)
                        target.close()
                        det2 = self.load_data(num)
                else:
                    if inst['Cryostat'] == 'Back':
                        fname = 'backward2left_cryo.par'
                        target = gzip.GzipFile(fname,'rb')
                        num = cp.load(target)
                        target.close()
                        det2 = self.load_data(num)
                    else:
                        fname = 'backward2left_nocryo.par'
                        target = gzip.GzipFile(fname,'rb')
                        num = cp.load(target)
                        target.close()
                        det2 = self.load_data(num)
            if inst['Detectors']['Active'][3]:
                if inst['Cryostat'] == 'Back':
                    fname = 'backward4_cryo.par'
                    target = gzip.GzipFile(fname,'rb')
                    num = cp.load(target)
                    target.close()
                    det4 = self.load_data(num)
                else:
                    fname = 'backward4_nocryo.par'
                    target = gzip.GzipFile(fname,'rb')
                    num = cp.load(target)
                    target.close()
                    det4 = self.load_data(num)
            if inst['Detectors']['Active'][4]:
                if inst['Detectors']['Forward'] == 'Fixed':
                    if inst['Cryostat'] == 'Front':
                        fname = 'forward5_cryo.par'
                        target = gzip.GzipFile(fname,'rb')
                        num = cp.load(target)
                        target.close()
                        det5 = self.load_data(num)
                    else:
                        fname = 'forward5_nocryo.par'
                        target = gzip.GzipFile(fname,'rb')
                        num = cp.load(target)
                        target.close()
                        det5 = self.load_data(num)
            if inst['Detectors']['Active'][5]:
                if inst['Detectors']['Forward'] == 'Fixed':
                    if inst['Cryostat'] == 'Front':
                        fname = 'forward7_cryo.par'
                        target = gzip.GzipFile(fname,'rb')
                        num = cp.load(target)
                        target.close()
                        det7 = self.load_data(num)
                    else:
                        fname = 'forward7_nocryo.par'
                        target = gzip.GzipFile(fname,'rb')
                        num = cp.load(target)
                        target.close()
                        det7 = self.load_data(num)
            self.current_panels = [det1, det2, det3, det4, det5, det7]
    def dump_data(self, fname = 'exed.par', dictionary = {}):
        data = {}
        data['dets'] = []
        data['numbers'] = []
        # data['tubes'] = self.tubes
        # data['widetubes'] = self.widetubes
        for ki in dictionary.keys():
            i = dictionary[ki]
            data['dets'].append(i.dump_data())
            data['numbers'].append(ki)
        target = gzip.GzipFile(fname,'wb')
        cp.dump(data,target)
        target.close()
    def load_data(self, data):
        dictionary = {}
        # self.tubes = data['tubes']
        # self.widetubes = data['widetubes']
        for n, i in enumerate(data['dets']):
            # dictionary[data['numbers'][n]] = parser.NewPanels(0,0,0,0,input_data = i)
            dictionary[data['numbers'][n]] = parser.Panel(0,0,input_data = i)
        return dictionary
    def get_right_panels(self, param, inst):
        angle = int(param['Field']['Direction'])
        active_list = inst['Detectors']['Active']
        while len(active_list) < 6:
            active_list.append(True)
        det1, det2, det3, det4, det5, det7 = [], [], [], [], [], []
        if active_list[0]:
            if inst['Detectors']['Forward'] == 'Fixed':
                if inst['Cryostat'] == 'Front':
                    det1 = self.det1_cryo
                else:
                    det1 = self.det1_nocryo
            else:
                if inst['Cryostat'] == 'Front':
                    det1 = self.SANS1_cryo
                else:
                    det1 = self.SANS1_nocryo
        if active_list[2]:
            if inst['Detectors']['Forward'] == 'Fixed':
                if inst['Cryostat'] == 'Front':
                    det3 = self.det3_cryo
                else:
                    det3 = self.det3_nocryo
            else:
                if inst['Cryostat'] == 'Front':
                    det3 = self.SANS3_cryo
                else:
                    det3 = self.SANS3_nocryo
        if active_list[1]:
            if inst['Detectors']['Backward'] == 'Symmetrical':
                if inst['Cryostat'] == 'Back':
                    det2 = self.det2right_cryo
                else:
                    det2 = self.det2right_nocryo
            else:
                if inst['Cryostat'] == 'Back':
                    det2 = self.det2left_cryo
                else:
                    det2 = self.det2left_nocryo
        if active_list[3]:
            if inst['Cryostat'] == 'Back':
                det4 = self.det4_cryo
            else:
                det4 = self.det4_nocryo
        if active_list[4]:
            if inst['Detectors']['Forward'] == 'Fixed':
                if inst['Cryostat'] == 'Front':
                    det5 = self.det5_cryo
                else:
                    det5 = self.det5_nocryo
        if active_list[5]:
            if inst['Detectors']['Forward'] == 'Fixed':
                if inst['Cryostat'] == 'Front':
                    det7 = self.det7_cryo
                else:
                    det7 = self.det7_nocryo
        return det1, det2, det3, det4, det5, det7
    def configure(self, param, inst):
        dets = self.get_right_panels(param, inst)
        angle = param['Field']['Direction']
        if inst['Guide'] == 'Short':
            spec = self.spectrum_shortguide
            # if abs(angle) > 5:
            #     dets = [], [], [], []
        elif inst['Guide'] == 'SANS':
            spec = self.spectrum_SANSguide
        else:
            spec = self.spectrum_longguide
            # if abs(angle) > 10:
            #     dets = [], [], [], []
        if inst['Cryostat'] == 'Front':
            self.angle_limit = (-14.0, 2.5)
        elif inst['Cryostat'] == 'None':
            self.angle_limit = (-14.0, 2.5)
        else:
            if inst['Guide'] == 'Short':
                self.angle_limit = (-12.0, 2.5)
            elif inst['Guide'] == 'SANS':
                self.angle_limit = (-7, 2.5)
            else:
                self.angle_limit = (-12.0, 2.5)
        self.current_spectrum = spec
        self.current_panels = dets
    def current_panels_at(self, inputangle):
        output = []
        angle = int(round(inputangle))
        for i in self.current_panels:
            if not i == []:
                output.append(i[angle])
            else:
                output.append([])
        return output
    def spectrum_read(self, fname = "2slit_lambda.dat"):
        """
        Reads in a wavelength spectrum of the instrument for future reference.
        At the moment we use a Vitess simulation result.
        """
        data = read_numbers(resource_path(fname))
        data[:,1] /= 1.3**2
        spec = scint.interp1d(data[:,0], data[:,1])
        return spec
    def check_wavelength(self, nlambdaa, panels = None):
        """
        Outputs a predicted range of dE accessible with a chosen lambda.
        Good to see if it is worth to recalculate the whole q-range.
        Returns the longest wavelengths one can observe in
        the front and back panels (returns 2 numbers)
        """
        if panels is None:
            panels = [self.fmax, self.bmax]
        nlambda = min(max(nlambdaa, 0.7), 49.0)
        fdist = panels[0]
        bdist = panels[1]
        tof = self.chopper["suppression"] / self.chopper["frequency"]
        vel_init = n_v(nlambda)
        tleft = tof - self.chopper["distance"] / vel_init
        flimit = n_lambda_from_v(fdist / tleft)
        blimit = n_lambda_from_v(bdist / tleft)
        return flimit, blimit
    def eq_map(self, ei = 20.0, de = 0.0, front = True, orient = None):
        """
        Calculates the q range for the input Ei and dE pair.
        Uses the current instrument settings.
        """
        if (orient >= -15.0) and (orient <= 15.0):
            ind = int(np.round(orient))
            objects = self.dets[ind-self.minangle]
        if front:
            full, trim, dark = objects.tubes_for_mapping(forward = True, segmentation = self.segments)
            div1, div2 = len(ei)/3, 2*len(ei)/3
            # energy_segments = [ei[0:div1+1], ei[div1:div2+1], ei[div2:-1]]
            energy_segments = [ei]
        else:
            full, trim, dark = objects.tubes_for_mapping(forward = False, segmentation = self.segments)
            energy_segments = [ei]
        # ef = ei + de
        ofull, otrim, odark = [], [], []
        for seg in energy_segments:
            # s is an array
            #if seg[0] < seg[-1]:
                #s = np.concatenate([[0.5*seg[0]], [0.999*seg[0]], seg, [1.001*seg[-1]], [1.1*seg[-1]]])
            #else:
                #s = np.concatenate([[1.1*seg[0]], [1.001*seg[0]], seg, [0.999*seg[-1]], [0.5*seg[-1]]])
            #if seg[0] < seg[-1]:
                #s = np.concatenate([[0.5*seg[0]], [0.999*seg[0]], seg])
            #else:
                #s = np.concatenate([seg, [0.999*seg[-1]], [0.5*seg[-1]]])
            # s = seg
            #ef = s + de
            #lambda_i = n_lambda(s)
            ef = seg + de
            lambda_i = n_lambda(seg)
            # lambda_i = np.concatenate([[0.99*lambda_i.min()], lambda_i, [1.01*lambda_i.max()]])
            x = len(lambda_i)
            k_i = np.zeros((x,3))
            k_i[:,2] = n_k(lambda_i)
            lambda_f = n_lambda(ef)
            k_f = n_k(lambda_f)
            direction = np.array([0.0,0.0,1.0])
            intensity = self.spectrum(lambda_i)
            #if seg[0] < seg[-1]:
                #intensity[0], intensity[1] = 0.0, 0.0
            #else:
                #intensity[-2], intensity[-1] = 0.0, 0.0
            # intensity[0], intensity[1], intensity[-2], intensity[-1] = 0.0, 0.0, 0.0, 0.0
            for n in full:
                temp = []
                for i in range(len(k_i)):
                    #temp.append(n * k_f[i] - k_i[i])
                    temp.append((n-direction) * k_f[i])
                ofull.append((temp, intensity))
            for n in trim:
                temp = []
                for i in range(len(k_i)):
                    #temp.append(n * k_f[i] - k_i[i])
                    temp.append((n-direction) * k_f[i])
                otrim.append((temp, intensity))
            for n in dark:
                temp = []
                for i in range(len(k_i)):
                    #temp.append(n * k_f[i] - k_i[i])
                    temp.append((n-direction) * k_f[i])
                odark.append((temp, intensity))
        return ofull, otrim, odark
    def absolute_range(self, ei = 20.0, magf = 0.0):
        """
        Shows a 2D-plot of the dE(|q|) that can be reached with
        the current instrument settings, assuming the
        initial neutron energy ei.
        """
        if (magf >= 0.0) and (magf <= 15.0):
            ind = int(np.round(magf))
            fxyz = self.fmaxs[ind]
            bxyz = self.bmaxs[ind]
            flambda, blambda = self.check_wavelength(n_lambda(ei), [fxyz, bxyz])
        else:
            flambda, blambda = self.check_wavelength(n_lambda(ei))
        fei, bei = n_energy(flambda), n_energy(blambda)
        front, back, fde, bde = [], [], [], []
        frange = np.arange(ei, fei - 0.001, (fei - ei)/100.0)
        brange = np.arange(ei, bei - 0.001, (bei - ei)/100.0)
        for i in frange:
            de = i - ei
            fmap, iii = self.eq_map(ei, de, orient = magf)
            front.append(length(np.row_stack(fmap)))
            fde.append(-de)
        for i in brange:
            de = i - ei
            bmap, iii = self.eq_map(ei, de, False, orient = magf)
            back.append(length(np.row_stack(bmap)))
            bde.append(-de)
        ff, fb = [], []
        for i in range(len(frange)):
            ff.append([front[i].min(), fde[i]])
            fb.append([front[i].max(), fde[i]])
        ff, fb = np.array(ff), np.array(fb)
        ff = ff[np.argsort(ff[:,1])]
        fb = fb[np.argsort(fb[:,1])]
        front = np.row_stack([ff, fb[::-1], ff[0]])
        ff, fb = [], []
        for i in range(len(brange)):
            ff.append([back[i].min(), bde[i]])
            fb.append([back[i].max(), bde[i]])
        ff, fb = np.array(ff), np.array(fb)
        ff = ff[np.argsort(ff[:,1])]
        fb = fb[np.argsort(fb[:,1])]
        back = np.row_stack([ff, fb[::-1], ff[0]])
        return front, back
    def elastic3D(self, br = 0.7, er = 15.0, ori = 0.0):
        """
        Maps the q-range for the specified wavelength range,
        assuming elastic scattering only.
        """
        lambdas = np.linspace(br**4, er**4, 8) # was 20 steps before
        lambdas = er + br - np.sqrt(np.sqrt(lambdas))
        eis = n_energy(lambdas)
#        front, back, intensity = [], [], []
        #for i in range(len(lambdas)):
##            tf, tb, intens = self.eq_map(eis[i], 0.0)
            #e, iii = self.eq_map(eis[i], 0.0, True, orient = ori)
            #front.append(e)
            #intensity.append(iii)
            #e, iii = self.eq_map(eis[i], 0.0, False, orient = ori)
            #back.append(e)
        a, b, c = self.eq_map(eis, 0.0, True, orient = ori)
        forward = [a,b,c]
        e, f, g = self.eq_map(eis, 0.0, False, orient = ori)
        backward = [e,f,g]
        return forward, backward
    def inelastic(self, ei = 20.0, ori = 0.0):
        """
        Maps the (E, q) range for the specified wavelength range.
        The range of energy transfer is determined by the chopper
        settings.
        """
        if ori == 0.0:
            flambda, blambda = self.check_wavelength(n_lambda(ei), panels = [self.fmax0, self.bmax0])
        elif ori == 15.0:
            flambda, blambda = self.check_wavelength(n_lambda(ei), panels = [self.fmax15, self.bmax15])
        fei, bei = n_energy(flambda), n_energy(blambda)
        front, back, fde, bde = [], [], [], []
        frange = np.arange(ei, fei, (fei - ei)/10.0)
        brange = np.arange(ei, bei, (bei - ei)/10.0)
        for i in frange:
            de = ei - i
            fmap, iii = self.eq_map(ei, de, orient = ori)
            front.append(fmap)
            fde.append(de)
        for i in brange:
            de = ei - i
            bmap, iii = self.eq_map(ei, de, False, orient = ori)
            back.append(bmap)
            bde.append(de)
        return front, back, fde, bde
    def new_elastic_coverage(self, param, samp, rfactor = 1.0):
        """
        First attempt to map the detector coverage EXACTLY,
        tube after tube.
        """
        br, er = param["Lambda"]["Min"], param["Lambda"]["Max"]
        ori = param["Field"]["Direction"]
        hkl = [param["hkl"]["h"], param["hkl"]["k"], param["hkl"]["l"]]
        # lambdas = np.linspace(br**4, er**4, 8) # was 20 steps before
        # lambdas = er + br - np.sqrt(np.sqrt(lambdas))
        # eis = n_energy(lambdas)
        tubes, goodtubes, shadowtubes = self.widetubes[int(round(ori))-self.minangle]
        # samp = Sample(param["Sample"]["Constants"], param["Sample"]["Angles"])
        ubinv = np.array(np.linalg.inv(np.matrix(samp.rotmat)*np.matrix(samp.B)))
        ub = np.array(np.matrix(samp.rotmat)*np.matrix(samp.B))
        print( "UB-1", "\n", ubinv)
        print( "UB", "\n", ub)
        # x = len(lambdas)
        # k_i = np.zeros((x,3))
        # k_i[:,2] = n_k(lambdas)
        # k_f = n_k(lambdas)
        # direction = np.array([0.0,0.0,1.0])
        tubes1, tubes2 = [], []
        for i in goodtubes:
            try:
                if i[0][-1] > 0.0:
                    tubes1.append(i)
                else:
                    tubes2.append(i)
            except IndexError:
                continue
        # print( tubes1
        # print( tubes2
        contours1, contours2 = [], []
        for t in tubes1:
            temp = tubepoints(np.array(t))
            #temp[:,0] = n_k(br)
            #vol = [q_vector(temp)]
            #temp[:,0] = n_k(er)
            #vol += [q_vector(temp)]
            #vol = np.row_stack(vol)
            points, inds = intercept(temp, hkl, ubinv, n_k(np.array([br, er])))
            if len(points[0]) > 2:
                # c = contour(points)
                # ints = self.spectrum(1.0/(reverse_q_vector(rotate(ub,c))[:,0])) * rfactor
                c = contour(points)
                ints = self.spectrum(1.0/(reverse_q_vector(c)[:,0])) * rfactor
                c = rotate(ubinv, c)
                contours1.append(np.column_stack((c[:, inds[0]], c[:,inds[1]], ints)))
        for t in tubes2:
            temp = tubepoints(np.array(t))
            # temp[:,0] = n_k(br)
            # vol = [q_vector(temp)]
            # temp[:,0] = n_k(er)
            # vol += [q_vector(temp)]
            # vol = np.row_stack(vol)
            points, inds = intercept(temp, hkl, ubinv, n_k(np.array([br, er])))
            if len(points[0]) > 2:
                c = contour(points)
                ints = self.spectrum(1.0/(reverse_q_vector(c)[:,0])) * rfactor
                c = rotate(ubinv, c)
                contours2.append(np.column_stack((c[:, inds[0]], c[:,inds[1]], ints)))
        return [contours1, contours2]

        
