#!/usr/bin/python

#    This file is part of EXEQ.
#
#    EXEQ is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <https://www.gnu.org/licenses/>.
# 
# Copyright (C) Maciej Bartkowiak, 2014-2018, 2020

__doc__ = """
An attempt to separate the input validation from the interface.
Also, low-level caching of the results should happen here.
Maciej Bartkowiak, 20 Jul 2014
"""

from geometry import *
from units import *
from sample import Sample
from instrument import Exed
from pyexe import parse_input
from file_io import parse_input_2018
import plot
# import plot_agg as plot
import copy
import yaml

params = {}
params["Choppers"] = {}
params["Sample"] = {}
params["Field"] = {}
params["Lambda"] = {}
params["hkl"] = {}

limits = {}
limits["Choppers"] = {}
limits["Sample"] = {}
limits["Field"] = {}
limits["Lambda"] = {}
limits["hkl"] = {}

instrument = {}
instrument['Detectors'] = {}

params["Field"]["Direction"] = 0.0
params["Choppers"]["Frequency"] = 120
params["Choppers"]["Suppression"] = 1
params["Lambda"]["Res"] = ""
params["Lambda"]["Min"] = 0.7
params["Lambda"]["Max"] = 4.0
params["Sample"]["Constants"] = np.array([5.0, 5.0, 5.0])
params["Sample"]["Angles"] = np.array(3*[90.0])
# params["Sample"]["u"] = np.array([-0.136,0.046,0.989])
# params["Sample"]["v"] = np.array([-0.929,0.367,-0.033])
params["Sample"]["u"] = np.array([0.0, 0.0, 1.0])
params["Sample"]["v"] = np.array([1.0, 0.0, 0.0])
params["Sample"]["phi"] = 0
params["Sample"]["chi"] = 0
params["Sample"]["omega"] = 0
params["hkl"]["h"] = 'h'
params["hkl"]["k"] = '0'
params["hkl"]["l"] = 'l'

limits["Field"]["Direction"] = (-15.0, 15.0)
limits["Choppers"]["Frequency"] = (5.0, 240.0)
limits["Choppers"]["Suppression"] = (1, 8)
limits["Lambda"]["Res"] = (1e-6, 1e3)
limits["Lambda"]["Min"] = (0.7, 48.9)
limits["Lambda"]["Max"] = (0.701, 49.0)
limits["Sample"]["Constants"] = (np.array([0.01, 0.01, 0.01]), np.array([40000.4, 40000.4, 40000.4]))
limits["Sample"]["Angles"] = (np.array(3*[0.01]), np.array(3*[179.99]))
limits["Sample"]["u"] = (np.array([-100.0, -100.0, -100.0]), np.array([100.0, 100.0, 100.0]))
limits["Sample"]["v"] = (np.array([-100.0, -100.0, -100.0]), np.array([100.0, 100.0, 100.0]))
limits["Sample"]["phi"] = (-360,360)
limits["Sample"]["chi"] = (-360,360)
limits["Sample"]["omega"] = (-360,360)
limits["hkl"]["h"] = (-10000, 10000)
limits["hkl"]["k"] = (-10000, 10000)
limits["hkl"]["l"] = (-10000, 10000)

instrument['Detectors']['Active'] = [1,1,1,1,1,1]
instrument['Detectors']['Forward'] = 'Fixed' # options: 'Fixed', 'SANS'
instrument['Detectors']['Backward'] = 'Symmetrical' # options: 'Symmetrical', 'Left side'
instrument['Guide'] = 'Long' # options: 'Long', 'Short', 'SANS'
instrument['Cryostat'] = 'Back' # options: 'Front', 'Back', 'None'

normals = [["Sample", "u"], ["Sample", "v"]]

fields = [
["Field","Direction"],
#["Choppers","Frequency"],
#["Choppers","Suppression"],
["Lambda","Res"],
["Lambda","Min"],
["Lambda","Max"],
["Sample","Constants"],
["Sample","Angles"],
["Sample","u"],
["Sample","v"],
["Sample","phi"],
["Sample","chi"],
["Sample","omega"],
["hkl","h"],
["hkl","k"],
["hkl","l"]
]

testfields = [
["Field","Direction"],
["Choppers","Frequency"],
["Choppers","Suppression"],
["Lambda","Min"],
["Lambda","Max"],
["Sample","Constants"],
["Sample","Angles"],
["Sample","u"],
["Sample","v"],
["Sample","phi"],
["Sample","chi"],
["Sample","omega"],
["hkl","h"],
["hkl","k"],
["hkl","l"]
]

# fangle = 2*np.degrees(np.arctan((0.06/100.0) / 0.01))

# fermi_at_600Hz = 15 * 1e-6 # pulse width of Fermi chopper at max speed
choppers = {}
choppers["Fermi"] = (20.359, 6.86)
choppers["1A"] = (20.51, 55.5)
choppers["1B"] = (20.56, 55.5)
choppers["2"] = (21.032, 40.4)
choppers["3"] = (22.034, 46.8)
choppers["4"] = (28.507, 69.3)
choppers["5"] = (48.134, 180)
choppers["6A"] = (71.1051, 2.6)
choppers["6B"] = (71.1351, 2.6)
detector_pos = 71.1351 + 2.42 + 3.0

def par_to_list(params):
    output = []
    output.append(params["Field"]["Direction"])
    output.append(params["Choppers"]["Frequency"])
    output.append(params["Choppers"]["Suppression"])
    output.append(params["Lambda"]["Min"])
    output.append(params["Lambda"]["Max"])
    output.append(params["Sample"]["Constants"]) 
    output.append(params["Sample"]["Angles"])
    output.append(params["Sample"]["u"])
    output.append(params["Sample"]["v"])
    output.append(params["Sample"]["phi"])
    output.append(params["Sample"]["chi"])
    output.append(params["Sample"]["omega"])
    output.append(params["hkl"]["h"])
    output.append(params["hkl"]["k"])
    output.append(params["hkl"]["l"])
    return output

def list_to_par(parlist):
    params = {}
    params["Choppers"] = {}
    params["Sample"] = {}
    params["Field"] = {}
    params["Lambda"] = {}
    params["hkl"] = {}
    params["Field"]["Direction"] = parlist[0]
    params["Choppers"]["Frequency"] = parlist[1]
    params["Choppers"]["Suppression"] = parlist[2]
    params["Lambda"]["Min"] = parlist[3]
    params["Lambda"]["Max"] = parlist[4]
    params["Sample"]["Constants"] = parlist[5]
    params["Sample"]["Angles"] = parlist[6]
    params["Sample"]["u"] = parlist[7]
    params["Sample"]["v"] = parlist[8]
    params["Sample"]["phi"] = parlist[9]
    params["Sample"]["chi"] = parlist[10]
    params["Sample"]["omega"] = parlist[11]
    params["hkl"]["h"] = parlist[12]
    params["hkl"]["k"] = parlist[13]
    params["hkl"]["l"] = parlist[14]
    return params
    
class Kernel:
    """
    This entity takes in the input parameters, validates them,
    and then produces plots on demand.
    It also keeps track of parameter changes, and only recalculates
    the picture if the parameters have changed since the last call.
    """
    def __init__(self, figure = None, precision = 1.0):
        self.orientations = np.arange(0, 16, 1).astype(np.int)
        self.results = []
        for i in self.orientations:
            self.results.append({'forward':None, 'backward':None,
                                 'fax':None, 'bax':None,
                                 'symbol':None, 'description':None,
                                 'settings':None})
        self.param = params
        self.instrument = instrument
        # self.exed = Exed(fname = "exed.par", spectrum = "2slit_lambdann.dat", cryo = False)
        # self.exed_hc = Exed(fname = "exed_hc.par", spectrum = "2slit_lambda.dat", cryo = True)
        self.exed = Exed(inst = [])
        self.exed.configure(self.param, self.instrument)
        self.figure = figure
        self.markers = []
        self.multiplier = precision
        self.cryo = False
        
    def writeout_settings(self, tname):
        param = copy.deepcopy(self.param)
        param['Sample']['u'] = ','.join([str(x) for x in self.param['Sample']['u']])
        param['Sample']['v'] = ','.join([str(x) for x in self.param['Sample']['v']])
        param['Sample']['Angles'] = ','.join([str(x) for x in self.param['Sample']['Angles']])
        param['Sample']['Constants'] = ','.join([str(x) for x in self.param['Sample']['Constants']])
        if type(param['hkl']['h']) == type([]):
            param['hkl']['h'] = ':'.join([str(x) for x in self.param['hkl']['h']])
        if type(param['hkl']['k']) == type([]):
            param['hkl']['k'] = ':'.join([str(x) for x in self.param['hkl']['k']])
        if type(param['hkl']['l']) == type([]):
            param['hkl']['l'] = ':'.join([str(x) for x in self.param['hkl']['l']])
        marks = []
        for i in self.markers:
            marks.append(','.join([str(x) for x in i]))
        if not tname.split(".")[-1] == 'exq':
            tname = ".".join(tname.split(".")+['exq'])
        target = open(tname, 'w')
        yaml.dump([param, self.instrument, marks, self.multiplier], stream = target)
        target.close()
        
    def load_settings(self, tname):
        target = open(tname, 'r')
        firstline = target.readline()
        target.close()
        if 'exeq' in firstline:
            selcom, params, markers, prec, instrument = parse_input_2018(tname)
            self.take_parameters(params)
            self.take_instrument(instrument)
            self.marker_points(markers)
            self.set_precision(prec)
            return None
        else:
            target = open(tname, 'r')
            xxxx = yaml.load(target)
            target.close()
            try:
                xxxx.split()
            except AttributeError:
                self.take_parameters(xxxx[0])
                self.take_instrument(xxxx[1])
                self.marker_points(xxxx[2])
                self.set_precision(xxxx[3])
            else:
                selcom, params, markers, prec, totalview, instrument = parse_input(tname)
                self.take_parameters(params)
                self.take_instrument(instrument)
                self.marker_points(markers)
                self.set_precision(prec)

    def take_instrument(self, inst):
        """
        instrument['Detectors']['Active'] = [1,2,3,4]
        instrument['Detectors']['Forward'] = 'Fixed' # options: 'Fixed', 'SANS'
        instrument['Detectors']['Backward'] = 'Symmetrical' # options: 'Symmetrical', 'Left side'
        instrument['Guide'] = 'Short' # options: 'Long', 'Short'
        instrument['Cryostat'] = 'Front' # options: 'Front', 'Back'
        """
        # param = list_to_par(params)
        temp = inst
        actlist = []
        try:
            for i in inst['Detectors']['Active']:
                if i:
                    actlist.append(True)
                else:
                    actlist.append(False)
        except KeyError:
            for x in range(6):
                actlist.append(True)
        if 'ixed' in inst['Detectors']['Forward']:
            forward = 'Fixed'
        else:
            forward = 'SANS'
        if 'metrical' in inst['Detectors']['Backward']:
            backward = 'Symmetrical'
        else:
            backward = 'Left side'
        if 'ong' in inst['Guide']:
            guide = 'Long'
        elif 'hort' in inst['Guide']:
            guide = 'Short'
        else:
            guide = 'SANS'
        if 'ront' in inst['Cryostat']:
            cryo = 'Front'
        elif 'ack' in inst['Cryostat']:
            cryo = 'Back'
        else:
            cryo = 'None'
        self.instrument['Detectors']['Active'] = actlist
        self.instrument['Detectors']['Forward'] = forward
        self.instrument['Detectors']['Backward'] = backward
        self.instrument['Guide'] = guide
        self.instrument['Cryostat'] = cryo
        self.exed.configure(self.param, self.instrument)

    def take_parameters(self, params):
        # param = list_to_par(params)
        param = params
        # step by step, we will check all the values
        # lim = limits["Field"]["Direction"]
        # anglim = self.exed.angle_limit
        # lim = (-anglim, anglim)
        lim = self.exed.angle_limit
        try:
            val = float(param["Field"]["Direction"])
        except ValueError:
            val = 0
        if val >= lim[0] and val <= lim[1]:
            self.param["Field"]["Direction"] = val
        else:
            self.param["Field"]["Direction"] = 0
        # choppers
        lim = limits["Lambda"]["Res"]
        try:
            val = float(param["Lambda"]["Res"])
        except ValueError:
            self.param["Lambda"]["Res"] = ""
            self.param["Choppers"]["Frequency"] = 120.0
            self.param["Choppers"]["Suppression"] = 1
        except KeyError:
            self.param["Lambda"]["Res"] = 0.05
            self.param["Choppers"]["Frequency"] = 120.0
            self.param["Choppers"]["Suppression"] = 1
        else:
            if val >= lim[0] and val <= lim[1]:
                self.param["Lambda"]["Res"] = val
                self.param["Choppers"]["Frequency"] = 120.0
                self.param["Choppers"]["Suppression"] = 1
        # wavelength
        lim = limits["Lambda"]["Min"]
        try:
            val = float(param["Lambda"]["Min"])
        except ValueError:
            val = 0.7
        except KeyError:
            val = 0.7
        if val >= lim[0] and val <= lim[1]:
            self.param["Lambda"]["Min"] = val
        else:
            self.param["Lambda"]["Min"] = 0.7
        lim = limits["Lambda"]["Max"]
        try:
            val = float(param["Lambda"]["Max"])
        except ValueError:
            val = 7.9
        except KeyError:
            val = 7.9
        if val >= lim[0] and val <= lim[1]:
            self.param["Lambda"]["Max"] = val
        else:
            self.param["Lambda"]["Max"] = 7.9
        if self.param["Lambda"]["Max"] < self.param["Lambda"]["Min"]:
            self.param["Lambda"]["Max"], self.param["Lambda"]["Min"] = self.param["Lambda"]["Min"], self.param["Lambda"]["Max"]
        # sample
        lim = limits["Sample"]["Constants"]
        val = param["Sample"]["Constants"]
        if ',' in val:
            val = val.split(',')
        else:
            val = val.split()
        for i in range(len(val)):
            try:
                a = float(val[i])
            except ValueError:
                val[i] = 4
        val = np.array(val).astype(np.float)
        if (val >= lim[0]).all() and (val <= lim[1]).all() and len(val) == 3:
            self.param["Sample"]["Constants"] = val
        else:
            self.param["Sample"]["Constants"] = np.array([4.0, 4.0, 4.0])
        lim = limits["Sample"]["Angles"]
        val = param["Sample"]["Angles"]
        if ',' in val:
            val = val.split(',')
        else:
            val = val.split()
        for i in range(len(val)):
            try:
                a = float(val[i])
            except ValueError:
                val[i] = 90
        val = np.array(val).astype(np.float)
        if (val >= lim[0]).all() and (val <= lim[1]).all() and len(val) == 3:
            self.param["Sample"]["Angles"] = val
        else:
            self.param["Sample"]["Angles"] = np.array([90.0, 90.0, 90.0])
        # sample orientation
        lim = limits["Sample"]["u"]
        val = param["Sample"]["u"]
        if ',' in val:
            val = val.split(',')
        else:
            val = val.split()
        for i in range(len(val)):
            try:
                a = float(val[i])
            except ValueError:
                val[i] = 0
        # val = normalise(np.array(val).astype(np.float))
        val = np.array(val).astype(np.float)
        if (val >= lim[0]).all() and (val <= lim[1]).all() and len(val) == 3 and not (val==0.0).all():
            self.param["Sample"]["u"] = val
        else:
            self.param["Sample"]["u"] = np.array([1.0, 0.0, 0.0])
        lim = limits["Sample"]["v"]
        val = param["Sample"]["v"]
        if ',' in val:
            val = val.split(',')
        else:
            val = val.split()
        for i in range(len(val)):
            try:
                a = float(val[i])
            except ValueError:
                val[i] = 0
        # val = normalise(np.array(val).astype(np.float))
        val = np.array(val).astype(np.float)
        if (val >= lim[0]).all() and (val <= lim[1]).all() and len(val) == 3 and not (val==0.0).all():
            self.param["Sample"]["v"] = val
        else:
            self.param["Sample"]["v"] = np.array([0.0, 0.0, 1.0])
        # goniometer angles
        h, k, l = param["Sample"]["phi"], param["Sample"]["chi"], param["Sample"]["omega"]
        hl, kl, ll = limits["Sample"]["phi"], limits["Sample"]["chi"], limits["Sample"]["omega"]
        count = 0
        try:
            val = float(h)
        except ValueError:
            count +=1
            self.param["Sample"]["phi"] = 0
        else:
            if val >= hl[0] and val <= hl[1]:
                self.param["Sample"]["phi"] = val
            else:
                self.param["Sample"]["phi"] = 0.0
        try:
            val = float(k)
        except ValueError:
            count +=1
            self.param["Sample"]["chi"] = 0
        else:
            if val >= kl[0] and val <= kl[1]:
                self.param["Sample"]["chi"] = val
            else:
                self.param["Sample"]["chi"] = 0.0
        try:
            val = float(l)
        except ValueError:
            count +=1
            self.param["Sample"]["omega"] = 0
        else:
            if val >= ll[0] and val <= ll[1]:
                self.param["Sample"]["omega"] = val
            else:
                self.param["Sample"]["omega"] = 0.0
        self.count = count
        # hkl
        h, k, l = str(param["hkl"]["h"]), str(param["hkl"]["k"]), str(param["hkl"]["l"])
        hl, kl, ll = limits["hkl"]["h"], limits["hkl"]["k"], limits["hkl"]["l"]
        count = 0
        hs = h.split(':')
        if len(hs) > 1:
            temp = []
            for p in hs:
                try:
                    val = float(p)
                except ValueError:
                    val = 0.0
                temp.append(val)
            self.param["hkl"]["h"] = [temp[0], temp[1]]
            self.hrange = temp
            count += 1
        else:
            try:
                val = float(h)
            except ValueError:
                count +=1
                self.param["hkl"]["h"] = 'h'
                self.hrange = []
            else:
                if val >= hl[0] and val <= hl[1]:
                    self.param["hkl"]["h"] = val
                    self.hrange = []
                else:
                    self.param["hkl"]["h"] = 0.0
                    self.hrange = []
        ks = k.split(':')
        if len(ks) > 1:
            temp = []
            for p in ks:
                try:
                    val = float(p)
                except ValueError:
                    val = 0.0
                temp.append(val)
            self.param["hkl"]["k"] = [temp[0], temp[1]]
            self.krange = temp
            count += 1
        else:
            try:
                val = float(k)
            except ValueError:
                count +=1
                self.param["hkl"]["k"] = 'k'
                self.krange = []
            else:
                if val >= kl[0] and val <= kl[1]:
                    self.param["hkl"]["k"] = val
                    self.krange = []
                else:
                    self.param["hkl"]["k"] = 0.0
                    self.krange = []
        ls = l.split(':')
        if len(ls) > 1:
            temp = []
            for p in ls:
                try:
                    val = float(p)
                except ValueError:
                    val = 0.0
                temp.append(val)
            self.param["hkl"]["l"] = [temp[0], temp[1]]
            self.lrange = temp
            count += 1
        else:
            try:
                val = float(l)
            except ValueError:
                count +=1
                self.param["hkl"]["l"] = 'l'
                self.lrange = []
            else:
                if val >= ll[0] and val <= ll[1]:
                    self.param["hkl"]["l"] = val
                    self.lrange = []
                else:
                    self.param["hkl"]["l"] = 0.0
                    self.lrange = []
        self.count = count
        
    def marker_points(self, mark_points):
        new = []
        self.markers = []
        for val in mark_points:
            if ',' in val:
                nval = val.split(',')
            else:
                nval = val.split()
            if len(nval) == 3:
                try:
                    a = float(nval[0])
                    b = float(nval[1])
                    c = float(nval[2])
                except ValueError:
                    new.append("")
                else:
                    nval = np.array(nval).astype(np.float)
                    new.append(" ".join([str(x)[:7] for x in [a,b,c]]))
                    self.markers.append(nval)
            else:
                new.append("")
        return new
        
    def set_precision(self, value):
        val = str(value)
        if ',' in val:
            nval = val.split(',')
        else:
            nval = val.split()
        if len(nval) == 1:
            try:
                a = float(nval[0])
            except ValueError:
                return str(self.multiplier)[:5]
            else:
                nval = float(nval[0])
        if (nval >= 0.5) and (nval <= 3.0):
            self.multiplier = nval
        self.exed.segments = 2*int(round(3 + 2.0*self.multiplier))
        # self.exed_hc.segments = 2*int(round(3 + 2.0*self.multiplier))
        return str(self.multiplier)[:5]
    
    def current_parameters(self):
        return self.param
    
    def current_parameters_str(self, filename = ""):
        param = copy.deepcopy(self.param)
        markers = []
        prec = -1
        for i in self.param.items():
            name1 = i[0]
            for j in i[1].items():
                name2 = j[0]
                try:
                    l = len(j[1])
                except TypeError:
                    l = 0
                if l:
                    if name1 == "hkl":
                        param[name1][name2] = []
                        val = j[1]
                        try:
                            val.split()
                        except AttributeError:
                            for k in range(len(val)):
                                if abs(val[k] - round(val[k])) < 1e-8:
                                    param[name1][name2].append(str(int(val[k])))
                                else:
                                    param[name1][name2].append(str(val[k]))
                            param[name1][name2] = ":".join(param[name1][name2])
                        else:
                            try:
                                round(val)
                            except TypeError:
                                param[name1][name2] = val
                            else:
                                if abs(val - round(val)) < 1e-8:
                                    param[name1][name2] = str(int(val))
                                else:
                                    param[name1][name2] = str(val)
                    else:
                        param[name1][name2] = []
                    # print( name1, name2
                        for k in range(l):
                            param[name1][name2].append("")
                            val = j[1][k]
                        # print( name1, name2, val
                            try:
                                round(val)
                            except TypeError:
                                param[name1][name2] = val
                            else:
                                if abs(val - round(val)) < 1e-8:
                                    param[name1][name2][k] = str(int(val))
                                else:
                                    param[name1][name2][k] = str(val)
                        param[name1][name2] = " ".join(param[name1][name2])
                    # print( param[name1][name2]
                else:
                    val = j[1]
                    # print( name1, name2, val
                    try:
                        round(val)
                    except TypeError:
                        param[name1][name2] = val
                    else:
                        if abs(val - round(val)) < 1e-8:
                            param[name1][name2] = str(int(val))
                        else:
                            param[name1][name2] = str(val)
        for i in range(len(self.markers)):
            try:
                l = len(self.markers[i])
            except TypeError:
                l = 0
            if l:
                markers.append([])
                for k in range(l):
                    val = self.markers[i][k]
                    if abs(val - round(val)) < 1e-8:
                        markers[-1].append(str(int(val)))
                    else:
                        markers[-1].append(str(val))
                markers[-1] = " ".join(markers[-1])
            else:
                markers.append([])
                val = j[1]
                if abs(val - round(val)) < 1e-8:
                    markers[-1].append(str(int(val)))
                else:
                    markers[-1].append(str(val))
        val = self.multiplier
        if abs(val - round(val)) < 1e-8:
            prec = (str(int(val)))
        else:
            prec = (str(val))
        if filename:
            dump = open(filename, 'w')
            for i in param.items():
                for j in i[1].items():
                    dump.write(i[0]+"_"+j[0]+"="+j[1]+'\n')
            #for i, j in enumerate(markers):
            #    dump.write("Point"+str(i)+"="+j+'\n')
            for i in range(5):
                try:
                    v = markers[i]
                except IndexError:
                    v = ""
                    markers.append("")
                dump.write("Point"+str(i+1)+"="+v+'\n')
            dump.write("Precision="+prec+'\n')
            dump.close()
        return param, markers, prec
    
    def request_parameters(self):
        fields = [
        ["Field","Direction"],
        # ["Choppers","Frequency"],
        # ["Choppers","Suppression"],
        ["Lambda","Res"],
        ["Lambda","Min"],
        ["Lambda","Max"],
        ["Sample","Constants"],
        ["Sample","Angles"],
        ["Sample","u"],
        ["Sample","v"],
        ["Sample","phi"],
        ["Sample","chi"],
        ["Sample","omega"],
        ["hkl","h"],
        ["hkl","k"],
        ["hkl","l"]]
        return fields

    def chopper_freqs(self):
        lmin, lmax = self.param["Lambda"]["Min"], self.param["Lambda"]["Max"]
        lres = self.param["Lambda"]["Res"]
        bwidth = lmax - lmin
        lmean = (lmax+lmin)/2.0
        if not lres:
            which, freq_first = "", ""
        else:
            which, freq_first = firstchopper(lres, lmean)
            freq_first = str(max(np.int(np.round(freq_first/2.0))*2, 5))
        freq_last = lastchopper(bwidth)
        if freq_last > 120:
            freq_last = "Band too narrow"
        else:
            freq_last = str(max(np.int(np.round(freq_last/2.0))*2, 5))
        self.fpulse = freq_first
        self.fband = freq_last
        self.choppertype = which
        self.redfactor = intensity_drop(int(freq_first), int(freq_last), which)
        return which, freq_first, freq_last
        
    def uv_range(self):
        parameters = self.param
        field = -parameters['Field']['Direction']
        bu = parameters["Sample"]["u"]
        bv = parameters["Sample"]["v"]
        gonio = [parameters["Sample"]["phi"], parameters["Sample"]["chi"], parameters["Sample"]["omega"]]
        sss = Sample(parameters["Sample"]["Constants"],parameters["Sample"]["Angles"])
        sss.orient(bu, bv, field, gonio)
        u, v = sss.u, sss.v
        # axis = np.cross(v, u)
        ## axis = np.array([0,0,1.0])
        # rmat = np.dot(np.dot(sss.Binv, arb_rotation(axis, rot)), sss.B)
        ## rotmin = np.dot(np.dot(sss.Binv, arb_rotation(axis, -15.0 - field)), sss.B)
        ## rotmax = np.dot(np.dot(sss.Binv, arb_rotation(axis, 15.0 - field)), sss.B)
        sss.orient(bu, bv, -15.0, gonio)
        newumin, newvmin = sss.u, sss.v
        sss.orient(bu, bv, 15.0, gonio)
        newumax, newvmax = sss.u, sss.v
        #newumax = np.dot(rotmax, u)
        #newumin = np.dot(rotmin, u)
        #newvmax = np.dot(rotmax, v)
        #newvmin = np.dot(rotmin, v)
        vecs = [newumax, newumin, newvmax, newvmin]
        newvecs = []
        for i in range(len(vecs)):
            try:
                l = len(vecs[i])
            except TypeError:
                l = 0
            if l:
                newvecs.append([])
                for k in range(l):
                    val = vecs[i][k]
                    if val > 0:
                        lim = 5
                    else:
                        lim = 6
                    if abs(val - round(val)) < 1e-8:
                        newvecs[-1].append(str(int(val)))
                    else:
                        newvecs[-1].append(str(val)[:lim])
                newvecs[-1] = " ".join(newvecs[-1])
            else:
                newvecs.append([])
                val = vecs[1]
                if val > 0:
                    lim = 5
                else:
                    lim = 6
                if abs(val - round(val)) < 1e-8:
                    newvecs[-1].append(str(int(val)))
                else:
                    newvecs[-1].append(str(val)[:5])
        return newvecs
        
    def show_slider_plot(self, outFile = ""):
        param = self.param
        exed = self.exed
        plot.inverse_plot(param, outFile)
        return True
        
    def show_total_eq_range(self, outFile = ""):
        param = self.param
        exed = self.exed
        exed.chopper["frequency"] = param["Choppers"]["Frequency"]
        exed.chopper["suppression"] = param["Choppers"]["Suppression"]
        lmin, lmax = (param["Lambda"]["Min"], param["Lambda"]["Max"])
        magf = -param["Field"]["Direction"]
        a, b = exed.absolute_range(n_energy(max(lmin, 0.7)), magf)
        plot.plot_eq_map(a,b, outFile, self.figure)
        return True
    
    def show_help(self, outFile = ""):
        plot.plot_help_message(outFile, self.figure)
        return True
    
    def show_welcome_message(self, outFile = ""):
        plot.plot_welcome_message(outFile, self.figure)
        return True
    
    def show_uv_range(self, outFile = ""):
        a = self.uv_range()
        plot.plot_uv_range(a, outFile, self.figure)
        return True
    
    def show_instrument(self, outFile = ""):
        param = self.param
        exed = self.exed
        ori = param["Field"]["Direction"]
        samp = Sample(param["Sample"]["Constants"], param["Sample"]["Angles"])
        gonio = [param["Sample"]["phi"], param["Sample"]["chi"], param["Sample"]["omega"]]
        if not samp.orient(param["Sample"]["u"], param["Sample"]["v"], -param["Field"]["Direction"], gonio):
            plot.error(outFile, self.figure)
            return False
        u, v = samp.u, samp.v
        vecs = [u, v]
        newvecs = []
        for i in range(len(vecs)):
            try:
                l = len(vecs[i])
            except TypeError:
                l = 0
            if l:
                newvecs.append([])
                for k in range(l):
                    val = vecs[i][k]
                    if val > 0:
                        lim = 5
                    else:
                        lim = 6
                    if abs(val - round(val)) < 1e-8:
                        newvecs[-1].append(str(int(val)))
                    else:
                        newvecs[-1].append(str(val)[:lim])
                newvecs[-1] = " ".join(newvecs[-1])
            else:
                newvecs.append([])
                val = vecs[1]
                if val > 0:
                    lim = 5
                else:
                    lim = 6
                if abs(val - round(val)) < 1e-8:
                    newvecs[-1].append(str(int(val)))
                else:
                    newvecs[-1].append(str(val)[:5])
        u, v = newvecs[0], newvecs[1]
        alltubes, goodtubes, shadowtubes = [], [], []
        for i in exed.current_panels_at(int(round(ori))):
            if i == []:
                continue
            for k in i.tubes:
                alltubes.append(k.frame)
                goodtubes += k.goodparts
                shadowtubes += k.badparts
        tubes = (alltubes, goodtubes, shadowtubes)
        # tubes = exed.tubes[exed.minangle - int(round(ori))]
        a = self.uv_range()
        a.append(u)
        a.append(v)
        plot.plot_instrument(ori, tubes, a, outFile = outFile, fig = self.figure, goniometer = gonio)
        return True
    
    def show_detector_panels(self, outFile = ""):
        param = self.param
        exed = self.exed
        ori = param["Field"]["Direction"]
        samp = Sample(param["Sample"]["Constants"], param["Sample"]["Angles"])
        gonio = [param["Sample"]["phi"], param["Sample"]["chi"], param["Sample"]["omega"]]
        if not samp.orient(param["Sample"]["u"], param["Sample"]["v"], -param["Field"]["Direction"], gonio):
            plot.error(outFile, self.figure)
            return False
        self.update_intensity()
        wrong = False
        if self.choppertype == "Resolution too high":
            wrong = True
        u, v = samp.u, samp.v
        vecs = [u, v]
        markers = self.markers
        endmarkers = []
        if len(markers) > 0:
            ub = samp.themat
            realcoords = rotate(ub, np.row_stack(markers))
            spherical = reverse_q_vector(realcoords)
            # print( "marker_point_coordinates\n", spherical
            for pum, i in enumerate(spherical):
                alpha, beta = i[1], i[2]
                # alpha = np.fmod(alpha + 4*np.pi, 2*np.pi)
                # alpha = alpha - 2*np.pi*(alpha > 3.0/2.0*np.pi)
                n = np.array([np.cos(alpha) * np.cos(beta), np.sin(beta), np.sin(alpha) * np.cos(beta)])
                # print( "normal\n", n
                for j in exed.current_panels_at(int(round(ori))):
                    # print( "panel limits\n", j.limits
                    if j == []:
                        continue
                    temp = np.array([0.0, alpha, beta])
                    if j.point_in_range(temp):
                    # if True:
                        point = plane_line_intersection(plane = [j.normal, j.middle],
                                                       line = [np.zeros(3), n])
                        print( "point ", point)
                        try:
                            plen = len(point)
                        except:
                            continue
                        else:
                            if (np.abs(point) < 1e-14).all():
                                continue
                        tubepoint = tubepoints(np.array([point]))
                        #if tubepoint[:,1] < np.pi:
                            #tubepoint[:,1] += 2*np.pi
                        #elif tubepoint[:,1] > np.pi:
                            #tubepoint[:,1] -= 2*np.pi
                        print( "tubepoint ", tubepoint)
                        wave = 1.0/i[0]
                        print( "wave ", wave)
                        if wave >= param["Lambda"]["Min"] and wave <= param["Lambda"]["Max"]:
                            spec = exed.current_spectrum(wave)*self.redfactor
                        else:
                            spec = 0.0
                        if realcoords[pum,2] >= 0.0:
                            wave = -wave
                            spec = 0.0
                        print( "spec ", spec)
                        endmarkers.append([point, tubepoint[0], wave, spec, markers[pum]])
                        print( "output", endmarkers[-1])
        # tubes = exed.tubes[int(round(ori))-exed.minangle]
        # tubes, goodtubes, shadowtubes = exed.widetubes[int(round(ori))-exed.minangle]
        alltubes, goodtubes, shadowtubes = [], [], []
        for i in exed.current_panels_at(int(round(ori))):
            if i == []:
                continue
            for k in i.tubes:
                alltubes.append(k.frame)
                goodtubes += k.goodparts
                try:
                    shadowtubes += k.badparts
                except TypeError:
                    print( shadowtubes)
                    print( k.badparts)
                    import sys
                    sys.exit(0)
        plot.plot_detector_panels(ori, [alltubes, goodtubes, shadowtubes],
                                  outFile = outFile, fig = self.figure, goniometer = gonio,
                                  markers = endmarkers, choppers_wrong = wrong)
        return True

    def update_intensity(self):
        ttt, tttt, ttttt = self.chopper_freqs()
        match = 0
        try:
            freqa = float(self.fpulse)
        except ValueError:
            freqa = 1.0
        try:
            freqb = float(self.fband)
        except ValueError:
            freqb = 1.0
        try:
            float(self.param["Lambda"]["Res"])
        except ValueError:
            redfactor = 1.0
        else:
            redfactor = intensity_drop(freqa, freqb, self.choppertype)
        self.redfactor = redfactor
     
    def guess_hkl_range(self, samp, panels, vecs, waves):
        lmin, lmax = waves[0], waves[1]
        vec1, vec2, origin = vecs[0], vecs[1], vecs[2]
        points = []
        rotmat = np.linalg.inv(samp.themat)
        for i in panels:
            for j in i.tubes:
                for knt in j.goodparts:
                    # k = j.angular
                    if len(knt) > 0:
                        k = tubepoints(knt)
                        line_a, line_b = k.copy(), k.copy()
                        # print( line_a
                        line_a[:,0] = n_k(lmin)
                        line_b[:,0] = n_k(lmax)
                        line_a, line_b = q_vector(line_a), q_vector(line_b)
                        points.append(rotate(rotmat, line_a))
                        points.append(rotate(rotmat, line_b))
        points = np.row_stack(points)
        pmax, pmin = points.max(0), points.min(0)
        range1 = (pmin[np.where(vec1 > 0.0)], pmax[np.where(vec1 > 0.0)])
        range2 = (pmin[np.where(vec2 > 0.0)], pmax[np.where(vec2 > 0.0)])
        return range1, range2
    
    def guess_hkl_range_3D(self, samp, panels, vecs, waves):
        lmin, lmax = waves[0], waves[1]
        vec1, vec2, vec3 = vecs[0], vecs[1], vecs[2]
        points = []
        rotmat = np.linalg.inv(samp.themat)
        for i in panels:
            for j in i.tubes:
                line_a, line_b = j.qvectors(lmin), j.qvectors(lmax)
                points.append(rotate(rotmat, line_a))
                points.append(rotate(rotmat, line_b))
        total = 0
        for i in points:
            total += len(i)
        if total > 0:
            points = np.row_stack(points)
            pmax, pmin = points.max(0), points.min(0)
            range1 = [pmin[0], pmax[0]]
            range2 = [pmin[1], pmax[1]]
            range3 = [pmin[2], pmax[2]]
        else:
            range1 = [1.0, 2.0]
            range2 = [1.0, 2.0]
            range3 = [1.0, 2.0]
        return range1, range2, range3
    
    def new_intensities(self, forward = True):
        param = self.param
        exed = self.exed
        ori = param["Field"]["Direction"]
        samp = Sample(param["Sample"]["Constants"], param["Sample"]["Angles"])
        gonio = [param["Sample"]["phi"], param["Sample"]["chi"], param["Sample"]["omega"]]
        if not samp.orient(param["Sample"]["u"], param["Sample"]["v"], -param["Field"]["Direction"], gonio):
            plot.error(outFile, self.figure)
            return False
        u, v = samp.u, samp.v
        vecs = [u, v]
        markers = self.markers
        rfactor = self.redfactor
        endmarkers = []
        # panels = exed.dets[ori-exed.minangle].panels
        # panels = exed.get_right_panels(param, self.instrument)
        panels = exed.current_panels_at(ori)
        temp = []
        for i in panels:
            if i == []:
                continue
            if (i.forward == forward):
                temp.append(i)
        panels = temp
        lmin, lmax = (param["Lambda"]["Min"], param["Lambda"]["Max"])
        h, k, l = param["hkl"]["h"], param["hkl"]["k"], param["hkl"]["l"]
        base = np.eye(3)
        if (h == "h" or '[' in str(h)) and (k == "k" or '[' in str(k)):
            vec1, vec2 = base[0], base[1]
            normal = np.cross(vec1, vec2)
            origin = float(l) * base[2]
            ind = (0, 1)
            perp = 2
            symbol = 'hk'
            range1 = self.hrange
            range2 = self.krange
            plane = " ".join(["(", symbol[0], symbol[1], str(origin.sum()), ")"])
        if (h == "h" or '[' in str(h)) and (l == "l" or '[' in str(l)):
            vec1, vec2 = base[0], base[2]
            normal = np.cross(vec1, vec2)
            origin = float(k) * base[1]
            ind = (0, 2)
            perp = 1
            symbol = 'hl'
            range1 = self.hrange
            range2 = self.lrange
            plane = " ".join(["(", symbol[0], str(origin.sum()), symbol[1], ")"])
        if (l == "l" or '[' in str(l)) and (k == "k" or '[' in str(k)):
            vec1, vec2 = base[1], base[2]
            normal = np.cross(vec1, vec2)
            origin = float(h) * base[0]
            ind = (1, 2)
            perp = 0
            symbol = 'kl'
            range1 = self.krange
            range2 = self.lrange
            plane = " ".join(["(",str(origin.sum()), symbol[0],  symbol[1], ")"])
        # all parameters defined now
        # print( "range1, ", range1
        # print( "range2, ", range2
        rans = self.guess_hkl_range_3D(samp, panels, (vec1, vec2, origin), (lmin, lmax))
        ran1 = rans[ind[0]]
        ran2 = rans[ind[1]]
        if forward:
            part1 = np.linspace(ran1[0]-0.2, ran1[1]+0.2, int(round(100*self.multiplier)))
            # part2 = np.linspace(-ran2[1]-0.2, -ran2[0]+0.2, int(round(100*self.multiplier)))
            part2 = np.linspace(ran2[0]-0.2, ran2[1]+0.2, int(round(100*self.multiplier)))
        else:
            part1 = np.linspace(ran1[0]-0.2, ran1[1]+0.2, int(round(100*self.multiplier)))
            part2 = np.linspace(ran2[0]-0.2, ran2[1]+0.2, int(round(100*self.multiplier)))
        if len(range1) == 2:
            part1 = np.linspace(range1[0], range1[1], int(round(100*self.multiplier)))
        if len(range2) == 2:
            part2 = np.linspace(range2[0], range2[1], int(round(100*self.multiplier)))
        ax, bx, vx = np.ix_(part1, part2, vec1)
        a2x, b2x, v2x = np.ix_(part1, part2, vec2)
        a3x, b3x, orr = np.ix_(part1, part2, origin)
        plotgrid = np.zeros((len(part1), len(part2), 3))
        plotgrid += ax * vx
        plotgrid += b2x * v2x
        plotgrid += orr
        image = np.zeros((len(part1), len(part2)))
        shadow = np.zeros((len(part1), len(part2)))
        rotmat = samp.themat
        for i in range(len(plotgrid)):
            points = rotate(rotmat, plotgrid[i])
            points[np.where(points[:,2] >= 0.0)] = 0.0
            qpoints = reverse_q_vector(points)
            # if not forward:
            #     qpoints[:,1] = np.fmod(qpoints[:,1] + 4*np.pi, 2*np.pi)
            lambdas = n_lambda_fromk(qpoints[:,0])
            mask = (lambdas >= lmin)*(lambdas <= lmax)
            anglemask = np.zeros(len(points))
            shadowmask = np.zeros(len(points))
            for k in range(len(anglemask)):
                if mask[k]:
                    for p in panels:
                        anglemask[k] = max(p.point_in_range(qpoints[k]), anglemask[k])
            for k in range(len(anglemask)):
                if mask[k] and not anglemask[k]:
                    for p in panels:
                        shadowmask[k] = max(p.point_in_bigframe(qpoints[k]), shadowmask[k])
            intensity = exed.current_spectrum(lambdas) * rfactor * anglemask
            image[i, :] = intensity
            shadow[i, :] = shadowmask
        tmarkers = self.markers
        endmarkers,  markers = [],  []
        for i in tmarkers:
            if (np.abs(i) < 1e-14).all():
                continue
            else:
                markers.append(i)
        if len(markers) > 0:
            ub = samp.themat
            realcoords = rotate(ub, np.row_stack(markers))
            spherical = reverse_q_vector(realcoords)
            for pum, i in enumerate(spherical):
                alpha, beta = i[1], i[2]
                # alpha = np.fmod(alpha + 4*np.pi, 2*np.pi)
                # alpha = alpha - 2*np.pi*(alpha > 3.0/2.0*np.pi)
                n = np.array([np.cos(alpha) * np.cos(beta), np.sin(beta), np.sin(alpha) * np.cos(beta)])
                exists = False
                for j in exed.current_panels_at(int(round(ori))):
                    if j == []:
                        continue
                    temp = np.array([0.0, alpha, beta])
                    if j.point_in_range(temp):
                        exists = True
                    if realcoords[pum,2] >= 0.0:
                        exists = False
                if abs(origin.sum()-markers[pum][perp]) < 1e-5:
                    wave = 1.0/i[0]
                    print( "wave ", wave)
                    if wave >= param["Lambda"]["Min"] and wave <= param["Lambda"]["Max"]:
                        spec = exed.current_spectrum(wave)*self.redfactor
                    else:
                        spec = 0.0
                    print( "spec ", spec)
                    if not exists:
                        spec = 0.0
                    endmarkers.append([markers[pum][np.array([ind[0], ind[1]])], None,
                                       wave, spec, markers[pum]])
                    print( "output", endmarkers[-1])
        return image, part1, part2, endmarkers, symbol, samp.description(), plane, shadow
 
    def show_planes_both(self, outFile = ""):
        self.update_intensity()
        if self.count < 2:
            plot.apology(outFile = outFile, fig = self.figure)
            return None
        elif self.count == 3:
            self.show_volumes_both(outFile)
            return None
        picf, ax1f, ax2f, markf, symf, desf, planf, shadf = self.new_intensities(forward = True)
        picb, ax1b, ax2b, markb, symb, desb, planb, shadb = self.new_intensities(forward = False)
        if self.choppertype == "Resolution too high":
            desf = "INTENSITY INCORRECT\nPlease adjust the resolution settings"
        plot.plot_simple_map((picf, picb), (ax1f, ax1b), (ax2f, ax2b), outFile = outFile, fig = self.figure,
                             markers = (markf, markb), symbol = symf, description = desf, plane = planf,
                             shadows = (shadf, shadb))

    def new_intensities_3D(self, forward = True):
        param = self.param
        exed = self.exed
        ori = param["Field"]["Direction"]
        samp = Sample(param["Sample"]["Constants"], param["Sample"]["Angles"])
        gonio = [param["Sample"]["phi"], param["Sample"]["chi"], param["Sample"]["omega"]]
        if not samp.orient(param["Sample"]["u"], param["Sample"]["v"], -param["Field"]["Direction"], gonio):
            plot.error(outFile, self.figure)
            return False
        u, v = samp.u, samp.v
        vecs = [u, v]
        markers = self.markers
        rfactor = self.redfactor
        endmarkers = []
        # panels = exed.dets[ori-exed.minangle].panels
        panels = exed.current_panels_at(int(round(ori)))
        temp = []
        for i in panels:
            if i == []:
                continue
            if (i.forward == forward):
                temp.append(i)
        panels = temp
        lmin, lmax = (param["Lambda"]["Min"], param["Lambda"]["Max"])
        h, k, l = param["hkl"]["h"], param["hkl"]["k"], param["hkl"]["l"]
        base = np.eye(3)
        if (h == "h" or '[' in str(h)) and (k == "k" or '[' in str(k)) and (l == "l" or '[' in str(l)):
            vec1, vec2, vec3 = base[0], base[1], base[2]
            # origin = float(l) * base[2]
            ind = (0, 1, 2)
            perp = None
            symbol = 'hkl'
            range1 = self.hrange
            range2 = self.krange
            range3 = self.lrange
            plane = " ".join(["(", symbol[0], symbol[1], symbol[2], ")"])
        # all parameters defined now
        #print( "range1, ", range1
        #print( "range2, ", range2
        ran1, ran2, ran3 = self.guess_hkl_range_3D(samp, panels, (vec1, vec2, vec3), (lmin, lmax))
        vollen = int(round(20*self.multiplier))
        if forward:
            part1 = np.linspace(ran1[0]-0.2, ran1[1]+0.2, int(round(20*self.multiplier)))
            part2 = np.linspace(-ran2[1]-0.2, -ran2[0]+0.2, int(round(20*self.multiplier)))
            part3 = np.linspace(ran3[0]-0.2, ran3[1]+0.2, int(round(20*self.multiplier)))
        else:
            part1 = np.linspace(ran1[0]-0.2, ran1[1]+0.2, int(round(20*self.multiplier)))
            part2 = np.linspace(ran2[0]-0.2, ran2[1]+0.2, int(round(20*self.multiplier)))
            part3 = np.linspace(ran3[0]-0.2, ran3[1]+0.2, int(round(20*self.multiplier)))
        if len(range1) == 2:
            part1 = np.linspace(range1[0], range1[1], int(round(20*self.multiplier)))
        if len(range2) == 2:
            part2 = np.linspace(range2[0], range2[1], int(round(20*self.multiplier)))
        if len(range3) == 2:
            part3 = np.linspace(range3[0], range3[1], int(round(20*self.multiplier)))
        V = np.zeros([vollen, vollen, vollen])
        coords = np.zeros([vollen, vollen, vollen, 3])
        rotmat = samp.themat
        for i in range(len(part1)):
            for j in range(len(part2)):
                points = np.zeros((vollen, 3))
                for k in range(len(part3)):
                    points[k] = [part1[i], part2[j], part3[k]]
                coords[i, j, :, :] = points.copy()
                points = rotate(rotmat, points)
                points[np.where(points[:,2] >= 0.0)] = 0.0
                qpoints = reverse_q_vector(points)
                # if not forward:
                #     qpoints[:,1] = np.fmod(qpoints[:,1] + 4*np.pi, 2*np.pi)
                lambdas = n_lambda_fromk(qpoints[:,0])
                mask = (lambdas >= lmin)*(lambdas <= lmax)
                anglemask = np.zeros(len(points))
                for k in range(len(anglemask)):
                    if mask[k]:
                        for p in panels:
                            anglemask[k] = max(p.point_in_range(qpoints[k]), anglemask[k])
                intensity = exed.current_spectrum(lambdas) * rfactor * anglemask
                V[i, j, :] = intensity
        mask = np.nonzero(V)
        V = np.nan_to_num(V[mask])
        coords = np.nan_to_num(coords[mask])
        tmarkers = self.markers
        endmarkers,  markers = [], []
        for i in tmarkers:
            if (np.abs(i) < 1e-14).all():
                continue
            else:
                markers.append(i)
        if len(markers) > 0:
            ub = samp.themat
            realcoords = rotate(ub, np.row_stack(markers))
            spherical = reverse_q_vector(realcoords)
            for pum, i in enumerate(spherical):
                alpha, beta = i[1], i[2]
                # alpha = np.fmod(alpha + 4*np.pi, 2*np.pi)
                # alpha = alpha - 2*np.pi*(alpha > 3.0/2.0*np.pi)
                n = np.array([np.sin(alpha) * np.cos(beta), np.sin(beta), np.cos(alpha) * np.cos(beta)])
                exists = False
                # for j in exed.dets[ori-exed.minangle].panels:
                for j in exed.current_panels_at(int(round(ori))):
                    if j == []:
                        continue
                    temp = np.array([0.0, alpha, beta])
                    # print( np.degrees(temp)
                    if j.point_in_range(temp):
                        exists = True
                if True:
                    wave = 1.0/i[0]
                    print( "wave ", wave)
                    if wave >= param["Lambda"]["Min"] and wave <= param["Lambda"]["Max"]:
                        spec = exed.current_spectrum(wave)*self.redfactor
                    else:
                        spec = 0.0
                    print( "spec ", spec)
                    if not exists:
                        spec = 0.0
                    endmarkers.append([markers[pum], None,
                                       wave, spec, markers[pum]])
                    print( "output", endmarkers[-1])
        return (V, coords), part1, part2, part3, endmarkers, symbol, samp.description(), plane
 
    def show_volumes_both(self, outFile = ""):
        self.update_intensity()
        picf, ax1f, ax2f, ax3f, markf, symf, desf, planf = self.new_intensities_3D(forward = True)
        picb, ax1b, ax2b, ax3b, markb, symb, desb, planb = self.new_intensities_3D(forward = False)
        if self.choppertype == "Resolution too high":
            desf = "INTENSITY INCORRECT\nPlease adjust the resolution settings"
        plot.plot_simple_volume((picf, picb), (ax1f, ax1b), (ax2f, ax2b), (ax3f, ax3b), outFile = outFile, fig = self.figure,
                             markers = (markf, markb), symbol = symf, description = desf, plane = planf)

    def new_total_plane(self, forward = True):
        param = self.param
        exed = self.exed
        ori = param["Field"]["Direction"]
        samp = Sample(param["Sample"]["Constants"], param["Sample"]["Angles"])
        gonio = [param["Sample"]["phi"], param["Sample"]["chi"], param["Sample"]["omega"]]
        lmin, lmax = (param["Lambda"]["Min"], param["Lambda"]["Max"])
        h, k, l = param["hkl"]["h"], param["hkl"]["k"], param["hkl"]["l"]
        base = np.eye(3)
        if (h == "h" or '[' in str(h)) and (k == "k" or '[' in str(k)):
            vec1, vec2 = base[0], base[1]
            normal = np.cross(vec1, vec2)
            origin = float(l) * base[2]
            ind = (0, 1)
            perp = 2
            symbol = 'hk'
            range1 = self.hrange
            range2 = self.krange
            plane = " ".join(["(", symbol[0], symbol[1], str(origin.sum()), ")"])
        if (h == "h" or '[' in str(h)) and (l == "l" or '[' in str(l)):
            vec1, vec2 = base[0], base[2]
            normal = np.cross(vec1, vec2)
            origin = float(k) * base[1]
            ind = (0, 2)
            perp = 1
            symbol = 'hl'
            range1 = self.hrange
            range2 = self.lrange
            plane = " ".join(["(", symbol[0], str(origin.sum()), symbol[1], ")"])
        if (l == "l" or '[' in str(l)) and (k == "k" or '[' in str(k)):
            vec1, vec2 = base[1], base[2]
            normal = np.cross(vec1, vec2)
            origin = float(h) * base[0]
            ind = (1, 2)
            perp = 0
            symbol = 'kl'
            range1 = self.krange
            range2 = self.lrange
            plane = " ".join(["(",str(origin.sum()), symbol[0],  symbol[1], ")"])
        # all parameters defined now
        # print( "range1, ", range1
        # print( "range2, ", range2
        rr1, rr2 = [], []
        limlim = (int(np.ceil(self.exed.angle_limit[0])), int(np.floor(self.exed.angle_limit[1])))
        for magf in range(limlim[0], limlim[1] + 1):
            if not samp.orient(param["Sample"]["u"], param["Sample"]["v"], -magf, gonio):
                plot.error(outFile, self.figure)
                return False
            u, v = samp.u, samp.v
            vecs = [u, v]
            markers = self.markers
            rfactor = self.redfactor
            endmarkers = []
            panels = exed.current_panels_at(magf)
            temp = []
            for i in panels:
                if i == []:
                    continue
                if (i.forward == forward):
                    temp.append(i)
            panels = temp
            rans = self.guess_hkl_range_3D(samp, panels, (vec1, vec2, origin), (lmin, lmax))
            rr1.append(rans[ind[0]])
            rr2.append(rans[ind[1]])
        ran1 = rr1[0]
        ran2 = rr2[0]
        for i in rr1[1:]:
            if ran1[0] > i[0]:
                ran1[0] = i[0]
            if ran1[1] < i[1]:
                ran1[1] = i[1]
        for i in rr2[1:]:
            if ran2[0] > i[0]:
                ran2[0] = i[0]
            if ran2[1] < i[1]:
                ran2[1] = i[1]
        if forward:
            part1 = np.linspace(ran1[0]-0.2, ran1[1]+0.2, int(round(100*self.multiplier)))
            part2 = np.linspace(ran2[0]-0.2, ran2[1]+0.2, int(round(100*self.multiplier)))
        else:
            part1 = np.linspace(ran1[0]-0.2, ran1[1]+0.2, int(round(100*self.multiplier)))
            part2 = np.linspace(ran2[0]-0.2, ran2[1]+0.2, int(round(100*self.multiplier)))
        if len(range1) == 2:
            part1 = np.linspace(range1[0], range1[1], int(round(100*self.multiplier)))
        if len(range2) == 2:
            part2 = np.linspace(range2[0], range2[1], int(round(100*self.multiplier)))
        ax, bx, vx = np.ix_(part1, part2, vec1)
        a2x, b2x, v2x = np.ix_(part1, part2, vec2)
        a3x, b3x, orr = np.ix_(part1, part2, origin)
        plotgrid = np.zeros((len(part1), len(part2), 3))
        plotgrid += ax * vx
        plotgrid += b2x * v2x
        plotgrid += orr
        images = []
        for magf in range(limlim[0], limlim[1] + 1):
            image = np.zeros((len(part1), len(part2)))
            if not samp.orient(param["Sample"]["u"], param["Sample"]["v"], -magf, gonio):
                plot.error(outFile, self.figure)
                return False
            u, v = samp.u, samp.v
            vecs = [u, v]
            markers = self.markers
            rfactor = self.redfactor
            endmarkers = []
            panels = exed.current_panels_at(magf)
            temp = []
            for i in panels:
                if i == []:
                    continue
                if (i.forward == forward):
                    temp.append(i)
            panels = temp
            rotmat = samp.themat
            for i in range(len(plotgrid)):
                points = rotate(rotmat, plotgrid[i])
                points[np.where(points[:,2] >= 0.0)] = 0.0
                qpoints = reverse_q_vector(points)
                # if not forward:
                #     qpoints[:,1] = np.fmod(qpoints[:,1] + 4*np.pi, 2*np.pi)
                lambdas = n_lambda_fromk(qpoints[:,0])
                mask = (lambdas >= lmin)*(lambdas <= lmax)
                anglemask = np.zeros(len(points))
                for k in range(len(anglemask)):
                    if mask[k]:
                        for p in panels:
                            anglemask[k] = max(p.point_in_range(qpoints[k]), anglemask[k])
                intensity = exed.current_spectrum(lambdas) * rfactor * anglemask
                image[i, :] = intensity
            images.append(image)
        tmarkers = self.markers
        endmarkers = []
        for i in tmarkers:
            endmarkers.append([i[np.array([ind[0], ind[1]])], None,
                                           0.0, 0.0, 0, i])
        markers = []
        for i in tmarkers:
            if (np.abs(i) < 1e-14).all():
                continue
            else:
                markers.append(i)
        if len(markers) > 0:
            for magf in range(limlim[0], limlim[1] + 1):
                samp.orient(param["Sample"]["u"], param["Sample"]["v"], -magf, gonio)
                ub = samp.themat
                realcoords = rotate(ub, np.row_stack(markers))
                spherical = reverse_q_vector(realcoords)
                for pum, i in enumerate(spherical):
                    alpha, beta = i[1], i[2]
                    # alpha = np.fmod(alpha + 4*np.pi, 2*np.pi)
                    # alpha = alpha - 2*np.pi*(alpha > 3.0/2.0*np.pi)
                    n = np.array([np.cos(alpha) * np.cos(beta), np.sin(beta), np.sin(alpha) * np.cos(beta)])
                    exists = False
                    for j in exed.current_panels_at(magf):
                        if j == []:
                            continue
                        temp = np.array([0.0, alpha, beta])
                        if j.point_in_range(temp):
                            exists = True
                    if abs(origin.sum()-markers[pum][perp]) < 1e-5:
                        wave = 1.0/i[0]
                        # print( "wave ", wave
                        if wave >= param["Lambda"]["Min"] and wave <= param["Lambda"]["Max"]:
                            spec = exed.current_spectrum(wave)*self.redfactor
                        else:
                            spec = 0.0
                        # print( "spec ", spec
                        if not exists:
                            spec = 0.0
                        if spec > endmarkers[pum][-3]:
                            endmarkers[pum] = [markers[pum][np.array([ind[0], ind[1]])], None,
                                           wave, spec, magf, markers[pum]]
                        # print( "output", endmarkers[-1]
        return images, part1, part2, endmarkers, symbol, samp.description(), plane

    def show_total_coverage(self, outFile = ""):
        self.update_intensity()
        if self.count == 3:
            plot.apology(outFile = outFile, fig = self.figure)
            return None
        picf, ax1f, ax2f, markf, symf, desf, planf = self.new_total_plane(forward = True)
        picb, ax1b, ax2b, markb, symb, desb, planb = self.new_total_plane(forward = False)
        if self.choppertype == "Resolution too high":
            desf = "INTENSITY INCORRECT\nPlease adjust the resolution settings"
        plot.plot_total_map((picf, picb), (ax1f, ax1b), (ax2f, ax2b), outFile = outFile, fig = self.figure,
                             markers = (markf, markb), symbol = symf, description = desf, plane = planf)
