#    This file is part of EXEQ.
#
#    EXEQ is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <https://www.gnu.org/licenses/>.
# 
# Copyright (C) Maciej Bartkowiak, 2014-2018, 2020

__doc__ = """
Matplotlib-based plotting routines for different
kinds of output from EXEQ.
Maciej Bartkowiak, 28 Mar 2014
"""

import numpy as np
import matplotlib
# matplotlib.use("GTK3Agg")
matplotlib.rcParams['legend.fontsize'] = 11
matplotlib.rcParams['legend.borderpad'] = 0.3
matplotlib.rcParams['legend.labelspacing'] = 0.2
matplotlib.rcParams['xtick.labelsize'] = 11
matplotlib.rcParams['ytick.labelsize'] = 11
matplotlib.rcParams['axes.titlesize'] = 12
matplotlib.rcParams['axes.labelsize'] = 11
matplotlib.rcParams['axes.grid'] = True
matplotlib.rcParams['font.size'] = 12
import matplotlib.pyplot as mpl
from mpl_toolkits.mplot3d import Axes3D
import matplotlib.colors as mc
from geometry import *
from units import *

deadzone_color = mc.rgb2hex((0.5, 0.6, 0.75))
# darkzone_color = mc.rgb2hex((0.68, 0.75, 0.72))

main_docstring = "Output of EXEQ 1.0, April 2020"

deadzone_color = mc.rgb2hex((0.88, 0.88, 0.90))
darkzone_color = mc.rgb2hex((0.93, 0.95, 0.93))
marksymbols = ['oy', 'vg', 'sb', 'dc', '^k']

def format_yticklabel(number):
    return '{0:.1e}'.format(number).replace('+0', '').replace('-0', '-')

cols = []
rgb = [np.array([1.0, 0.0, 0.0]), np.array([0.0, 1.0, 0.0]), np.array([0.0, 0.0, 1.0])]
for i in range(16):
    temp = np.array([0.0, 0.0, 0.0])
    if i < 3:
        temp += rgb[i%3]
    elif i <6:
        temp += rgb[i%3] + rgb[(i+1)%3]
    elif i < 9:
        temp += 0.3*rgb[i%3] + 0.7*rgb[(i+1)%3]
    elif i < 12:
        temp += 0.2*rgb[i%3] + 0.5*rgb[(i+1)%3] + 1.0*rgb[(i+2)%3]
    elif i < 15:
        temp += 0.1*rgb[i%3] + 0.8*rgb[(i+1)%3] + 0.8*rgb[(i+2)%3]
    else:
        temp += 0.6*rgb[i%3] + 0.6*rgb[(i+1)%3] + 0.7*rgb[(i+2)%3]
    cols.append(mc.rgb2hex((temp[0], temp[1], temp[2])))

def plot_eq_map(range1, range2, outFile = "", fig = None):
    if fig == None:
        fig = mpl.figure(figsize = [7.0, 6.0], dpi=75, frameon = False)
        trigger = True
    else:
        fig.clear()
        trigger = False
    ax = fig.add_subplot(111)
    ax.plot(range1[:,0], range1[:,1], 'b-', label = "Forward scattering")
    ax.plot(range2[:,0], range2[:,1], 'r-', label = "Backscattering")
    ax.set_xlabel("q [A$^{-1}$]")
    ax.set_ylabel("E [meV]")
    ax.legend(loc = 0)
    ax.grid(True)
    if not outFile:
        if trigger:
            mpl.show()
        else:
            fig.canvas.draw()
    else:
        mpl.savefig(outFile)
        mpl.close()

# Code sample for contour labels.       
#        for i in contours:
#            con = xxx.axes.contour(i[0]*10*i[1], [i[1]], extent = [xs[0],xs[-1],ys[-1],ys[0]], aspect = 'auto',
#                linestyles = 'solid', linewidths = 1.0, label = "$\lambda$="+str(i[1]) ) 
#            xxx.axes.clabel(con, fontsize = 8)

def error(outFile = "", fig = None):
    if fig == None:
        fig = mpl.figure(figsize = [7.0, 6.0], dpi=75, frameon = False)
        trigger = True
    else:
        fig.clear()
        trigger = False
    ax = fig.add_subplot(111)
    ax.plot([0.0, 1.0], [0.0, 1.0], '-k')
    ax.plot([0.0, 1.0], [1.0, 0.0], '-k')
    ax.set_title("The orientation of the sample was not quite right.")
    if not outFile:
        if trigger:
            mpl.show()
        else:
            fig.canvas.draw()
    else:
        mpl.savefig(outFile)
        mpl.close()

def apology(outFile = "", fig = None):
    if fig == None:
        fig = mpl.figure(figsize = [7.0, 6.0], dpi=75, frameon = False)
        trigger = True
    else:
        fig.clear()
        trigger = False
    ax = fig.add_subplot(111)
    ax.plot([0.0, 1.0], [0.0, 1.0], '-k')
    ax.plot([0.0, 1.0], [1.0, 0.0], '-k')
    ax.set_title("Sorry, this kind of plot is not implemented.")
    ax.text(0.1, 0.9, "Please change the h, k, l values.")
    if not outFile:
        if trigger:
            mpl.show()
        else:
            fig.canvas.draw()
    else:
        mpl.savefig(outFile)
        mpl.close()
        
def apology2(outFile = "", fig = None):
    if fig == None:
        fig = mpl.figure(figsize = [7.0, 6.0], dpi=75, frameon = False)
        trigger = True
    else:
        fig.clear()
        trigger = False
    ax = fig.add_subplot(111)
    ax.plot([0.0, 1.0], [0.0, 1.0], '-k')
    ax.plot([0.0, 1.0], [1.0, 0.0], '-k')
    ax.set_title("We apologise for a software glitch.")
    ax.text(0.1, 0.9, "The input caused a problem in the QHull library.")
    ax.text(0.1, 0.8, "In the future QHull will be replaced in our code.")
    ax.text(0.1, 0.7, "For now, please change the input parameters.")
    ax.text(0.1, 0.6, "A different (h, k, l) plane could help the most.")
    if not outFile:
        if trigger:
            mpl.show()
        else:
            fig.canvas.draw()
    else:
        mpl.savefig(outFile)
        mpl.close()

def plot_help_message(outFile = "", fig = None):
    if fig == None:
        fig = mpl.figure(figsize = [7.0, 6.0], dpi=75, frameon = False)
        trigger = True
    else:
        fig.clear()
        trigger = False
    ax = fig.add_subplot(111)
    ax.plot([0.0, 0.0, 1.0, 1.0, 0.0], [0.0, 1.0, 1.0, 0.0, 0.0], '-k')
    ax.grid(False)
    ax.tick_params(bottom = 'off', top = 'off', left = 'off', right = 'off',
                   labelbottom = 'off', labeltop = 'off', labelleft = 'off', labelright = 'off')
    ax.set_title("EXEQ, the EXED (E,q)-range calculator TIPS:")
    ax.text(0.1, 0.9, "1. The interface is full of hovertext comments (tooltips).")
    ax.text(0.1, 0.8, "2. Main functionality is the reciprocal-space mapping.")
    ax.text(0.1, 0.7, "3. By putting 'h', 'k', 'l' in the h, k, l fields you can make a 3D-plot.")
    ax.text(0.1, 0.6, "4. Wrong input parameters will be replaced with default values.")
    ax.text(0.1, 0.5, "5. Desired resolution in $\\frac{\Delta \lambda}{\lambda}$ determines the 1st chopper speed.")
    ax.text(0.1, 0.4, "6. The bandwidth determines the last chopper speed.")
    ax.text(0.1, 0.3, "7. The chopper speeds affect the calculated neutron flux.")
    ax.text(0.1, 0.2, "8. After finding the peak in the hkl-map, you can see it on the detector.")
    ax.text(0.1, 0.1, "9. The total-range plot has been disabled in 3D, but improved in 2D.")
    if not outFile:
        if trigger:
            mpl.show()
        else:
            fig.canvas.draw()
    else:
        mpl.savefig(outFile)
        mpl.close()
        
def plot_welcome_message_normal(outFile = "", fig = None):
    if fig == None:
        fig = mpl.figure(figsize = [7.0, 6.0], dpi=75, frameon = False)
        trigger = True
    else:
        fig.clear()
        trigger = False
    ax = fig.add_subplot(111)
    for i in np.arange(0.0, 0.1, 0.01):
        ax.plot([i, i, 1.0-i, 1.0-i, i], [i, 1.0-i, 1.0-i, i, i], '-', color = (0.2, 1.0-5*i, 0.1 + 5*i),
                linewidth = 3.0)
    ax.grid(False)
    ax.set_xlim([0.0, 1.0])
    ax.set_ylim([0.0, 1.0])
    ax.tick_params(bottom = 'off', top = 'off', left = 'off', right = 'off',
                   labelbottom = 'off', labeltop = 'off', labelleft = 'off', labelright = 'off')
    ax.set_title("EXEQ, the EXED (E,q)-range calculator, version 0.17")
    ax.text(0.12, 0.70, "The 1-hour fix release, improved in no time at all.\n"+
                      "This software has been developed at Helmholtz-Zentrum Berlin,\n"+
                      "in order to assist the users of the Extreme-Environment Diffractometer (EXED)\n"+
                      "combined with the High-Field Magnet (HFM).")
    ax.text(0.12, 0.50, "The HFM may apply over 26 T constant magnetic field to the sample mounted inside.\n"+
                      "However, at the same time it limits the available space around the sample.\n"+
                      "It becomes necessary to plan the sample alignment carefully, so that the desired\n"+
                      "Bragg reflections may be observed in the experiment.")
    ax.text(0.12, 0.20, "We encourage EXED users to specify the lattice parameters of their sample,\n"+
                       "and try out different sample orientations (by means of $\\vec{Bu}$ and $\\vec{Bv}$),\n"+
                       "to see if the reflections of interest are within the instrument's range.\n"+
                       "The magnet may be rotated by at most 15$^{\circ}$. Also, the choice of the neutron wavelenght band\n"+
                       "is essential for the measurement.")
    ax.annotate("For a few usage tips, click 'Help!'", (0.8, 0.99), xytext = (0.58, 0.16),
                arrowprops={'arrowstyle':'-|>', 'mutation_scale':30, 'linewidth':3.0})
                #, 'shrinkA':5, 'shrinkB':7, 'mutation_scale':20, 'mutation_aspect':1.3})
                #, 'width':1.5, 'frac':0.2, 'headwidth':5})
    if not outFile:
        if trigger:
            mpl.show()
        else:
            fig.canvas.draw()
    else:
        mpl.savefig(outFile)
        mpl.close()

def plot_welcome_message(outFile = "", fig = None):
    if fig == None:
        fig = mpl.figure(figsize = [7.0, 6.0], dpi=75, frameon = False)
        trigger = True
    else:
        fig.clear()
        trigger = False
    ax = fig.add_subplot(111)
    for i in np.arange(0.0, 0.1, 0.01):
        ax.plot([i, i, 1.0-i, 1.0-i, i], [i, 1.0-i, 1.0-i, i, i], '-', color = (0.8-4*i, 1.0-5*i, 0.1 + 5*i),
                linewidth = 3.0)
    ax.grid(False)
    ax.set_xlim([0.0, 1.0])
    ax.set_ylim([0.0, 1.0])
    ax.tick_params(bottom = 'off', top = 'off', left = 'off', right = 'off',
                   labelbottom = 'off', labeltop = 'off', labelleft = 'off', labelright = 'off')
    ax.set_title("EXEQ, the EXED (E,q)-range calculator, version 1.0")
    ax.text(0.12, 0.70, "This is the final release of the software.\n"+
                      "Source code is available under GNU GPL v3.\n"+
                      "Written at Helmholtz Zentrum Berlin.\n"+
                      "Main author: Maciej Bartkowiak.")
    ax.text(0.12, 0.40, "This is the version written in Python 3,\n"+
                      "and using PyQt 5 for the GUI.\n"+
                      "The binary version was created with PyInstaller.\n")
    ax.annotate("For a few usage tips, click 'Help!'", (0.8, 0.99), xytext = (0.58, 0.16),
                arrowprops={'arrowstyle':'-|>', 'mutation_scale':30, 'linewidth':3.0})
                #, 'shrinkA':5, 'shrinkB':7, 'mutation_scale':20, 'mutation_aspect':1.3})
                #, 'width':1.5, 'frac':0.2, 'headwidth':5})
    if not outFile:
        if trigger:
            mpl.show()
        else:
            fig.canvas.draw()
    else:
        mpl.savefig(outFile)
        mpl.close()
        
def plot_uv_range(a, outFile = "", fig = None):
    if fig == None:
        fig = mpl.figure(figsize = [7.0, 6.0], dpi=75, frameon = False)
        trigger = True
    else:
        fig.clear()
        trigger = False
    ax = fig.add_subplot(111)
    for i in np.arange(0.0, 0.0001, 0.01):
        ax.plot([i, i, 1.0-i, 1.0-i, i], [i, 1.0-i, 1.0-i, i, i], '-', color = (0.2, 1.0-5*i, 0.1 + 5*i),
                linewidth = 3.0)
    ax.grid(False)
    ax.set_xlim([0.0, 1.0])
    ax.set_ylim([0.0, 1.0])
    ax.tick_params(bottom = 'off', top = 'off', left = 'off', right = 'off',
                   labelbottom = 'off', labeltop = 'off', labelleft = 'off', labelright = 'off')
    newumax, newumin, newvmax, newvmin = a[0], a[1], a[2], a[3]
    ax.set_title("Sample orientation: $\\vec{u}$ and $\\vec{v}$")
    ax.text(0.12, 0.8, "$\\vec{Bu}$ and $\\vec{Bv}$ define the orientation of the sample in respect to the magnetic field direction.\n"+
                       "The $\\vec{u}$ and $\\vec{v}$, which are relative to the incoming beam, change with the magnet rotation angle.")
    ax.text(0.12, 0.5, "For the current settings, at 15$^{\circ}$ magnet rotation, the vectors are:\n"+
                      "$\\vec{u}$ = "+newumax+"\n"+
                      "$\\vec{v}$ = "+newvmax)
    ax.text(0.12, 0.25, "For the current settings, at 0$^{\circ}$ magnet rotation, the vectors are:\n"+
                      "$\\vec{u}$ = "+newumin+"\n"+
                      "$\\vec{v}$ = "+newvmin)
    if not outFile:
        if trigger:
            mpl.show()
        else:
            fig.canvas.draw()
    else:
        mpl.savefig(outFile)
        mpl.close()
    
def plot_instrument(ori, tubes, a, outFile = "", fig = None, goniometer = [0.0,0.0,0.0]):
    ori *= -1
    # standard beginning for 3D plots
    axes = []
    if fig == None:
        fig = mpl.figure(figsize = [2 * 7.0, 6.0], dpi=75, frameon = False)
        trigger = True
    else:
        fig.clear()
        trigger = False
    for i in range(2):
        axes.append(fig.add_subplot(100 + 10*2 + i+1, projection='3d'))
       #fig, axes = mpl.subplots(1, len(axs), figsize = [len(axs) * 8.0, 6.0], frameon = False)
    # fig.subplots_adjust(wspace = 0, hspace = 0)
    # Now the real thing
    umax, umin, vmax, vmin, ucurr, vcurr = a[0], a[1], a[2], a[3], a[4], a[5]
    magfield =  np.row_stack([3.3*np.array([-np.sin(np.radians(ori)), 0.0, -np.cos(np.radians(ori))]),
                      4.5*np.array([np.sin(np.radians(ori)), 0.0, np.cos(np.radians(ori))])])
    # magfield = 5* np.array([[-np.sin(np.radians(ori)), 0.0, -np.cos(np.radians(ori))],
    #                         [np.sin(np.radians(ori)), 0.0, np.cos(np.radians(ori))]])
    fulltubes, goodtubes, shadowtubes = tubes[0], tubes[1], tubes[2]
    phi, chi, omega = goniometer[0], goniometer[1], goniometer[2]
    vecu, vecv = np.array([1.2, 0.0, 0.0]), np.array([0.0, 1.2, 0.0])
    # vecbu, vecbv = np.array([1.2, 0.0, 0.0]), np.array([0.0, 1.2, 0.0])
    rmat = rotZ(ori)
    axis_phi, axis_chi, axis_omega = np.array([1.0, 0.0, 0.0]), np.array([0.0, 1.0, 0.0]), np.array([0.0, 0.0, 1.0])
    axis_phi = np.dot(rmat, axis_phi)
    axis_chi = np.dot(rmat, axis_chi)
    axis_omega = np.dot(rmat, axis_omega)
    rmat1 = arb_rotation(axis_phi, phi)
    axis_chi = np.dot(rmat1, axis_chi)
    axis_omega = np.dot(rmat1, axis_omega)
    rmat2 = arb_rotation(axis_chi, chi)
    axis_omega = np.dot(rmat2, axis_omega)
    rmat3 = arb_rotation(axis_omega, -omega)
    for i in range(len(axes)):
        axes[i].view_init(45, 180.0-45*(i))
        axes[i].plot((-5, 5), (0, 0), (0, 0), '--', c = 'g', label = 'Direct beam')
        #for t in fulltubes:
            #if len(t) > 1:
                #axes[i].plot(t[:,2], t[:,0], t[:,1], c = deadzone_color)
        for t in goodtubes:
            if len(t) > 1:
                if t[:, 2].mean() > 0.0:
                    axes[i].plot(t[:,2], t[:,0], t[:,1], c = 'r')
        axes[i].plot(magfield[:,2], magfield[:,0], magfield[:,1], c = 'b', label = '$\\vec{B}$')
        for t in goodtubes:
            if len(t) > 1:
                if t[:, 2].mean() <= 0.0:
                    axes[i].plot(t[:,2], t[:,0], t[:,1], c = 'r')
        #for t in shadowtubes:
        #    if len(t) > 1:
        #        axes[i].plot(t[:,2], t[:,0], t[:,1], c = 'g')
        axes[i].scatter([0.0], [0.0], [0.0], color = 'g', s = 80, label = "Sample")
        axes[i].grid(False)
        axes[i].set_zlim([-2.0, 2.0])
        axes[i].set_ylim([-3.0, 3.0])
        axes[i].set_xlim([-3.0, 3.0])
        axes[i].set_xlabel("Length [m]")
        axes[i].set_ylabel("Width [m]")
        axes[i].set_zlabel("Height [m]")
        # new part
        rmat = rotZ(-ori)
        # rmat = rotZ(ori)
        endu, endv = rotate(rmat, np.array([1.0, 0.0, 0.0])), rotate(rmat,np.array([0.0, 1.0, 0.0]))
        vecbu = np.dot(rmat1, endu)
        vecbv = np.dot(rmat1, endv)
        vecbu = np.dot(rmat2, vecbu)
        vecbv = np.dot(rmat2, vecbv)
        endu = np.dot(rmat3, vecbu)
        endv = np.dot(rmat3, vecbv)
        # axes[i].quiver(xs, yz, zs, us, vs, ws) # {'length':0.8, 'label':'\\vec{u}'})
        axes[i].plot([0.0, vecu[0]], [0.0, vecu[1]], [0.0, vecu[2]], color='k', linewidth = 4.0, label = '$\\vec{u}$')
        axes[i].plot([0.0, vecv[0]], [0.0, vecv[1]], [0.0, vecv[2]], color='m', linewidth = 4.0, label = '$\\vec{v}$')
        axes[i].plot([0.0, 0.9*endu[0]], [0.0, 0.9*endu[1]], [0.0, 0.9*endu[2]], linewidth = 3.0, color='0.55', label = '$\\vec{Bu}$')
        axes[i].plot([0.0, 0.9*endv[0]], [0.0, 0.9*endv[1]], [0.0, 0.9*endv[2]], linewidth = 3.0, color='r', label = '$\\vec{Bv}$')
        # end of new part
        axes[i].legend(loc = 3)
#        ax.tick_params(bottom = 'off', top = 'off', left = 'off', right = 'off',
#                   labelbottom = 'off', labeltop = 'off', labelleft = 'off', labelright = 'off')
        #if i == 0:
            #axes[i].set_title("EXED orientation, view " + str(i+1) + "\n"
                          #+ "$\\vec{u}_{max}$=" + umax +", $\\vec{u}_{min}$=" + umin + ", $\\vec{u}_{current}$="+ucurr +"\n"+
                          #"")
        #else:
            #axes[i].set_title("EXED orientation, view " + str(i+1) + "\n"
                          #+"\n"+
                          #"$\\vec{v}_{max}$=" + vmax +", $\\vec{v}_{min}$=" + vmin + ", $\\vec{v}_{current}$="+vcurr)
    topbar_title = ("EXED orientation view\n"
                  + "$\\vec{u}_{current}$="+ucurr +"\n"+
                  "$\\vec{v}_{current}$="+vcurr)
    axtext_up = mpl.axes([0.15, 0.82, 0.65, 0.01], frameon = False)
    # fig.subplots_adjust(left=0.1, bottom=0.10, top=0.8, wspace = 0.35)
    axtext_up.set_yticks([])
    axtext_up.set_xticks([])
    axtext_up.set_title(topbar_title)
    # standard ending
    axtext = mpl.axes([0.15, 0.01, 0.65, 0.01], frameon = False)
    fig.subplots_adjust(left=0.1, bottom=0.10, top=0.8, wspace = 0.35)
    axtext.set_yticks([])
    axtext.set_xticks([])
    axtext.set_title(main_docstring)
    if not outFile:
        if trigger:
            mpl.show()
        else:
            fig.canvas.draw()
    else:
        mpl.savefig(outFile)
        mpl.close()
    
def plot_total_maps(array, axs, symbol = "hk", description = "", plane = 'hk0', outFile = "", fig = None,
                  markers = [], labels = []):
    plane = " ".join([str(x)[:4] for x in plane])
    axes = []
#    for i in range(len(axs)):
#        axes.append(1)
    if fig == None:
        fig = mpl.figure(figsize = [len(axs) * 7.0, 6.0], dpi=75, frameon = False)
        trigger = True
    else:
        fig.clear()
        trigger = False
    for i in range(len(axs)):
        axes.append(fig.add_subplot(100 + 10*len(axs) + i+1))
    # fig, axes = mpl.subplots(1, len(axs), figsize = [len(axs) * 7.0, 6.0], frameon = False)
    # fig.subplots_adjust(wspace = 0, hspace = 0)
    for i in range(len(axes)):
        ax1, ax2 = axs[i][0], axs[i][1]
        leg = ([],[])
        for j in range(len(array[i])):
            b = axes[i].contourf(np.abs(array[i][j].T[:,:]), [1.0, 0.01, 0.0001, 1e-7][::-1],
                        extent = [ax1[0], ax1[-1], ax2[0],ax2[-1]],#  aspect = 'auto',
                        label = labels[j], 
                        colors = [cols[j%len(cols)], cols[(j+1)%len(cols)], cols[(j+2)%len(cols)], 'k'][::-1])
            temp = b.legend_elements()[0][0]
                # temp.set_height(0.25)
                # temp.set_bounds(0.2, 0.2, 0.4, 0.4)
                # temp.set_clip_box(Bbox(points = np.array([[0.3, 0.3],[0.4,0.4]])))
            leg[0].append(temp)
            leg[1].append(labels[j])
        ms = markers[i]
        if len(ms) > 0:
            #if array[i].max() > 0.0:
                #ms[:,2] /= 2 * array[i].max()
            #elif ms[:,2].max() > 0.0:
                #ms[:,2] /= 2 * ms[:,2].max()
            for u in ms:
                if u[2] == 0.0:
                    mark = axes[i].plot(u[0], u[1], 'v', color='0.8')
                else:
                    mark = axes[i].plot(u[0], u[1], 'v', color=(0.1, min(0.5 + u[2], 1.0), 0.1), markersize = 6.0 + 4)
                    axes[i].text(u[0], u[1], '  Int. = ' + str(u[2]).split(".")[0])
    # mpl.plot(otherpoints[:,0], otherpoints[:,1], 'o')
        axes[i].set_ylabel(symbol[1])
        axes[i].set_xlabel(symbol[0])
        axes[i].grid(True)
        tpos_x = axes[i].get_xlim()[0]
        ty1, ty2 = axes[i].get_ylim()
        tpos_y = ty2 + 0.05 * (ty2-ty1)
        if not outFile:
            if i == 0:
                axes[i].text(tpos_x, tpos_y, "Forward scattering\n" + plane + " plane" + description)
            else:
                axes[i].text(tpos_x, tpos_y, "Backward scattering\n" + plane + " plane" + description)
            fig.subplots_adjust(left=0.1, bottom=0.1, top=0.8)
            # axlabel = mpl.axes([0.1, 1.0, 0.8, 0.2])
            # axlabel.text(0.0, 0.0, plane + " plane\n" + description)
#            mpl.show()
            # mpl.show(block = False)
        else:
            if i == 0:
                axes[i].text(tpos_x, tpos_y, "Forward scattering\n" + plane + " plane" + description)
            else:
                axes[i].text(tpos_x, tpos_y, "Backward scattering\n" + plane + " plane" + description)
            fig.subplots_adjust(left=0.1, bottom=0.10, top=0.8)
            ## axes[i].set_title(plane + " plane")
#            mpl.savefig(outFile, bbox_inches = 'tight')
    axtext = mpl.axes([0.15, 0.01, 0.65, 0.01], frameon = False)
    axtext.set_yticks([])
    axtext.set_xticks([])
    axtext.set_title(main_docstring)
    if not outFile:
#        mpl.title(plane + " plane\n" + description)
        # mpl.show()
        if trigger:
            mpl.show()
        else:
            fig.canvas.draw()
        # mpl.show(block = False)
    else:
#        mpl.title(plane + " plane")
        mpl.savefig(outFile, bbox_inches = 'tight')
        mpl.close()

def plot_total_vols(arrays, axs, symbol = "hk", description = "", plane = 'hk0', outFile = "", fig = None,
                  markers = [], labels = []):
    plane = " ".join([str(x)[:4] for x in plane])
    axes = []
    if fig == None:
        fig = mpl.figure(figsize = [len(axs) * 7.0, 6.0], dpi=75, frameon = False)
        trigger = True
    else:
        fig.clear()
        trigger = False
    for i in range(len(axs)):
        axes.append(fig.add_subplot(100 + 10*len(axs) + i+1, projection='3d'))
       #fig, axes = mpl.subplots(1, len(axs), figsize = [len(axs) * 8.0, 6.0], frameon = False)
    # fig.subplots_adjust(wspace = 0, hspace = 0)
    for i in range(len(axes)):
    # debug:
    # for i in range(1):
        ax1, ax2, ax3 = axs[i][0], axs[i][1], axs[i][2]
        axes[i].view_init(20, 60)
        mxx = 0.0
        for n, array in enumerate(arrays[i]):
            # con = axes[i].plot_wireframe(array[0], array[1], array[2])
            if len(array[3]) < 1:
                continue
            normalised = array[3]/(array[3].max())
            mxx = max(array[3].max(), mxx)
            con = axes[i].scatter(array[0], array[1], array[2], s = 40*normalised, c=cols[n%len(cols)])
        ms = markers[i]
        if len(ms) > 0:
            if mxx > 0.0:
                ms[:,3] /= 2*mxx
            for u in ms:
                if u[3] == 0.0:
                    mark = axes[i].scatter(u[0], u[1], u[2], c='0.8')
                else:
                    mark = axes[i].scatter(u[0], u[1], u[2], c=(0.1, min(0.5 + u[3], 1.0), 0.1), s = 40.0 + 6*u[3])
        #if len(markers) > 0:
            #for u in markers:
                #mark = axes[i].scatter(u[0], u[1], u[2], s = 40, c='0.8')
        # mpl.plot(otherpoints[:,0], otherpoints[:,1], 'o')
        axes[i].set_ylabel(symbol[1])
        axes[i].set_xlabel(symbol[0])
        axes[i].set_zlabel(symbol[2])
        axes[i].set_xlim((ax1[0], ax1[-1]))
        axes[i].set_ylim((ax2[0], ax2[-1]))
        axes[i].set_zlim((ax3[0], ax3[-1]))
        axes[i].grid(True)
        xyz = (ax1, ax2, ax3)
        for k in range(3):
            x, y = xyz[k%3], xyz[(k+2)%3]
            X, Y = np.meshgrid(np.arange(np.round(x[0]-0.5), x[-1]+0.5, 1.0),
                               np.arange(np.round(y[0]-0.5), y[-1]+0.5, 1.0))
            Z = X.copy()
            Z = 0.0
            XYZ = (X, Y, Z)
            axes[i].plot_wireframe(XYZ[k%3], XYZ[(k+2)%3], XYZ[(k+1)%3], linewidth = 0.5, color = '0.5')
        if not outFile:
            if i == 0:
                axes[i].set_title("Forward scattering\n" + plane + " plane" + description)
            else:
                axes[i].set_title("Backward scattering\n" + plane + " plane" + description)
            fig.subplots_adjust(left=0.1, bottom=0.1, top=0.8)
    #            mpl.show()
                # mpl.show(block = False)
        else:
            if i == 0:
                axes[i].set_title("Forward scattering\n" + plane + " plane" + description)
            else:
                axes[i].set_title("Backward scattering\n" + plane + " plane" + description)
            fig.subplots_adjust(left=0.1, bottom=0.10, top=0.80)
    #            mpl.savefig(outFile, bbox_inches = 'tight')
    axtext = mpl.axes([0.15, 0.01, 0.65, 0.01], frameon = False)
    axtext.set_yticks([])
    axtext.set_xticks([])
    axtext.set_title(main_docstring)
    if not outFile:
#        mpl.title(plane + " plane\n" + description)
        # mpl.show()
        if trigger:
            mpl.show()
        else:
            fig.canvas.draw()
        # mpl.show(block = False)
    else:
#        mpl.title(plane + " plane")
        mpl.savefig(outFile, bbox_inches = 'tight')
        mpl.close()

def plot_contours(contours, symbol = "hk", description = "", plane = 'hk0', outFile = "", fig = None,
                  markers = []):
    plane = " ".join([str(x)[:4] for x in plane])
    axes = []
    if fig == None:
        fig = mpl.figure(figsize = [len(contours) * 7.0, 6.0], dpi=75, frameon = False)
        trigger = True
    else:
        fig.clear()
        trigger = False
    for i in range(len(contours)):
        axes.append(fig.add_subplot(100 + 10*len(contours) + i+1))
#    axes.append(fig.add_subplot(111))
    for i in range(len(axes)):
        for con in contours[i]:
            axes[i].plot(con[:,0], con[:,1])
        axes[i].grid(True)
    if not outFile:
#        mpl.title(plane + " plane\n" + description)
        # mpl.show()
        if trigger:
            mpl.show()
        else:
            fig.canvas.draw()
        # mpl.show(block = False)
    else:
#        mpl.title(plane + " plane")
        mpl.savefig(outFile, bbox_inches = 'tight')
        mpl.close()

def plot_detector_panels(ori, tubepack,
                         outFile = "", fig = None, goniometer = [0.0,0.0,0.0],
                         markers = [], choppers_wrong = False):
    tubes, goodtubes, shadowtubes = tubepack[0], tubepack[1], tubepack[2]
    if fig == None:
        fig = mpl.figure(figsize = [2 * 7.0, 8.0], dpi=75, frameon = False)
        trigger = True
    else:
        fig.clear()
        trigger = False
    axes = []
    for i in range(2):
        axes.append(fig.add_subplot(100 + 10*2 + i+1))    
        # Shrink current axis's height by 10% on the bottom
        box = axes[-1].get_position()
        axes[-1].set_position([box.x0, box.y0 + box.height * 0.2,
                 box.width, box.height * 0.8])
    for t in tubes:
        if len(t) > 1:
            w = np.row_stack([t, t[0]])
            xx = np.degrees(tubepoints(w)[:,1])
            # xx += 360.0 * (xx < -180.0)
            if (w[:,2] > 0.0).any():
                axes[0].plot(xx - 90.0, w[:,1], c = deadzone_color)
            else:
                axes[1].plot(xx - 90.0, w[:,1], c = deadzone_color)
    for t in shadowtubes:
        if len(t) > 1:
            w = np.row_stack([t, t[0]])
            xx = np.degrees(tubepoints(w)[:,1])
            # xx += 360.0 * (xx < -180.0)
            if (w[:,2] > 0.0).any():
                axes[0].plot(xx - 90.0, w[:,1], c = 'g')
            else:
                axes[1].plot(xx - 90.0, w[:,1], c = 'g')
    for t in goodtubes:
        if len(t) > 1:
            w = np.row_stack([t, t[0]])
            xx = np.degrees(tubepoints(w)[:,1])
            # xx += 360.0 * (xx < -180.0)
            if (w[:,2] > 0.0).any():
                axes[0].plot(xx - 90.0, w[:,1], c = 'r', linewidth = 2.0)
            else:
                axes[1].plot(xx - 90.0, w[:,1], c = 'r', linewidth = 2.0)
    symbolcount = 0
    for t in markers:
        xx = np.degrees(t[1][1])
        if (t[0][2] > 0.0):
            # xx = xx - 360.0*(xx > 90.0) + 360.0*(xx < -90.0)
            axes[0].plot(xx - 90.0, t[0][1], marksymbols[symbolcount],
                   label = str(t[-1]) + "\n$\lambda$ = " + str(t[2])[:6] + "$\AA{}$\nInt. = " + str(t[3]).split(".")[0])
            # axes[0].text(xx, t[0][1], "\n$\lambda$ = " + str(t[2])[:6] + "$\AA{}$\nInt. = " + str(t[3]).split(".")[0])
        else:
            # xx = xx - 360.0*(xx > -90.0)
            # xx = xx - 360.0*(xx > -90.0)
            axes[1].plot(xx - 90.0, t[0][1], marksymbols[symbolcount],
                   label = str(t[-1]) + "\n$\lambda$ = " + str(t[2])[:6] + "$\AA{}$\nInt. = " + str(t[3]).split(".")[0])
            # axes[1].text(xx, t[0][1], "\n$\lambda$ = " + str(t[2])[:6] + "$\AA{}$\nInt. = " + str(t[3]).split(".")[0])
        symbolcount += 1
    axes[0].grid(True)
    # axes.set_zlim([-2.0, 2.0])
    axes[0].set_ylim([-1.5, 1.5])
    # axes.set_xlim([-3.0, 3.0])
    axes[0].set_xlabel("Angle [$^{\circ}$]")
    axes[0].set_ylabel("Height [m]")
    # axes.set_zlabel("Height [m]")
    axes[1].grid(True)
    # axes.set_zlim([-2.0, 2.0])
    axes[1].set_ylim([-0.5, 0.5])
    # axes.set_xlim([-3.0, 3.0])
    axes[1].set_xlabel("Angle [$^{\circ}$]")
    axes[1].set_ylabel("Height [m]")
    # axes.set_zlabel("Height [m]")
    if choppers_wrong:
        axes[0].set_title("Wrong chopper/resolution settings, please adjust")
        axes[1].set_title("Wrong chopper/resolution settings, please adjust")
    axtext = mpl.axes([0.15, 0.01, 0.65, 0.01], frameon = False)
    axtext.set_yticks([])
    axtext.set_xticks([])
    axtext.set_title(main_docstring)
    fig.subplots_adjust(left=0.1, bottom=0.30, top=0.95, wspace = 0.25)
    # Put a legend below current axis
    axes[0].legend(loc='upper center', bbox_to_anchor=(0.5, -0.15),
                fancybox=True, shadow=True, ncol=3, scatterpoints = 1, numpoints = 1,
                fontsize = 9.0)
    axes[1].legend(loc='upper center', bbox_to_anchor=(0.5, -0.15),
                fancybox=True, shadow=True, ncol=3, scatterpoints = 1, numpoints = 1,
                fontsize = 9.0)
    if not outFile:
        if trigger:
            mpl.show()
        else:
            fig.canvas.draw()
    else:
        mpl.savefig(outFile, bbox_inches = 'tight')
        mpl.close()

def plot_simple_map(pic, ax1, ax2, outFile = "", fig = None, symbol = "hk", description = "", plane = 'hk0',
                    markers = [], shadows = ([],[])):
    if fig == None:
        fig = mpl.figure(figsize = [2 * 7.0, 8.0], dpi=75, frameon = False)
        trigger = True
    else:
        fig.clear()
        trigger = False
    axes = []
    # labels = ['Neutron flux [n/s/cm$^{2}$]', 'Angle 1', 'Angle 2']
    labels = ['Neutron flux [n/s/cm$^{2}$]','Neutron flux [n/s/cm$^{2}$]']
    symbolcount = 0
    handles = []
    ptlabels = []
    for i in range(len(pic)):
        axes.append(fig.add_subplot(100 + 10*(len(pic)) + i+1))
        # asp = abs((ax1[i][0] - ax1[i][-1]) / (ax2[i][-1]-  ax2[i][0]))
        ##xxx = axes[i].imshow(np.nan_to_num(pic[i].T[:,::-1]), extent = [ax1[i][-1], ax1[i][0],
                                            ##ax2[i][-1], ax2[i][0]], interpolation = 'bessel',
                                            ##cmap = mpl.get_cmap('OrRd'), aspect = 'auto')# , aspect = asp)#, aspect = 'auto')
        ##cb = mpl.colorbar(xxx, ax = xxx.axes, format = '%.1e', pad = 0.02)
        ##cb.set_label(labels[i])
        ##axes[i].contour(np.nan_to_num(pic[i].T[::-1,::-1]), [1e-3*np.nan_to_num(pic[i]).max()], extent = [ax1[i][-1], ax1[i][0],
                                   ##ax2[i][-1], ax2[i][0]], aspect = 'auto', linestyles = 'solid', linewidths = 1.0)
        ##norm1 = shadows[i].max()
        ##axes[i].contourf(np.nan_to_num(shadows[i].T[::-1,::-1]),[1.01*norm1, 0.5*norm1, 0.01*norm1, -0.01],
                        ##extent = [ax1[i][-1], ax1[i][0], ax2[i][-1], ax2[i][0]],
                        ##aspect = 'auto', linestyles = 'solid', linewidths = 1.0,
                        ##colors = (deadzone_color, deadzone_color,deadzone_color,deadzone_color))
        topval = np.nan_to_num(pic[i]).max()
        if topval == 0.0:
            topval = 1.0
        xxx = axes[i].imshow(np.nan_to_num(pic[i].T[::-1,:]), extent = [ax1[i][0], ax1[i][-1],
                                            ax2[i][0], ax2[i][-1]], interpolation = 'bessel',
                                            cmap = mpl.get_cmap('OrRd'), aspect = 'auto',
                                            vmin = 0.0, vmax = topval)# , aspect = asp)#, aspect = 'auto')
        cb = mpl.colorbar(xxx, ax = xxx.axes, format = '%.1e', pad = 0.02)
        cb.set_label(labels[i])
        axes[i].contour(np.nan_to_num(pic[i].T), [1e-3*np.nan_to_num(pic[i]).max()], extent = [ax1[i][0], ax1[i][-1],
                                   ax2[i][0], ax2[i][-1]], aspect = 'auto', linestyles = 'solid', linewidths = 1.0)
        norm1 = np.nan_to_num(shadows[i]).max()
        if not (norm1 == 0.0):
            try:
                axes[i].contourf(np.nan_to_num(shadows[i].T),[1.01*norm1, 0.5*norm1, 0.01*norm1],
                        extent = [ax1[i][0], ax1[i][-1], ax2[i][0], ax2[i][-1]],
                        aspect = 'auto', linestyles = 'solid', linewidths = 1.0,
                        colors = (deadzone_color, deadzone_color,deadzone_color,deadzone_color))
            except ValueError:
                axes[i].contourf(np.nan_to_num(shadows[i].T),[1.01*norm1, 0.5*norm1, 0.01*norm1][::-1],
                        extent = [ax1[i][0], ax1[i][-1], ax2[i][0], ax2[i][-1]],
                        aspect = 'auto', linestyles = 'solid', linewidths = 1.0,
                        colors = (deadzone_color, deadzone_color,deadzone_color))
        axes[i].grid(True)
        axes[i].set_xlabel(symbol[0])
        axes[i].set_ylabel(symbol[1])
        box = axes[-1].get_position()
        axes[-1].set_position([box.x0, box.y0 + box.height * 0.2,
                 box.width, box.height * 0.8])
        tpos_x = axes[i].get_xlim()[0]
        ty1, ty2 = axes[i].get_ylim()
        tpos_y = ty2 + 0.05 * (ty2-ty1)
        if i == 0:
            axtextf = mpl.axes([0.20, 0.81, 0.10, 0.01], frameon = False)
            axtextf.set_yticks([])
            axtextf.set_xticks([])
            axtextf.set_title("Forward scattering, " + plane + " plane\n" + description)
        else:
            axtextf = mpl.axes([0.65, 0.81, 0.10, 0.01], frameon = False)
            axtextf.set_yticks([])
            axtextf.set_xticks([])
            axtextf.set_title("Backward scattering, " + plane + " plane\n" + description)
        for t in markers[i]:
            if not t[3] == 0.0:
                #handles.append(axes[i].plot(t[0][0], t[0][1], marksymbols[symbolcount]))
                #ptlabels.append(str(t[-1]) + "\n$\lambda$ = " + str(t[2])[:6] + "$\AA{}$\nInt. = " + str(t[3]).split(".")[0])
                axes[i].plot(t[0][0], t[0][1], marksymbols[symbolcount],
                             label = str(t[-1]) + "\n$\lambda$ = " + str(t[2])[:6] + "$\AA{}$\nInt. = " + str(t[3]).split(".")[0])
                symbolcount += 1
            else:
                #handles.append(axes[i].plot(t[0][0], t[0][1], 'k.'))
                #ptlabels.append(str(t[-1]) + "\n$\lambda$ = " + str(t[2])[:6] + "$\AA{}$\nInt. = " + str(t[3]).split(".")[0])
                axes[i].plot(t[0][0], t[0][1], 'k.',
                             label = str(t[-1]) + "\n$\lambda$ = " + str(t[2])[:6] + "$\AA{}$\nInt. = " + str(t[3]).split(".")[0])
        symbolcount = 0
    fig.subplots_adjust(left=0.1, bottom=0.30, top=0.8, wspace = 0.25)
    axtext = mpl.axes([0.15, 0.01, 0.65, 0.01], frameon = False)
    axtext.set_yticks([])
    axtext.set_xticks([])
    axtext.set_title(main_docstring)
    #axes[0].legend(loc='upper center', bbox_to_anchor=(0.5, -0.05),
                #fancybox=True, shadow=True, ncol=5, scatterpoints = 1, numpoints = 1)
    axes[1].legend(loc='upper center', bbox_to_anchor=(-0.15, -0.15),
                fancybox=True, shadow=True, ncol=5, scatterpoints = 1, numpoints = 1)
    #mpl.figlegend(handles, ptlabels, loc='upper center', bbox_to_anchor=(0.5, -0.05),
                #fancybox=True, shadow=True, ncol=5, scatterpoints = 1, numpoints = 1)
    if not outFile:
        if trigger:
            mpl.show()
        else:
            fig.canvas.draw()
    else:
        mpl.savefig(outFile, bbox_inches = 'tight')
        mpl.close()

def plot_simple_volume(pic, ax1, ax2, ax3, outFile = "", fig = None, symbol = "hk", description = "", plane = 'hk0',
                    markers = []):
    if fig == None:
        fig = mpl.figure(figsize = [2 * 7.0, 8.0], dpi=75, frameon = False)
        trigger = True
    else:
        fig.clear()
        trigger = False
    axes = []
    # labels = ['Neutron flux [n/s/cm$^{2}$]', 'Angle 1', 'Angle 2']
    labels = ['Neutron flux [n/s/cm$^{2}$]','Neutron flux [n/s/cm$^{2}$]']
    symbolcount = 0
    handles = []
    ptlabels = []
    for i in range(len(pic)):
        axes.append(fig.add_subplot(100 + 10*(len(pic)) + i+1, projection='3d'))
        # asp = abs((ax1[i][0] - ax1[i][-1]) / (ax2[i][-1]-  ax2[i][0]))
        normalised = pic[i][0]#/(array[3].max())
        if len(pic[i][0]) > 0:
            mxx = pic[i][0].max()
        else:
            mxx = 1.0
        if mxx == 0.0:
            mxx = 1e7
        coords = pic[i][1]
        con = axes[i].scatter(coords[:,0], coords[:,1], coords[:,2], s = 40*normalised/mxx, c=normalised)
        cb = mpl.colorbar(con, ax = con.axes, format = '%.1e', pad = 0.02)
        cb.set_label(labels[i])
        axes[i].grid(True)
        axes[i].set_xlabel(symbol[0])
        axes[i].set_ylabel(symbol[1])
        axes[i].set_zlabel(symbol[2])
        box = axes[-1].get_position()
        axes[-1].set_position([box.x0, box.y0 + box.height * 0.2,
                 box.width, box.height * 0.8])
        tpos_x = axes[i].get_xlim()[0]
        ty1, ty2 = axes[i].get_ylim()
        tpos_y = ty2 + 0.05 * (ty2-ty1)
        if i == 0:
            axtextf = mpl.axes([0.20, 0.81, 0.10, 0.01], frameon = False)
            axtextf.set_yticks([])
            axtextf.set_xticks([])
            axtextf.set_title("Forward scattering, " + plane + " volume\n" + description)
        else:
            axtextf = mpl.axes([0.65, 0.81, 0.10, 0.01], frameon = False)
            axtextf.set_yticks([])
            axtextf.set_xticks([])
            axtextf.set_title("Backward scattering, " + plane + " volume\n" + description)
        for t in markers[i]:
            if not t[3] == 0.0:
                #handles.append(axes[i].plot(t[0][0], t[0][1], marksymbols[symbolcount]))
                #ptlabels.append(str(t[-1]) + "\n$\lambda$ = " + str(t[2])[:6] + "$\AA{}$\nInt. = " + str(t[3]).split(".")[0])
                axes[i].plot([t[0][0]], [t[0][1]], [t[0][2]],# s= 100,
                            c = 'g', marker = 's',
                             label = str(t[-1]) + "\n$\lambda$ = " + str(t[2])[:6] + "$\AA{}$\nInt. = " + str(t[3]).split(".")[0])
                symbolcount += 1
            else:
                #handles.append(axes[i].plot(t[0][0], t[0][1], 'k.'))
                #ptlabels.append(str(t[-1]) + "\n$\lambda$ = " + str(t[2])[:6] + "$\AA{}$\nInt. = " + str(t[3]).split(".")[0])
                axes[i].plot([t[0][0]], [t[0][1]], [t[0][2]],# s= 40,
                            c = 'k', marker = 'x',
                             label = str(t[-1]) + "\n$\lambda$ = " + str(t[2])[:6] + "$\AA{}$\nInt. = " + str(t[3]).split(".")[0])
        symbolcount = 0
    fig.subplots_adjust(left=0.1, bottom=0.30, top=0.8, wspace = 0.35)
    axtext = mpl.axes([0.15, 0.01, 0.65, 0.01], frameon = False)
    axtext.set_yticks([])
    axtext.set_xticks([])
    axtext.set_title(main_docstring)
    #axes[0].legend(loc='upper center', bbox_to_anchor=(0.5, -0.05),
                #fancybox=True, shadow=True, ncol=5, scatterpoints = 1, numpoints = 1)
    axes[1].legend(loc='upper center', bbox_to_anchor=(-0.15, -0.15),
                fancybox=True, shadow=True, ncol=5, scatterpoints = 1, numpoints = 1)
    #mpl.figlegend(handles, ptlabels, loc='upper center', bbox_to_anchor=(0.5, -0.05),
                #fancybox=True, shadow=True, ncol=5, scatterpoints = 1, numpoints = 1)
    if not outFile:
        if trigger:
            mpl.show()
        else:
            fig.canvas.draw()
    else:
        mpl.savefig(outFile, bbox_inches = 'tight')
        mpl.close()

def plot_total_map(pic, ax1, ax2, outFile = "", fig = None, symbol = "hk", description = "", plane = 'hk0',
                    markers = []):
    if fig == None:
        fig = mpl.figure(figsize = [2 * 7.0, 8.0], dpi=75, frameon = False)
        trigger = True
    else:
        fig.clear()
        trigger = False
    axes = []
    # labels = ['Neutron flux [n/s/cm$^{2}$]', 'Angle 1', 'Angle 2']
    labels = ['Neutron flux [n/s/cm$^{2}$]','Neutron flux [n/s/cm$^{2}$]']
    symbolcount = 0
    handles = []
    ptlabels = []
    for i in range(len(pic)):
        axes.append(fig.add_subplot(100 + 10*(len(pic)) + i+1))
        # asp = abs((ax1[i][0] - ax1[i][-1]) / (ax2[i][-1]-  ax2[i][0]))
        for j, pp in enumerate(pic[i]):
            pom = np.nan_to_num(pp.T)
            pommax = pom.max()
            if pommax == 0.0:
                pommax = 1.0
            axes[i].contourf(pom, [1.0*pommax, 0.5*pommax, 1e-3*pommax, 1e-8*pommax][::-1], extent = [ax1[i][0], ax1[i][-1],
                                    ax2[i][0], ax2[i][-1]], aspect = 'auto',
                                    colors = [cols[j%len(cols)], cols[j%len(cols)], cols[j%len(cols)], 'k'][::-1])
        axes[i].grid(True)
        axes[i].set_xlabel(symbol[0])
        axes[i].set_ylabel(symbol[1])
        box = axes[-1].get_position()
        axes[-1].set_position([box.x0, box.y0 + box.height * 0.2,
                 box.width, box.height * 0.8])
        tpos_x = axes[i].get_xlim()[0]
        ty1, ty2 = axes[i].get_ylim()
        tpos_y = ty2 + 0.05 * (ty2-ty1)
        if i == 0:
            axtextf = mpl.axes([0.20, 0.81, 0.10, 0.01], frameon = False)
            axtextf.set_yticks([])
            axtextf.set_xticks([])
            axtextf.set_title("Forward scattering, " + plane + " plane\n" + description)
        else:
            axtextf = mpl.axes([0.65, 0.81, 0.10, 0.01], frameon = False)
            axtextf.set_yticks([])
            axtextf.set_xticks([])
            axtextf.set_title("Backward scattering, " + plane + " plane\n" + description)
        for t in markers[i]:
            if not t[3] == 0.0:
                #handles.append(axes[i].plot(t[0][0], t[0][1], marksymbols[symbolcount]))
                #ptlabels.append(str(t[-1]) + "\n$\lambda$ = " + str(t[2])[:6] + "$\AA{}$\nInt. = " + str(t[3]).split(".")[0])
                axes[i].plot(t[0][0], t[0][1], marksymbols[symbolcount],
                             label = (str(t[-1]) + "'\nmag. rot. " + str(t[-2]) + "$^{\circ}$\n$\lambda$ = "
                                      + str(t[2])[:6] + "$\AA{}$\nInt. = " + str(t[3]).split(".")[0]))
                symbolcount += 1
            else:
                #handles.append(axes[i].plot(t[0][0], t[0][1], 'k.'))
                #ptlabels.append(str(t[-1]) + "\n$\lambda$ = " + str(t[2])[:6] + "$\AA{}$\nInt. = " + str(t[3]).split(".")[0])
                axes[i].plot(t[0][0], t[0][1], 'k.',
                             label = (str(t[-1]) + "\nmag. rot. " + str(t[-2]) + "$^{\circ}$\n$\lambda$ = "
                                      + str(t[2])[:6] + "$\AA{}$\nInt. = " + str(t[3]).split(".")[0]))
        symbolcount = 0
    fig.subplots_adjust(left=0.1, bottom=0.30, top=0.8, wspace = 0.25)
    axtext = mpl.axes([0.15, 0.01, 0.65, 0.01], frameon = False)
    axtext.set_yticks([])
    axtext.set_xticks([])
    axtext.set_title(main_docstring)
    #axes[0].legend(loc='upper center', bbox_to_anchor=(0.5, -0.05),
                #fancybox=True, shadow=True, ncol=5, scatterpoints = 1, numpoints = 1)
    axes[1].legend(loc='upper center', bbox_to_anchor=(-0.15, -0.15),
                fancybox=True, shadow=True, ncol=5, scatterpoints = 1, numpoints = 1,
                fontsize = 9.0)
    #mpl.figlegend(handles, ptlabels, loc='upper center', bbox_to_anchor=(0.5, -0.05),
                #fancybox=True, shadow=True, ncol=5, scatterpoints = 1, numpoints = 1)
    if not outFile:
        if trigger:
            mpl.show()
        else:
            fig.canvas.draw()
    else:
        mpl.savefig(outFile, bbox_inches = 'tight')
        mpl.close()
