#!/usr/bin/python

#    This file is part of EXEQ.
#
#    EXEQ is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <https://www.gnu.org/licenses/>.
# 
# Copyright (C) Maciej Bartkowiak, 2014-2018, 2020

import numpy as np

__doc__ = """
This module is meant to create an XML definition of the EXED detector,
of the kind that is used by Mantid.
The output is used by EXEQ, but the module itself is not accessed by
the EXED code in any way.
At the moment this script produces 16 files, each for a different position
of the High-Field Magnet, from 0 to 15 in 1-degree steps.
Maciej Bartkowiak, 21 Aug 2014
"""

inch = 2.54 * 10**(-2)
spacing = 0.1 * 10**(-2)
cone = np.radians(15)
width = 2.500
pixel = np.array((0.02,0.02))
oldpixel = np.array((0.5*inch,0.01))
panels = ['L', 'M', 'S']
tubes = [32, 16, 8]
distance = 2.500
height = 2*distance*np.tan(np.radians(15))
radius = height / 2.0
lengths = [height, height * 0.842, height * 0.65, 0.9]
centre = 0.0
l, s, m, x = 32, 8, 16, 0
output = 'EXED_detectors_straight.xml'

#
# x = int(np.radians(15)*distance/pixel[1])
x=0
#

sets = [
        (x -l -m -s/2, 'S', s),
        (x -l -m/2, 'M', m),
        (x -l/2, 'L', l),
#        (x/2, 'L', x),
        (x + l/2, 'L', l),
        (x + l + m/2, 'M', m),
        (x + l + m +s/2, 'S', s)
        ]

flatsets0 = [
        (np.radians(-172.0),),
        (np.radians(172.0),),
#        (np.radians(-160.0),),
        ]

frontsets = [
        (np.radians(-7.95),),
        (np.radians(7.95),),
#        (np.radians(-160.0),),
        ]

SANSsets = [
        (np.radians(-1.744),),
        (np.radians(6.2816),),
#        (np.radians(-160.0),),
        ]
        
flatsets15 = [
#        (np.radians(172.0),),
        (np.radians(-172.0),),
        (np.radians(-158.68),)
        ]
        
flatsetsm15 = [
#        (np.radians(172.0),),
        (np.radians(158.68),),
        (np.radians(172.0),)
        ]

#frontsets = [
        #(np.radians(8.5),),
        #(np.radians(-8.5),),
##        (np.radians(-160.0),),
        #]
        
#flatsets15 = [
##        (np.radians(172.0),),
        #(np.radians(-172.0),),
        #(np.radians(-157.7),)
        #]
        
#flatsetsm15 = [
##        (np.radians(172.0),),
        #(np.radians(157.7),),
        #(np.radians(172.0),)
        #]        
        
#def height(position):
#    angle = position / distance

header = """<?xml version="1.0" encoding="UTF-8"?>
<instrument name="EXED" date=""
valid-from="1900-01-31 23:59:59"
valid-to="2100-01-31 23:59:59"
last-modified="2013-08-16 13:19:03">
<defaults>
<length unit="meter"/>
<angle unit="degree"/>
<location x="0.0" y="0.0" z="0.0" rot="0.0" axis-x="0.0" axis-y="0.0" axis-z="1.0"/>
<reference-frame>
<!-- The z-axis is set parallel to and in the direction of the beam. the
     y-axis points up and the coordinate system is right handed. -->
<along-beam axis="z"/>
<pointing-up axis="y"/>
<handedness val="right"/>
<origin val="beam" />
</reference-frame>
<offsets spherical="delta" />
<components-are-facing x="0.0" y="0.0" z="0.0" />
</defaults>"""

post_header = """<!-- source and sample-position components -->
<component type="double_disk">
<location z="-53.063"><facing val="none"/></location>
</component>
<component type="sample-holder"><location><facing val="none"/></location>
</component>

<!-- Source types -->
<type name="double_disk" is="Source">
<properties />
<cylinder id="some-shape">
<centre-of-bottom-base r="0.0" t="0.0" p="0.0" />
<axis x="0.0" y="0.0" z="1.0" />
<radius val="0.01" />
<height val="0.03" />
</cylinder>
</type>

<!-- Sample-position types -->
<type name="sample-holder" is="SamplePos">
<properties />
<sphere id="some-shape">
<centre x="0.0" y="0.0" z="0.0" />
<radius val="0.03" />
</sphere>
<algebra val="some-shape" />
</type>

<!-- Detectors types -->
<type name="monitors" idlist="monitors">
<component type="monitor" is="monitor">
<location x="0" y="0" z="-0.973" name="monitor1" />
</component>
</type>"""

other_definitions = """<type name="monitor" is="detector">
<properties/>
<cylinder id="some-shape">
<centre-of-bottom-base r="0.0" t="0.0" p="0.0" />
<axis x="0.0" y="0.0" z="1.0" />
<radius val="0.05" />
<height val="0.02" />
</cylinder>
<algebra val="some-shape" />
</type>

<type name="pixel" is="detector">
<cylinder id="cyl-approx">
<centre-of-bottom-base x="0.0" y="0.0" z="0.0" />
<axis x="0.0" y="0.2" z="0.0" />
<radius val="0.0095" />
<height val="0.02" />
</cylinder>
<algebra val="cyl-approx" />
</type>

<type name="oldpixel" is="detector">
<cylinder id="cyl-approx">
<centre-of-bottom-base x="0.0" y="0.0" z="0.0" />
<axis x="0.0" y="0.2" z="0.0" />
<radius val="0.004" />
<height val="0.01" />
</cylinder>
<algebra val="cyl-approx" />
</type>
"""

monitor_id = """<!-- MONITOR ID LISTS -->
<idlist idname="monitors">
<id start="1601001" end="1601001" />
</idlist>
</instrument>"""

def rotY(coords, angle = -15.0):
    ang = np.radians(angle)
    rot1 = np.array([np.cos(ang), 0.0, np.sin(ang)])
    rot2 = np.array([0.0, 1.0, 0.0])
    rot3 = np.array([- np.sin(ang), 0.0, np.cos(ang)])
    result = coords.copy()
    result[0] = (coords*rot1).sum()
    result[1] = (coords*rot2).sum()
    result[2] = (coords*rot3).sum()
    return result

def make_tube(height, name = 'tube'):
    pixellist = []
    number = 0
    for i in np.arange(-height/2.0 + pixel[1]/2.0, height/2.0, pixel[1]):
        number += 1
        pixellist.append('<location x="0" y="'+str(i)+'" z="0" name="Det'+str(number)+'"/>"')
    return pixellist, number
    
def make_tube_old(height = 0.9, name = 'oldtube'):
    pixellist = []
    number = 0
    for i in np.arange(-height/2.0 + oldpixel[1]/2.0, height/2.0, oldpixel[1]):
        number += 1
        pixellist.append('<location x="0" y="'+str(i)+'" z="0" name="Det'+str(number)+'"/>"')
    return pixellist, number

def make_panel(ntubes, centre, starting_number, pixels_per_tube):
    limits = [(centre + ntubes/2)*pixel[0], (centre-ntubes/2)*pixel[0]]
    tubelist = []
    pnumber = starting_number
    tnumber = 1
    for i in np.arange(limits[1] + pixel[0]/2.0, limits[0], pixel[0]):
        angle = i / distance
        pos = [np.sin(angle)*distance, np.cos(angle)*distance]
        tubelist.append('<location x="'+str(pos[0])+'" y="0" z="'+str(pos[1])+'" name="Tube'+str(tnumber)+'"/>')
        pnumber += pixels_per_tube
        tnumber += 1
    return tubelist, tnumber, pnumber

def make_flat_panel(ntubes, centre, starting_number, pixels_per_tube):
    dist = 3.0
    angle = centre
    # cent = np.array([np.sin(angle)*dist, 0.0, np.cos(angle)*dist])
    cent = rotY(np.array([0.0, 0.0, dist]), np.degrees(angle))
    perp = rotY(cent, 90.0)
    perp /= np.sqrt((perp**2).sum())
    spac = perp.copy()
    perp *= oldpixel[0]
    spac *= spacing
    tubelist = []
    pnumber = starting_number
    tnumber = 1
    for i in np.linspace(-(ntubes-1)/2.0 , (ntubes-1)/2.0, 48):
        pos = cent + i * (perp + spac)
        tubelist.append('<location x="'+str(pos[0])+'" y="0" z="'+str(pos[2])+'" name="Tube'+str(tnumber)+'"/>')
        pnumber += pixels_per_tube
        tnumber += 1
    return tubelist, tnumber, pnumber    

def make_front_panel(ntubes, centre, starting_number, pixels_per_tube):
    dist = 2.5
    angle = centre
    # cent = np.array([np.sin(angle)*dist, 0.0, np.cos(angle)*dist])
    cent = rotY(np.array([0.0, 0.0, dist]), np.degrees(angle))
    perp = rotY(cent, 90.0)
    perp /= np.sqrt((perp**2).sum())
    spac = perp.copy()
    perp *= oldpixel[0]
    spac *= spacing
    tubelist = []
    pnumber = starting_number
    tnumber = 1
    for i in np.linspace(-(ntubes-1)/2.0 , (ntubes-1)/2.0, 48):
        pos = cent + i * (perp + spac)
        tubelist.append('<location x="'+str(pos[0])+'" y="0" z="'+str(pos[2])+'" name="Tube'+str(tnumber)+'"/>')
        pnumber += pixels_per_tube
        tnumber += 1
    return tubelist, tnumber, pnumber    

def make_SANS_panel(ntubes, centre, starting_number, pixels_per_tube):
    dist = 5.7
    angle = centre
    # cent = np.array([np.sin(angle)*dist, 0.0, np.cos(angle)*dist])
    cent = rotY(np.array([0.0, 0.0, dist]), np.degrees(angle))
    perp = rotY(cent, 90.0)
    perp /= np.sqrt((perp**2).sum())
    spac = perp.copy()
    perp *= oldpixel[0]
    spac *= spacing
    tubelist = []
    pnumber = starting_number
    tnumber = 1
    for i in np.linspace(-(ntubes-1)/2.0 , (ntubes-1)/2.0, 48):
        pos = cent + i * (perp + spac)
        tubelist.append('<location x="'+str(pos[0])+'" y="0" z="'+str(pos[2])+'" name="Tube'+str(tnumber)+'"/>')
        pnumber += pixels_per_tube
        tnumber += 1
    return tubelist, tnumber, pnumber    
    
def write_component(name = "bank", parts = [], part_type = "tube", ofile = ""):
    lines = []
    lines.append('<type name="'+name+'">')
    lines.append('<properties />')
    lines.append('<component type="'+part_type+'">')
    for i in lines:
        ofile.write(i + "\n")
    for i in parts:
        ofile.write(i + "\n")
    ofile.write('</component>')
    ofile.write('</type>' + "\n")

def idlist(name, start, increment, steps):
    result = []
    result.append('<idlist idname= "'+name+'">')
    beg, end = start, start + increment -1
    for i in range(steps):
        result.append('<id start="'+str(beg)+'" end="'+str(end)+'" />')
        beg += 1000
        end += 1000
    result.append('</idlist>')
    return result
    
def generate_bank(name = "bank0"):
    return '<component type="'+name+'" idlist="'+name+'"><location/></component>'

def generate_bank_head(name):
    lines = []
    pname = 'panel' + name[-1:]
    lines.append('<type name="'+pname+'" idlist="'+pname+'">')
    lines.append('<component type="'+name+'">')
    lines.append('<location x="0" y="0" z="0"><facing val="none"/></location>')
    lines.append('</component>')
    lines.append('</type>')
    return lines
    
def make_full_ins(angle, starting_number = 100000):
    if angle < -7.5:
        flatsets = flatsetsm15
    elif angle < 7.5:
        flatsets = flatsets0
    else:
        flatsets = flatsets15
    parts = ['<component type="monitors" idlist="monitors"><location/></component>']
    tubes = []
    small_tubes, n_small_tubes = make_tube(lengths[2], "tube_S")
    medium_tubes, n_medium_tubes = make_tube(lengths[1], "tube_M")
    large_tubes, n_large_tubes = make_tube(lengths[0], "tube_L")
    old_tubes, n_old_tubes = make_tube_old(lengths[3], "tube")
    tubes.append(["tube_S", small_tubes, "pixel"])
    tubes.append(["tube_M", medium_tubes, "pixel"])
    tubes.append(["tube_L", large_tubes, "pixel"])
    tubes.append(["tube", old_tubes, "oldpixel"])
    banks = []
    bank_heads = []
    idlists = []
    nn = starting_number
    it = 1
    for i in frontsets:
        center = i[0]
        center += np.radians(angle)
        name = 'bank' + str(it)
        ntubes = 48
        idlists.append(idlist(name, nn, n_old_tubes, ntubes))
        tl, tn, nn = make_front_panel(ntubes, center, nn, n_small_tubes)
        banks.append([name, tl, "tube"])
        parts.append(generate_bank(name))
        bank_heads.append(generate_bank_head(name))
        it += 1
        nn = starting_number * it
    for i in flatsets:
        center = i[0]
        name = 'bank' + str(it)
        ntubes = 48
        idlists.append(idlist(name, nn, n_old_tubes, ntubes))
        tl, tn, nn = make_flat_panel(ntubes, center, nn, n_small_tubes)
        banks.append([name, tl, "tube"])
        parts.append(generate_bank(name))
        bank_heads.append(generate_bank_head(name))
        it += 1
        nn = starting_number * it
    output = open("EXED_configuration_" + str(angle) + "degrees.xml", 'w')
    output.write(header + "\n")
    for i in parts:
        output.write(i + "\n")
    output.write(post_header + "\n")
    for i in bank_heads:
        for j in i:
            output.write(j + "\n")
    for i in banks:
        write_component(i[0], i[1], i[2], output)
    for i in tubes:
        write_component(i[0], i[1], i[2], output)
    output.write(other_definitions + "\n")
    for i in idlists:
        for j in i:
            output.write(j + "\n")
    output.write(monitor_id + "\n")
    output.close()

def make_forward_only(angle, starting_number = 100000):
    parts = ['<component type="monitors" idlist="monitors"><location/></component>']
    tubes = []
    small_tubes, n_small_tubes = make_tube(lengths[2], "tube_S")
    medium_tubes, n_medium_tubes = make_tube(lengths[1], "tube_M")
    large_tubes, n_large_tubes = make_tube(lengths[0], "tube_L")
    old_tubes, n_old_tubes = make_tube_old(lengths[3], "tube")
    tubes.append(["tube_S", small_tubes, "pixel"])
    tubes.append(["tube_M", medium_tubes, "pixel"])
    tubes.append(["tube_L", large_tubes, "pixel"])
    tubes.append(["tube", old_tubes, "oldpixel"])
    banks = []
    bank_heads = []
    idlists = []
    nn = starting_number
    it = 1
    for i in frontsets:
        center = i[0]
        center += np.radians(angle)
        name = 'bank' + str(it)
        ntubes = 48
        idlists.append(idlist(name, nn, n_old_tubes, ntubes))
        tl, tn, nn = make_front_panel(ntubes, center, nn, n_small_tubes)
        banks.append([name, tl, "tube"])
        parts.append(generate_bank(name))
        bank_heads.append(generate_bank_head(name))
        it += 2
        nn = starting_number * it
    output = open("EXED_FWpanels_" + str(angle) + "degrees.xml", 'w')
    output.write(header + "\n")
    for i in parts:
        output.write(i + "\n")
    output.write(post_header + "\n")
    for i in bank_heads:
        for j in i:
            output.write(j + "\n")
    for i in banks:
        write_component(i[0], i[1], i[2], output)
    for i in tubes:
        write_component(i[0], i[1], i[2], output)
    output.write(other_definitions + "\n")
    for i in idlists:
        for j in i:
            output.write(j + "\n")
    output.write(monitor_id + "\n")
    output.close()
    
def make_SANS_only(angle, starting_number = 100000):
    parts = ['<component type="monitors" idlist="monitors"><location/></component>']
    tubes = []
    small_tubes, n_small_tubes = make_tube(lengths[2], "tube_S")
    medium_tubes, n_medium_tubes = make_tube(lengths[1], "tube_M")
    large_tubes, n_large_tubes = make_tube(lengths[0], "tube_L")
    old_tubes, n_old_tubes = make_tube_old(lengths[3], "tube")
    tubes.append(["tube_S", small_tubes, "pixel"])
    tubes.append(["tube_M", medium_tubes, "pixel"])
    tubes.append(["tube_L", large_tubes, "pixel"])
    tubes.append(["tube", old_tubes, "oldpixel"])
    banks = []
    bank_heads = []
    idlists = []
    nn = starting_number
    it = 1
    for i in SANSsets:
        center = i[0]
        # center += np.radians(angle)
        name = 'bank' + str(it)
        ntubes = 48
        idlists.append(idlist(name, nn, n_old_tubes, ntubes))
        tl, tn, nn = make_SANS_panel(ntubes, center, nn, n_small_tubes)
        banks.append([name, tl, "tube"])
        parts.append(generate_bank(name))
        bank_heads.append(generate_bank_head(name))
        it += 2
        nn = starting_number * it
    output = open("EXED_SANSpanels.xml", 'w')
    output.write(header + "\n")
    for i in parts:
        output.write(i + "\n")
    output.write(post_header + "\n")
    for i in bank_heads:
        for j in i:
            output.write(j + "\n")
    for i in banks:
        write_component(i[0], i[1], i[2], output)
    for i in tubes:
        write_component(i[0], i[1], i[2], output)
    output.write(other_definitions + "\n")
    for i in idlists:
        for j in i:
            output.write(j + "\n")
    output.write(monitor_id + "\n")
    output.close()

def make_backward_only(angle, starting_number = 300000):
    if angle < -7.5:
        flatsets = flatsets15
        label = 'left'
    elif angle < 7.5:
        flatsets = flatsets0
        label = 'middle'
    else:
        flatsets = flatsetsm15
        label = 'right'
    parts = ['<component type="monitors" idlist="monitors"><location/></component>']
    tubes = []
    old_tubes, n_old_tubes = make_tube_old(lengths[3], "tube")
    tubes.append(["tube", old_tubes, "oldpixel"])
    banks = []
    bank_heads = []
    idlists = []
    nn = starting_number
    it = 2
    for i in flatsets:
        center = i[0]
        name = 'bank' + str(it)
        ntubes = 48
        idlists.append(idlist(name, nn, n_old_tubes, ntubes))
        tl, tn, nn = make_flat_panel(ntubes, center, nn, n_old_tubes)
        banks.append([name, tl, "tube"])
        parts.append(generate_bank(name))
        bank_heads.append(generate_bank_head(name))
        it += 2
        nn = starting_number * it
    output = open("EXED_BWpanels_" + label + ".xml", 'w')
    output.write(header + "\n")
    for i in parts:
        output.write(i + "\n")
    output.write(post_header + "\n")
    for i in bank_heads:
        for j in i:
            output.write(j + "\n")
    for i in banks:
        write_component(i[0], i[1], i[2], output)
    for i in tubes:
        write_component(i[0], i[1], i[2], output)
    output.write(other_definitions + "\n")
    for i in idlists:
        for j in i:
            output.write(j + "\n")
    output.write(monitor_id + "\n")
    output.close()

def main():
    for i in range(-15,16):
        make_forward_only(i)
    # for i in range(-15,16,15):
    #     make_backward_only(i)
    # make_SANS_only(0)

if __name__ == '__main__':
    main()
