#    This file is part of EXEQ.
#
#    EXEQ is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <https://www.gnu.org/licenses/>.
# 
# Copyright (C) Maciej Bartkowiak, 2014-2018, 2020

__doc__ = """
A packet of utilities for re-calculating between different
properties of neutrons. This is a non-SI approach, as
the wavelengths are not expressed in metres, but in Angstroms.
Maciej Bartkowiak, 28 Mar 2014

A new feature is the calculation of the TOF instrument chopper
frequencies, wavelength band and wavelength resolution.
 Maciej Bartkowiak, 21 Aug 2014
"""

import numpy as np

h = 6.626068 * 10**(-34) # it is Planck's constant given in J*s
meV = 1.602176487 * 10**(-22) # that is 1 meV expressed in Joule
mn = 1.67492729 * 10**(-27) # in kilograms
hbar = h / (2 * np.pi)

dL = 0.01 # the precision of our distance from chopper to the detector
          # set to 1 cm here, as we will have done calibration

def n_lambda(nenergy):
    """
    Converts energy in meV into wavelength in Angstrom.
    """
    wavel = h / np.sqrt(2*mn*nenergy*meV)
    return wavel * 10**10
    
def n_energy(nlambda):
    """
    Converts lambda in Angstrom into energy in meV.
    """
    e = h*h / (2*mn*nlambda*nlambda*10**(-20))
    return e / meV

def n_k(nlambda):
    """
    Gives the absolute value of quasimomentum that corresponds to the input wavelength.
    """
    # return 2*np.pi / nlambda
    return 1.0 / nlambda

def n_v(nlambda):
    """
    Returns the velocity [m/s] of a neutron with an input wavelength nlambda [A].
    """
    v = h / mn / (nlambda*10**-10)
    return v

def n_lambda_from_v(v):
    """
    Returns the wavelength nlambda [A] of a neutron with an input velocity [m/s].
    """
    nlambda = h / (v * mn) * 10**10
    return nlambda

def n_lambda_fromk(nk):
    """
    Gives the absolute value of quasimomentum that corresponds to the input wavelength.
    """
    # return 2*np.pi / nlambda
    return 1.0 / nk

def bandwidth(f, L = 56.1961):
    return h/(mn*f*L) * 1e10
    
def lastchopper(band):
    L = 56.1961
    wave = band * 1e-10
    return h/(mn*wave*L)
    
def delta_lambda(f, window, wave, L = 56.1961):
    part1 = dL/L
    dt = (1.0/f) * (window/360.0)
    # print "dt, ", dt
    TOF = L / n_v(wave)
    # print "TOF, ", TOF
    part2 = dt/TOF
    result = np.sqrt(part1**2 + part2**2)
    return result

def firstchopper(res, wave):
    wind1 = 6.86
    wind2 = 18.5
    L1 = 56.1961
    L2 = 56.0451
    TOF1, TOF2 = L1 / n_v(wave), L2 / n_v(wave)
    # temp1, temp2 = res**2 - (dL/L1)**2, res**2 - (dL/L2)**2
    temp1, temp2 = res**2, res**2
    if temp1 > 0.0:
        # dtt1 = np.sqrt(res**2 - (dL/L1)**2)
        dtt1 = np.sqrt(res**2)
    else:
        dtt1 = 1e-10
    if temp2 > 0.0:
        # dtt2 = np.sqrt(res**2 - (dL/L2)**2)
        dtt2 = np.sqrt(res**2)
    else:
        dtt2 = 1e-10
    dt1, dt2 = dtt1 * TOF1, dtt2 * TOF2
    f1 = (1.0/dt1) * (wind1/(2*360.0)) # the factor of 2 is added to account for
    f2 = (1.0/dt2) * (wind2/(2*360.0)) # the FWHM of the peak, not the base width
    # print "Fermi at ", f1
    # print "CH1AB at ", f2
    if f2 <= 201.0:
        return "CH1", f2
    elif f1 <= 610.0:
        return "Fermi", f1
    else:
        return "Resolution too high", -1
    
def intensity_drop(fpulse, fband, choppertype = ""):
    if fpulse > 600:
        fpulse = 600
    if fpulse < 0:
        fpulse = 1200
    if choppertype == "Resolution too high":
        wind = 0.01
    elif choppertype == "CH1":
        wind = 18.5
    else:
        wind = 6.86
    int_per_sec = (min(fband,fpulse)/fpulse) * (wind/(2*360.0))
    # int_per_sec = (1.0/fpulse) * (wind/(2*360.0))
    return int_per_sec
    
