#!/usr/bin/python

#    This file is part of EXEQ.
#
#    EXEQ is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <https://www.gnu.org/licenses/>.
# 
# Copyright (C) Maciej Bartkowiak, 2014-2018, 2020

import numpy as np

__doc__ = """
This module is meant to create an XML definition of the EXED detector,
of the kind that is used by Mantid.
The output is used by EXEQ, but the module itself is not accessed by
the EXED code in any way.
At the moment this script produces 16 files, each for a different position
of the High-Field Magnet, from 0 to 15 in 1-degree steps.
Maciej Bartkowiak, 21 Aug 2014
"""

inch = 2.54 * 10**(-2)
spacing = 0.1 * 10**(-2)
cone = np.radians(15)
width = 2.500
pixel = np.array((0.02,0.02))
oldpixel = np.array((0.5*inch,0.01))
panels = ['L', 'M', 'S']
tubes = [31, 15]
distance = 4.500
height = 2*distance*np.tan(np.radians(15))
radius = height / 2.0
lengths = [height, height * 0.842, height * 0.65, 0.9]
centre = 0.0
l, s, m, x = 31, 8, 15, 0
output = 'EXED_detectors_straight.xml'

#
# x = int(np.radians(15)*distance/pixel[1])
x=0
#

sets = [
        # (x -l -m -s/2, 'S', s),
        (x -l -m/2, 'M', m),
        (x -l/2, 'L', l),
#        (x/2, 'L', x),
        (x + l/2, 'L', l),
        (x + l + m/2, 'M', m),
        # (x + l + m +s/2, 'S', s)
        ]

flatsets0 = [
        (np.radians(-172.0),),
        (np.radians(172.0),),
#        (np.radians(-160.0),),
        ]

frontsets = [
        (np.radians(-7.95),),
        (np.radians(7.95),),
#        (np.radians(-160.0),),
        ]

SANSsets = [
        (np.radians(-1.744),),
        (np.radians(6.2816),),
#        (np.radians(-160.0),),
        ]
        
flatsets15 = [
#        (np.radians(172.0),),
        (np.radians(-172.0),),
        (np.radians(-158.68),)
        ]
        
flatsetsm15 = [
#        (np.radians(172.0),),
        (np.radians(158.68),),
        (np.radians(172.0),)
        ]

#frontsets = [
        #(np.radians(8.5),),
        #(np.radians(-8.5),),
##        (np.radians(-160.0),),
        #]
        
#flatsets15 = [
##        (np.radians(172.0),),
        #(np.radians(-172.0),),
        #(np.radians(-157.7),)
        #]
        
#flatsetsm15 = [
##        (np.radians(172.0),),
        #(np.radians(157.7),),
        #(np.radians(172.0),)
        #]        
        
#def height(position):
#    angle = position / distance

panel05 = np.array([[1038.84146121, 4378.44817469 ],
[1018.59227545, 4383.20314113 ],
[998.321327572, 4387.86446086 ],
[978.029050655, 4392.43203431 ],
[957.715878244, 4396.90576389 ],
[937.38224433, 4401.28555402 ],
[917.028583337, 4405.57131112 ],
[896.65533012, 4409.76294362 ],
[876.262919951, 4413.86036199 ],
[855.851788512, 4417.86347866 ],
[835.422371886, 4421.77220813 ],
[814.975106543, 4425.58646687 ],
[794.510429339, 4429.30617339 ],
[774.028777499, 4432.93124824 ],
[753.530588611, 4436.46161395 ],
[733.016300617, 4439.8971951 ],
[712.486351804, 4443.23791829 ],
[691.94118079, 4446.48371214 ],
[671.381226522, 4449.63450731 ],
[650.806928261, 4452.69023649 ],
[630.218725575, 4455.65083438 ],
[609.617058329, 4458.51623774 ],
[589.002366675, 4461.28638534 ]])

panel06 = np.array([[534.424286382, 4468.15293853 ],
[513.765855157, 4470.57542673 ],
[493.096447376, 4472.90240153 ],
[472.416504639, 4475.13381321 ],
[451.72646877, 4477.26961411 ],
[431.026781811, 4479.30975859 ],
[410.317886007, 4481.25420306 ],
[389.600223802, 4483.10290598 ],
[368.874237827, 4484.85582786 ],
[348.140370892, 4486.51293124 ],
[327.399065971, 4488.07418072 ],
[306.650766203, 4489.53954294 ],
[285.895914871, 4490.90898659 ],
[265.134955402,4492.18248243 ],
[244.368331351, 4493.36000323 ],
[223.596486395, 4494.44152385 ],
[202.819864322, 4495.42702117 ],
[182.038909023, 4496.31647414 ],
[161.254064481, 4497.10986376 ],
[140.465774762, 4497.80717307 ],
[119.674484006, 4498.40838718 ],
[98.880636416, 4498.91349325 ],
[78.0846762514, 4499.32248048 ],
[57.2870478154, 4499.63534013 ],
[36.4881954473, 4499.85206552 ],
[15.6885635127, 4499.97265203 ],
[-5.11140360664, 4499.99709706 ],
[-25.9112615213, 4499.92540011 ],
[-46.7105658444, 4499.7575627 ],
[-67.508872201, 4499.49358841 ],
[-88.3057362373, 4499.1334829] ])

panel07 = np.array([[-140.107963747, 4497.8183332 ],
[-160.896308872, 4497.12267765 ],
[-181.681216465, 4496.33094151 ],
[-202.462242456, 4495.44314171 ],
[-223.238942861, 4494.45929722 ],
[-244.010873789, 4493.37942906 ],
[-264.777591449, 4492.20356029 ],
[-285.538652162, 4490.93171604 ],
[-306.29361237, 4489.56392348 ],
[-327.042028646, 4488.10021184 ],
[-347.783457701, 4486.54061238 ],
[-368.517456398, 4484.88515843 ],
[-389.243581755, 4483.13388536 ],
[-409.961390963, 4481.28683058 ],
[-430.670441386, 4479.34403355 ],
[-451.370290578, 4477.30553579 ],
[-472.06049629, 4475.17138083 ],
[-492.740616475, 4472.94161429 ],
[-513.410209307, 4470.6162838 ],
[-534.06883318, 4468.19543904 ],
[-554.716046727, 4465.67913172 ],
[-575.35140882, 4463.06741562 ],
[-595.974478588, 4460.36034653 ],
[-616.58481542, 4457.55798228 ],
[-637.18197898, 4454.66038275 ],
[-657.76552921, 4451.66760985 ],
[-678.335026345, 4448.57972751 ],
[-698.89003092, 4445.39680171 ],
[-719.43010378, 4442.11890045 ],
[-739.954806086, 4438.74609377 ],
[-760.463699333, 4435.27845372 ]])

panel08 = np.array([[-813.174096601, 4425.91774535 ],
[-833.62291242, 4422.11180771 ],
[-854.053917969, 4418.21139209 ],
[-874.46667674, 4414.21658183 ],
[-894.860752619, 4410.12746226 ],
[-915.235709887, 4405.94412077 ],
[-935.591113235, 4401.66664672 ],
[-955.926527772, 4397.2951315 ],
[-976.241519034, 4392.82966851 ],
[-996.535652994, 4388.27035315 ],
[-1016.80849607, 4383.61728283 ],
[-1037.05961513, 4378.87055697 ],
[-1057.28857752, 4374.03027697 ],
[-1077.49495104, 4369.09654625 ],
[-1097.67830399, 4364.06947022 ],
[-1117.83820516, 4358.94915629 ],
[-1137.97422382, 4353.73571384 ],
[-1158.08592977, 4348.42925426 ],
[-1178.17289334, 4343.02989092 ],
[-1198.23468536, 4337.53773918 ],
[-1218.27087721, 4331.95291638 ],
[-1238.28104083, 4326.27554184 ],
[-1258.2647487, 4320.50573685 ]])

panel05[:,0] *=-1
panel06[:,0] *=-1
panel07[:,0] *=-1
panel08[:,0] *=-1
panel05 /= 1e3
panel06 /= 1e3
panel07 /= 1e3
panel08 /= 1e3

header = """<?xml version="1.0" encoding="UTF-8"?>
<instrument name="EXED" date=""
valid-from="1900-01-31 23:59:59"
valid-to="2100-01-31 23:59:59"
last-modified="2013-08-16 13:19:03">
<defaults>
<length unit="meter"/>
<angle unit="degree"/>
<location x="0.0" y="0.0" z="0.0" rot="0.0" axis-x="0.0" axis-y="0.0" axis-z="1.0"/>
<reference-frame>
<!-- The z-axis is set parallel to and in the direction of the beam. the
     y-axis points up and the coordinate system is right handed. -->
<along-beam axis="z"/>
<pointing-up axis="y"/>
<handedness val="right"/>
<origin val="beam" />
</reference-frame>
<offsets spherical="delta" />
<components-are-facing x="0.0" y="0.0" z="0.0" />
</defaults>"""

post_header = """<!-- source and sample-position components -->
<component type="double_disk">
<location z="-53.063"><facing val="none"/></location>
</component>
<component type="sample-holder"><location><facing val="none"/></location>
</component>

<!-- Source types -->
<type name="double_disk" is="Source">
<properties />
<cylinder id="some-shape">
<centre-of-bottom-base r="0.0" t="0.0" p="0.0" />
<axis x="0.0" y="0.0" z="1.0" />
<radius val="0.01" />
<height val="0.03" />
</cylinder>
</type>

<!-- Sample-position types -->
<type name="sample-holder" is="SamplePos">
<properties />
<sphere id="some-shape">
<centre x="0.0" y="0.0" z="0.0" />
<radius val="0.03" />
</sphere>
<algebra val="some-shape" />
</type>

<!-- Detectors types -->
<type name="monitors" idlist="monitors">
<component type="monitor" is="monitor">
<location x="0" y="0" z="-0.973" name="monitor1" />
</component>
</type>"""

other_definitions = """<type name="monitor" is="detector">
<properties/>
<cylinder id="some-shape">
<centre-of-bottom-base r="0.0" t="0.0" p="0.0" />
<axis x="0.0" y="0.0" z="1.0" />
<radius val="0.05" />
<height val="0.02" />
</cylinder>
<algebra val="some-shape" />
</type>

<type name="pixel" is="detector">
<cylinder id="cyl-approx">
<centre-of-bottom-base x="0.0" y="0.0" z="0.0" />
<axis x="0.0" y="0.2" z="0.0" />
<radius val="0.0095" />
<height val="0.02" />
</cylinder>
<algebra val="cyl-approx" />
</type>

<type name="oldpixel" is="detector">
<cylinder id="cyl-approx">
<centre-of-bottom-base x="0.0" y="0.0" z="0.0" />
<axis x="0.0" y="0.2" z="0.0" />
<radius val="0.004" />
<height val="0.01" />
</cylinder>
<algebra val="cyl-approx" />
</type>
"""

monitor_id = """<!-- MONITOR ID LISTS -->
<idlist idname="monitors">
<id start="1601001" end="1601001" />
</idlist>
</instrument>"""

def rotY(coords, angle = -15.0):
    ang = np.radians(angle)
    rot1 = np.array([np.cos(ang), 0.0, np.sin(ang)])
    rot2 = np.array([0.0, 1.0, 0.0])
    rot3 = np.array([- np.sin(ang), 0.0, np.cos(ang)])
    result = coords.copy()
    result[0] = (coords*rot1).sum()
    result[1] = (coords*rot2).sum()
    result[2] = (coords*rot3).sum()
    return result

def rotYnp(coords, angle = -15.0):
    ang = np.radians(angle)
    rot1 = np.array([np.cos(ang), 0.0, np.sin(ang)])
    rot2 = np.array([0.0, 1.0, 0.0])
    rot3 = np.array([- np.sin(ang), 0.0, np.cos(ang)])
    rotmat = np.row_stack([rot1, rot2, rot3])
    result = rotmat.dot(coords.T).T
    return result

def make_tube(height, name = 'tube'):
    pixellist = []
    number = 0
    for i in np.arange(-height/2.0 + pixel[1]/2.0, height/2.0, pixel[1]):
        number += 1
        pixellist.append('<location x="0" y="'+str(i)+'" z="0" name="Det'+str(number)+'"/>"')
    return pixellist, number
    
def make_tube_old(height = 0.9, name = 'oldtube'):
    pixellist = []
    number = 0
    for i in np.arange(-height/2.0 + oldpixel[1]/2.0, height/2.0, oldpixel[1]):
        number += 1
        pixellist.append('<location x="0" y="'+str(i)+'" z="0" name="Det'+str(number)+'"/>"')
    return pixellist, number

def make_panel(ntubes, angle, starting_number, pixels_per_tube, dr = 0):
    tubelist = []
    pnumber = starting_number
    tnumber = 1
    if dr == 0:
        current_panel = panel05
    if dr == 1:
        current_panel = panel06
    if dr == 2:
        current_panel = panel07
    if dr == 3:
        current_panel = panel08
    current_panel = np.column_stack([current_panel[:,0],np.zeros(len(current_panel)),current_panel[:,1]])
    current_panel = rotYnp(current_panel, angle)
    for i in current_panel:
        pos = [i[0], i[2]]
        tubelist.append('<location x="'+str(pos[0])+'" y="0" z="'+str(pos[1])+'" name="Tube'+str(tnumber)+'"/>')
        pnumber += pixels_per_tube
        tnumber += 1
    return tubelist, tnumber, pnumber

def make_flat_panel(ntubes, centre, starting_number, pixels_per_tube):
    dist = 3.0
    angle = centre
    # cent = np.array([np.sin(angle)*dist, 0.0, np.cos(angle)*dist])
    cent = rotY(np.array([0.0, 0.0, dist]), np.degrees(angle))
    perp = rotY(cent, 90.0)
    perp /= np.sqrt((perp**2).sum())
    spac = perp.copy()
    perp *= oldpixel[0]
    spac *= spacing
    tubelist = []
    pnumber = starting_number
    tnumber = 1
    for i in np.linspace(-(ntubes-1)/2.0 , (ntubes-1)/2.0, 48):
        pos = cent + i * (perp + spac)
        tubelist.append('<location x="'+str(pos[0])+'" y="0" z="'+str(pos[2])+'" name="Tube'+str(tnumber)+'"/>')
        pnumber += pixels_per_tube
        tnumber += 1
    return tubelist, tnumber, pnumber    

def make_front_panel(ntubes, centre, starting_number, pixels_per_tube):
    dist = 2.5
    angle = centre
    # cent = np.array([np.sin(angle)*dist, 0.0, np.cos(angle)*dist])
    cent = rotY(np.array([0.0, 0.0, dist]), np.degrees(angle))
    perp = rotY(cent, 90.0)
    perp /= np.sqrt((perp**2).sum())
    spac = perp.copy()
    perp *= oldpixel[0]
    spac *= spacing
    tubelist = []
    pnumber = starting_number
    tnumber = 1
    for i in np.linspace(-(ntubes-1)/2.0 , (ntubes-1)/2.0, 48):
        pos = cent + i * (perp + spac)
        tubelist.append('<location x="'+str(pos[0])+'" y="0" z="'+str(pos[2])+'" name="Tube'+str(tnumber)+'"/>')
        pnumber += pixels_per_tube
        tnumber += 1
    return tubelist, tnumber, pnumber    

def make_SANS_panel(ntubes, centre, starting_number, pixels_per_tube):
    dist = 5.7
    angle = centre
    # cent = np.array([np.sin(angle)*dist, 0.0, np.cos(angle)*dist])
    cent = rotY(np.array([0.0, 0.0, dist]), np.degrees(angle))
    perp = rotY(cent, 90.0)
    perp /= np.sqrt((perp**2).sum())
    spac = perp.copy()
    perp *= oldpixel[0]
    spac *= spacing
    tubelist = []
    pnumber = starting_number
    tnumber = 1
    for i in np.linspace(-(ntubes-1)/2.0 , (ntubes-1)/2.0, 48):
        pos = cent + i * (perp + spac)
        tubelist.append('<location x="'+str(pos[0])+'" y="0" z="'+str(pos[2])+'" name="Tube'+str(tnumber)+'"/>')
        pnumber += pixels_per_tube
        tnumber += 1
    return tubelist, tnumber, pnumber    
    
def write_component(name = "bank", parts = [], part_type = "tube", ofile = ""):
    lines = []
    lines.append('<type name="'+name+'">')
    lines.append('<properties />')
    lines.append('<component type="'+part_type+'">')
    for i in lines:
        ofile.write(i + "\n")
    for i in parts:
        ofile.write(i + "\n")
    ofile.write('</component>')
    ofile.write('</type>' + "\n")

def idlist(name, start, increment, steps):
    result = []
    result.append('<idlist idname= "'+name+'">')
    beg, end = start, start + increment -1
    for i in range(steps):
        result.append('<id start="'+str(beg)+'" end="'+str(end)+'" />')
        beg += 1000
        end += 1000
    result.append('</idlist>')
    return result
    
def generate_bank(name = "bank0"):
    return '<component type="'+name+'" idlist="'+name+'"><location/></component>'

def generate_bank_head(name):
    lines = []
    pname = 'panel' + name[-1:]
    lines.append('<type name="'+pname+'" idlist="'+pname+'">')
    lines.append('<component type="'+name+'">')
    lines.append('<location x="0" y="0" z="0"><facing val="none"/></location>')
    lines.append('</component>')
    lines.append('</type>')
    return lines
    
def make_full_ins(angle, starting_number = 100000):
    if angle < -7.5:
        flatsets = flatsetsm15
    elif angle < 7.5:
        flatsets = flatsets0
    else:
        flatsets = flatsets15
    parts = ['<component type="monitors" idlist="monitors"><location/></component>']
    tubes = []
    small_tubes, n_small_tubes = make_tube(lengths[2], "tube_S")
    medium_tubes, n_medium_tubes = make_tube(lengths[1], "tube_M")
    large_tubes, n_large_tubes = make_tube(lengths[0], "tube_L")
    old_tubes, n_old_tubes = make_tube_old(lengths[3], "tube")
    tubes.append(["tube_S", small_tubes, "pixel"])
    tubes.append(["tube_M", medium_tubes, "pixel"])
    tubes.append(["tube_L", large_tubes, "pixel"])
    tubes.append(["tube", old_tubes, "oldpixel"])
    banks = []
    bank_heads = []
    idlists = []
    nn = starting_number
    it = 1
    for i in frontsets:
        center = i[0]
        center += np.radians(angle)
        name = 'bank' + str(it)
        ntubes = 48
        idlists.append(idlist(name, nn, n_old_tubes, ntubes))
        tl, tn, nn = make_front_panel(ntubes, center, nn, n_small_tubes)
        banks.append([name, tl, "tube"])
        parts.append(generate_bank(name))
        bank_heads.append(generate_bank_head(name))
        it += 1
        nn = starting_number * it
    for i in flatsets:
        center = i[0]
        name = 'bank' + str(it)
        ntubes = 48
        idlists.append(idlist(name, nn, n_old_tubes, ntubes))
        tl, tn, nn = make_flat_panel(ntubes, center, nn, n_small_tubes)
        banks.append([name, tl, "tube"])
        parts.append(generate_bank(name))
        bank_heads.append(generate_bank_head(name))
        it += 1
        nn = starting_number * it
    output = open("EXED_configuration_" + str(angle) + "degrees.xml", 'w')
    output.write(header + "\n")
    for i in parts:
        output.write(i + "\n")
    output.write(post_header + "\n")
    for i in bank_heads:
        for j in i:
            output.write(j + "\n")
    for i in banks:
        write_component(i[0], i[1], i[2], output)
    for i in tubes:
        write_component(i[0], i[1], i[2], output)
    output.write(other_definitions + "\n")
    for i in idlists:
        for j in i:
            output.write(j + "\n")
    output.write(monitor_id + "\n")
    output.close()

def make_forward_only(angle, starting_number = 100000):
    parts = ['<component type="monitors" idlist="monitors"><location/></component>']
    tubes = []
    small_tubes, n_small_tubes = make_tube(lengths[2], "tube_S")
    medium_tubes, n_medium_tubes = make_tube(lengths[1], "tube_M")
    large_tubes, n_large_tubes = make_tube(lengths[0], "tube_L")
    old_tubes, n_old_tubes = make_tube_old(lengths[3], "tube")
    tubes.append(["tube_S", small_tubes, "pixel"])
    tubes.append(["tube_M", medium_tubes, "pixel"])
    tubes.append(["tube_L", large_tubes, "pixel"])
    tubes.append(["tube", old_tubes, "oldpixel"])
    banks = []
    bank_heads = []
    idlists = []
    nn = starting_number
    it = 1
    offset = np.radians(angle)*distance/pixel[0]
    for dr, i in enumerate(sets):
        center, size, ntubes = i[0]+offset, i[1], i[2]
        name = 'bank' + str(it)
        if size == 'S':
            idlists.append(idlist(name, nn, n_small_tubes, ntubes))
            tl, tn, nn = make_panel(ntubes, angle, nn, n_small_tubes, dr)
            banks.append([name, tl, "tube_S"])
            parts.append(generate_bank(name))
            bank_heads.append(generate_bank_head(name))
        if size == 'M':
            idlists.append(idlist(name, nn, n_medium_tubes, ntubes))
            tl, tn, nn = make_panel(ntubes, angle, nn, n_medium_tubes, dr)
            banks.append([name, tl, "tube_M"])
            parts.append(generate_bank(name))
            bank_heads.append(generate_bank_head(name))
        if size == 'L':
            idlists.append(idlist(name, nn, n_large_tubes, ntubes))
            tl, tn, nn = make_panel(ntubes, angle, nn, n_large_tubes, dr)
            banks.append([name, tl, "tube_L"])
            parts.append(generate_bank(name))
            bank_heads.append(generate_bank_head(name))
        it += 2
        nn = starting_number * it
    output = open("EXED_FWpanels_" + str(angle) + "degrees.xml", 'w')
    output.write(header + "\n")
    for i in parts:
        output.write(i + "\n")
    output.write(post_header + "\n")
    for i in bank_heads:
        for j in i:
            output.write(j + "\n")
    for i in banks:
        write_component(i[0], i[1], i[2], output)
    for i in tubes:
        write_component(i[0], i[1], i[2], output)
    output.write(other_definitions + "\n")
    for i in idlists:
        for j in i:
            output.write(j + "\n")
    output.write(monitor_id + "\n")
    output.close()
    
def make_SANS_only(angle, starting_number = 100000):
    parts = ['<component type="monitors" idlist="monitors"><location/></component>']
    tubes = []
    small_tubes, n_small_tubes = make_tube(lengths[2], "tube_S")
    medium_tubes, n_medium_tubes = make_tube(lengths[1], "tube_M")
    large_tubes, n_large_tubes = make_tube(lengths[0], "tube_L")
    old_tubes, n_old_tubes = make_tube_old(lengths[3], "tube")
    tubes.append(["tube_S", small_tubes, "pixel"])
    tubes.append(["tube_M", medium_tubes, "pixel"])
    tubes.append(["tube_L", large_tubes, "pixel"])
    tubes.append(["tube", old_tubes, "oldpixel"])
    banks = []
    bank_heads = []
    idlists = []
    nn = starting_number
    it = 1
    for i in SANSsets:
        center = i[0]
        # center += np.radians(angle)
        name = 'bank' + str(it)
        ntubes = 48
        idlists.append(idlist(name, nn, n_old_tubes, ntubes))
        tl, tn, nn = make_SANS_panel(ntubes, center, nn, n_small_tubes)
        banks.append([name, tl, "tube"])
        parts.append(generate_bank(name))
        bank_heads.append(generate_bank_head(name))
        it += 2
        nn = starting_number * it
    output = open("EXED_SANSpanels.xml", 'w')
    output.write(header + "\n")
    for i in parts:
        output.write(i + "\n")
    output.write(post_header + "\n")
    for i in bank_heads:
        for j in i:
            output.write(j + "\n")
    for i in banks:
        write_component(i[0], i[1], i[2], output)
    for i in tubes:
        write_component(i[0], i[1], i[2], output)
    output.write(other_definitions + "\n")
    for i in idlists:
        for j in i:
            output.write(j + "\n")
    output.write(monitor_id + "\n")
    output.close()

def make_backward_only(angle, starting_number = 300000):
    if angle < -7.5:
        flatsets = flatsets15
        label = 'left'
    elif angle < 7.5:
        flatsets = flatsets0
        label = 'middle'
    else:
        flatsets = flatsetsm15
        label = 'right'
    parts = ['<component type="monitors" idlist="monitors"><location/></component>']
    tubes = []
    old_tubes, n_old_tubes = make_tube_old(lengths[3], "tube")
    tubes.append(["tube", old_tubes, "oldpixel"])
    banks = []
    bank_heads = []
    idlists = []
    nn = starting_number
    it = 2
    for i in flatsets:
        center = i[0]
        name = 'bank' + str(it)
        ntubes = 48
        idlists.append(idlist(name, nn, n_old_tubes, ntubes))
        tl, tn, nn = make_flat_panel(ntubes, center, nn, n_old_tubes)
        banks.append([name, tl, "tube"])
        parts.append(generate_bank(name))
        bank_heads.append(generate_bank_head(name))
        it += 2
        nn = starting_number * it
    output = open("EXED_BWpanels_" + label + ".xml", 'w')
    output.write(header + "\n")
    for i in parts:
        output.write(i + "\n")
    output.write(post_header + "\n")
    for i in bank_heads:
        for j in i:
            output.write(j + "\n")
    for i in banks:
        write_component(i[0], i[1], i[2], output)
    for i in tubes:
        write_component(i[0], i[1], i[2], output)
    output.write(other_definitions + "\n")
    for i in idlists:
        for j in i:
            output.write(j + "\n")
    output.write(monitor_id + "\n")
    output.close()

def main():
    for i in range(-15,16):
        make_forward_only(i)
    # for i in range(-15,16,15):
    #     make_backward_only(i)
    # make_SANS_only(0)

if __name__ == '__main__':
    main()
